#!/bin/bash

SCRIPT_NAME=`basename $0`
SCRIPT_PATH=$(realpath "$(dirname "$0")")
LOG_FILE="/opt/oss/log/manager/easysuite_upgrade/${SCRIPT_NAME}.log"

# 日志记录
function log()
{
    local msg=$1
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] $msg" |tee -a $LOG_FILE
}

# 校验入参
function check_input()
{
    if [ "${1}" == "-input" ] && [ -n "${2}" ]; then
        INPUT_PATH=${2}
    else
        log "ERROR The input param is invalid."
        exit 0
    fi

    if [ "${3}" == "-error_path" ] && [ -n "${4}" ]; then
        ERROR_PATH=${4}
    else
        log "ERROR The error_path param is invalid."
        exit 0
    fi
}

# 检查版本
function check_version()
{
    src_version=`cat ${INPUT_PATH}/plandata.json | grep src_version | tr '"' ' ' | awk '{print $3 }'`
    is_R21C00=`echo ${src_version} | grep V100R021C00`
    is_R21C10=`echo ${src_version} | grep V100R021C10`
    if [ -z "${is_R21C00}" ] && [ -z "${is_R21C10}" ]; then
        return 0
    fi
    log "src_version:[${src_version}] need enable cloudlicenseproxy"
    return 1
}

# 在某个节点起某个服务
function enable_service()
{
    local target_ip=$1
    log "Start to enable CloudLicenseProxy on ${target_ip}"
    # 先激活，如果节点上没这个服务，执行默认成功，不会输出内容
    bash ${SSHTOOL} -exectype cmd -ip ${target_ip} -cmd "bash /opt/oss/manager/agent/bin/ipmc_adm -cmd enable -app CloudLicenseProxy" -u ossadm -timeout 3600 -encrypt N >> ${LOG_FILE} 2>&1
    if [ $? -ne 0 ]
    then
        log "Node ${target_ip}:Failed to enable CloudLicenseProxy."
        return 1
    fi

    # 再查状态，否则statusapp先执行会出现：Application CloudLicenseProxy has been disabled.
    # 添加命令级的重试，重试12次，间隔10s
    statusRes=1
    for i in {1..12}; do
        status=$(bash ${SSHTOOL} -exectype cmd -ip ${target_ip} -cmd "bash /opt/oss/manager/agent/bin/ipmc_adm -cmd statusapp -tenant ${PRODUCTNAME} -app CloudLicenseProxy" -u ossadm -timeout 60 -encrypt N 2>&1)
        statusRes=$?
        if [ $statusRes -eq 0 ] && [ -n "$status" ]; then
            break
        fi
        log "get cloudlicenseproxy service status failed, retry count=$i"
        sleep 10
    done
    log "service deploy node status is ${status}, cmd status is ${statusRes}"
    if [ $statusRes -ne 0 ]; then
         return 1
    fi

    echo ${status} | grep -qw "CloudLicenseProxy"
    if [ $? -eq 0 ]
    then
        bash ${SSHTOOL} -exectype cmd -ip ${target_ip} -cmd "bash /opt/oss/manager/agent/bin/ipmc_adm -cmd startapp -tenant ${PRODUCTNAME} -app CloudLicenseProxy" -u ossadm -timeout 3600 -encrypt N >> ${LOG_FILE} 2>&1
        if [ $? -ne 0 ]
        then
            log "Node ${target_ip}:Failed to start CloudLicenseProxy."
            return 1
        fi
    fi
    log "Finished to enable CloudLicenseProxy on ${target_ip}"
}

# 在所有节点起某个服务
function enable_all_node_server()
{
    PRODUCTNAME=`cat ${INPUT_PATH}/plandata.json | grep productname | tr '"' ' ' | awk '{print $3 }'`
    SSHTOOL="${OSS_ROOT}/tools/common/sshmgr/sshmgr.sh"

    # 查询所有NMS节点的管理ip
    node_ips=$(cat ${INPUT_PATH}/nodes_${PRODUCTNAME}.json 2>/dev/null | sed 's/,{/\n/g' | grep NMS_ | sed 's/,/\n/g' | grep nodemgrip | awk -F\" '{print $4}' 2>/dev/null)
    log "node ip list is ${node_ips}"
    for node_ip in ${node_ips}
    do
        enable_service "${node_ip}"
        if [ $? -ne 0 ]
        then
            log "Failed to enable CloudLicenseProxy."
            return 1
        fi
    done
}

# 检查入参
check_input "$@"
check_version
verMatch=$?
log "src_version match=${verMatch}"

. /opt/oss/manager/bin/engr_profile.sh
if [ ${verMatch} == 1 ]
then
    if ! enable_all_node_server; then # 执行失败，复制错误码文件到指定目录
        cp -rf "$SCRIPT_PATH"/error_code/cllproxy_00000001.json "$ERROR_PATH"/
        sed -i "s/script_name/$SCRIPT_NAME/g" "$ERROR_PATH"/cllproxy_00000001.json
        exit 1
    fi
fi

# 脚本执行完正常退出，不影响整个升级流程
exit 0