#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# -*- coding: utf-8 -*-
import os
import json
import time

from .common_utils.retry_util import retry
from .common_utils.local_cmd_util import exec_local_cmd
from utils.ssh_util import SSHClient, Host
from utils.log_util import EasysuiteLogger
from utils.thread_util import MultiThreadPool

CUR_DIR = os.path.dirname(os.path.abspath(__file__))
OSSADM_HOME_PATH = '/home/ossadm'
SOPUSER_HOME_PATH = '/home/sopuser'


class StandbyNodeInfo:
    """ 备节点登录信息 """

    def __init__(self, kvs, logger):
        self.kvs = kvs
        self.logger = logger
        self.login_ip = ""
        self.login_port = 22
        self.login_user = "sopuser"
        self.login_pwd = ""
        self.exec_user = "ossadm"
        self.exec_pwd = ""
        self._init_standby_node_info()

    def _init_standby_node_info(self):
        login_user = self.kvs.get("standby_login_user", "")
        if self.kvs.get('single_mgr_domain', '') == 'yes':
            # 单管场景
            port = self.kvs.get('node_standby_nmsserver_ssh_port', "")
            ossadm_pwd = self.kvs.get("node_standby_nmsserver_ossadm_pwd", "")
            login_ip = self.kvs.get("node_standby_nmsserver_ip", "")
            if login_user == "sopuser":
                login_pwd = self.kvs.get("node_standby_nmsserver_sopuser_pwd", "")
            else:
                login_pwd = ossadm_pwd
        else:
            # 分布式场景
            port = self.kvs.get('node_standby_omp01_ssh_port', "")
            ossadm_pwd = self.kvs.get("node_standby_omp01_ossadm_pwd", "")
            login_ip = self.kvs.get('node_standby_omp01_ip', "")
            if login_user == "sopuser":
                login_pwd = self.kvs.get("node_standby_omp01_sopuser_pwd", "")
            else:
                login_pwd = ossadm_pwd
        self.login_ip = login_ip
        self.login_user = login_user
        self.login_port = port
        self.login_pwd = login_pwd
        self.exec_pwd = ossadm_pwd

    @retry(success_condition=lambda result: result[0] == 0)
    def exec_cmd(self, cmd, timeout=60) -> tuple:
        """
        执行ssh命令
        :param cmd: 命令
        :param timeout: 超时时间
        :return: tuple
        """
        if self.login_user == self.exec_user:
            return self.exec_login_cmd(cmd, timeout)
        ssh_host = Host(self.login_ip, self.login_user, self.login_pwd, self.login_port, su_user=self.exec_user,
                        su_password=self.exec_pwd)
        ssh_client = SSHClient(ssh_host)
        if not ssh_client.open():
            self.logger.warning(f"Login {self.login_ip} unsuccessfully")
        return ssh_client.run_su_cmd(cmd, timeout=timeout)

    @retry(success_condition=lambda result: result[0] == 0)
    def exec_login_cmd(self, cmd, timeout=60) -> tuple:
        """
        执行ssh命令，以登录用户权限执行
        :param cmd: 命令
        :param timeout: 超时时间
        :return: tuple
        """
        ssh_host = Host(self.login_ip, self.login_user, self.login_pwd, self.login_port)
        ssh_client = SSHClient(ssh_host)
        if not ssh_client.open():
            self.logger.warning(f"Login {self.login_ip} unsuccessfully")
        return ssh_client.run_cmd(cmd, timeout=timeout)

    @retry()
    def put(self, local, remote) -> bool:
        """
        传输软件包
        :param local:本地文件
        :param remote:目标文件
        :return: bool
        """
        return SSHClient(Host(self.login_ip, self.login_user, self.login_pwd, self.login_port)).put(local, remote)


class CheckHardwareAlarm:
    """ 升级前检查硬件告警 """

    EXEC_TMP_DIT = "/opt/oss/tmp/manager/HardwareAlarmCheck"
    PREVIOUS_CHECK_RESULT = f"{EXEC_TMP_DIT}/previous_check.json"
    SYNC_SCRIPT_PATH = "/opt/oss/manager/bin/syncalarm.sh"
    SYNC_JSON_PATH = f"{EXEC_TMP_DIT}/sync.json"
    STANDBY_HARDWARE_IP = f"{EXEC_TMP_DIT}/standby_hardware_ip.json"
    CHECK_FILE_DIR = 'hardware_alarm_check'
    REMOTE_UNZIP_PATH = f"{OSSADM_HOME_PATH}/{CHECK_FILE_DIR}"
    REMOTE_SHELL_PATH = REMOTE_UNZIP_PATH + f"{CUR_DIR}/{CHECK_FILE_DIR}/setup_env.sh"

    def __init__(self, kvs):
        self.kvs = kvs
        self.logger = EasysuiteLogger.get_logger(kvs, "task_check_hardware_alarm")
        self.standby_node = StandbyNodeInfo(kvs, self.logger)

    def _is_disaster_recovery_scene(self):
        return self.kvs.get("protection", "") == "id_protection_hot"

    def _execute_local_sync_alarm_operation(self):
        """
        执行本地环境告警同步
        """
        sync_alarm_cmd = f"bash {self.SYNC_SCRIPT_PATH} -output {self.SYNC_JSON_PATH}"
        result, stdout, stderr = exec_local_cmd(sync_alarm_cmd)
        if result != 0:
            self.logger.error(
                f"Execute local sync alarm operation failed, result: {result}, stdout: {stdout}, stderr: {stderr}.")
            return False
        return True

    def _execute_remote_sync_alarm_operation(self):
        """
        执行备站点环境告警同步
        """
        sync_alarm_cmd = f"bash {self.SYNC_SCRIPT_PATH} -output {self.SYNC_JSON_PATH}"
        result, stdout, stderr = self.standby_node.exec_cmd(sync_alarm_cmd)
        if result != 0:
            self.logger.error(
                f"Execute remote sync alarm operation failed, result: {result}, stdout: {stdout}, stderr: {stderr}.")
            return False
        return True

    def _check_previous_result_failed(self) -> bool:
        """
        获取上次检查结果方法，判断是否执行同步操作
        :return: 上次检查是否失败
        """
        if not os.path.exists(self.PREVIOUS_CHECK_RESULT):
            return True
        with open(self.PREVIOUS_CHECK_RESULT, 'r') as previous_result_file:
            content = previous_result_file.read()
            result = json.loads(content)
        primary_need_sync, standby_need_sync = False, False
        if result.get("check_result", "") == "Failed":
            primary_need_sync = True
            self.logger.info("Active site previous check unsuccessfully, start to sync hardware alarm information.")

        if self._is_disaster_recovery_scene():
            _, stdout, _ = self.standby_node.exec_cmd(
                f"[ -f {self.PREVIOUS_CHECK_RESULT} ] && cat {self.PREVIOUS_CHECK_RESULT} || echo 'FileNotExist'")
            if "FileNotExist" not in stdout:
                standby_previous_result = {}
                try:
                    standby_previous_result = json.loads(stdout)
                except json.JSONDecodeError as _:
                    self.logger.warning("Standby site previous result unsuccessfully")
                if standby_previous_result.get("check_result", "Failed") == "Failed":
                    standby_need_sync = True
                    self.logger.info("Standby site previous check unsuccessfully, start to sync hardware alarm information.")

        if primary_need_sync or standby_need_sync:
            if not self._execute_local_sync_alarm_operation():
                return False
            if self._is_disaster_recovery_scene():
                if not self._execute_remote_sync_alarm_operation():
                    return False
            self.logger.info("Wait for 5 minutes.")
            time.sleep(300)
        self.logger.info("Previous check success, continue current checking process.")
        return True

    def _process_result_list(self, result_list) -> bool:
        """ 检查硬件告警信息调用接口 """
        total_status = True
        for item in result_list:
            if not item:
                self.logger.error("Check result is empty.")
                return False
            check_status = bool(item.get("CheckStatus", ""))
            if not check_status:
                total_status = False
            self._print_result_to_logger(item)
        return total_status

    def _print_result_to_logger(self, result):
        """
        打印检查结果信息并显示
        """
        check_status = bool(result.get("CheckStatus", ""))
        check_msg = result.get("ResultMessage", "")
        check_site = result.get("site", "")
        self.logger.info(f"Logging checking result details message of %s site. ====Start====", check_site)
        for msg in check_msg:
            if not check_status:
                self.logger.error(msg)
            else:
                self.logger.info(msg)
        self.logger.info(f"Logging checking result details message of %s site. ====Finish====", check_site)

    def _get_standby_hardware_ip(self):
        """
        获取备站点硬件IP，异地容灾场景下，备站点告警会自动转发到主站点，主站点需要能获取到备站点的硬件IP，用于过滤告警
        """
        if not self._is_disaster_recovery_scene():
            return
        if os.path.exists(self.STANDBY_HARDWARE_IP):
            os.remove(self.STANDBY_HARDWARE_IP)
        exec_cmd = f"bash {self.REMOTE_SHELL_PATH} get_hardware_ip"
        result, stdout, stderr = self.standby_node.exec_cmd(exec_cmd)
        stdout = stdout.strip()
        if result != 0:
            self.logger.warning(
                f"Get standby hardware ip unsuccessfully, result: {result}.")
            return
        self.logger.info(f"Get standby hardware ip: {stdout}.")
        if not stdout:
            return
        open_mode = os.O_CREAT | os.O_WRONLY | os.O_TRUNC
        with os.fdopen(os.open(self.STANDBY_HARDWARE_IP, open_mode, mode=0o600), 'w') as file_obj:
            file_obj.write(json.dumps({"standby_hardware_ip": stdout}))

    def _get_primary_site_check_result(self) -> dict:
        """
        执行主站点检查
        """
        result_dict = {"site": "active"}
        exec_cmd = f'bash {CUR_DIR}/{self.CHECK_FILE_DIR}/setup_env.sh check'
        result, stdout, stderr = exec_local_cmd(exec_cmd)
        if result != 0:
            self.logger.error(
                f"Get active site check result is empty, result: {result}, stdout: {stdout}, stderr: {stderr}.")
            return {}
        result_dict.update(json.loads(stdout))
        return result_dict

    def _get_secondary_site_check_result(self) -> dict:
        """
        执行备站点检查
        """
        result_dict = {"site": "standby"}
        try:
            result_dict.update(json.loads(self._execute_secondary_operation()))
        except Exception as e:
            self.logger.error(f"Get standby site check result catch an exception, {e}.")
            return {}
        return result_dict

    def _execute_secondary_operation(self) -> str:
        """
        备站点上执行操作
        :return: 返回检查结果
        """
        try:
            exec_cmd = f"bash {self.REMOTE_SHELL_PATH} check"
            _, stdout, _ = self.standby_node.exec_cmd(exec_cmd, timeout=300)
            return stdout
        finally:
            self.standby_node.exec_cmd(f'rm -rf {self.REMOTE_UNZIP_PATH} 2>/dev/null')

    @retry()
    def _execute_pre_operation(self):
        """
        前置操作，用于将临时文件放入ossadm用户文件夹中，同时创建临时工作目录
        """
        result, stdout, stderr = exec_local_cmd(f"mkdir -p {self.EXEC_TMP_DIT}")
        if result != 0:
            self.logger.error(f"Create exec dir failed, "
                              f"result: {result}, stdout: {stdout}, stderr: {stderr}.")
            return False
        if not self._is_disaster_recovery_scene():
            return True
        self.logger.info("Start upload script to the standby node.")
        zip_cmd = f"tar -cvf  {CUR_DIR}/{self.CHECK_FILE_DIR}.tar {CUR_DIR}/{self.CHECK_FILE_DIR}"
        mkdir_remote_cmd = f'rm -rf {self.REMOTE_UNZIP_PATH} 2>/dev/null;' \
                           f'mkdir -p {self.REMOTE_UNZIP_PATH};mkdir -p {self.EXEC_TMP_DIT}'
        cp_file_cmd = f"cp -f {SOPUSER_HOME_PATH}/{self.CHECK_FILE_DIR}.tar {self.REMOTE_UNZIP_PATH}/{self.CHECK_FILE_DIR}.tar"
        unzip_cmd = f"tar -xvf {self.REMOTE_UNZIP_PATH}/{self.CHECK_FILE_DIR}.tar -C {self.REMOTE_UNZIP_PATH}"
        try:
            result, stdout, stderr = exec_local_cmd(zip_cmd)
            if result != 0:
                self.logger.error(f"Tar check script package failed, "
                                  f"result: {result}, stdout: {stdout}, stderr: {stderr}.")
                return False
            result, stdout, stderr = self.standby_node.exec_cmd(mkdir_remote_cmd)
            if result != 0:
                self.logger.error(f"create standby dir failed, stdout: {stdout}, stderr: {stderr}.")
                return False
            result = self.standby_node.put(f"{CUR_DIR}/{self.CHECK_FILE_DIR}.tar",
                                           f"{SOPUSER_HOME_PATH}/{self.CHECK_FILE_DIR}.tar")
            if not result:
                self.logger.error(f"upload package to standby failed, stdout: {stdout}, stderr: {stderr}.")
                return False
            result, stdout, stderr = self.standby_node.exec_cmd(cp_file_cmd + ";" + unzip_cmd)
            if result != 0:
                self.logger.error(f"unzip package failed, stdout: {stdout}, stderr: {stderr}.")
                return False
            self.logger.info("Upload script to the standby node successfully.")
            self._get_standby_hardware_ip()
            return True
        finally:
            self.standby_node.exec_login_cmd(f"rm -f {SOPUSER_HOME_PATH}/{self.CHECK_FILE_DIR}.tar")

    def _execute(self):
        if not self._check_previous_result_failed():
            return False
        if not self._execute_pre_operation():
            return False
        thread_pool = MultiThreadPool()
        thread_pool.add_thread(self._get_primary_site_check_result)
        if self._is_disaster_recovery_scene():
            thread_pool.add_thread(self._get_secondary_site_check_result)
        result_list = thread_pool.execute_with_result()
        self.logger.info("Finish check site hardware alarm, start to process result.")
        return self._process_result_list(result_list)

    def execute_task_check(self):
        """ 升级前检查主流程 """
        try:
            self.logger.easysuite_start("Begin to check hardware alarm.")
            self.logger.info(f"The environment is DR environment: {self._is_disaster_recovery_scene()}.")
            if not self._execute():
                self.logger.easysuite_error("Execute hardware alarm check task failed.")
                return False
            self.logger.easysuite_finish("Finish hardware alarm check task.")
            return True
        except Exception as e:
            self.logger.error(f"Catch an exception: {e}")
            return False


def _check_port_usage_exists() -> bool:
    """
    检查环境是否为一体机发货场景
    :return: bool
    """
    cmd = "cat /opt/oss/manager/etc/sysconf/nodelists.json | grep -c 'MGMTALM_SBI'"
    ret_code, ret_msg, ret_err = exec_local_cmd(cmd)
    if ret_code != 0:
        return False
    if int(ret_msg) > 0:
        return True
    return False


def check(kvs):
    """
    功能描述：判断此任务是否需要执行
    :param kvs: 全局字典
    :return: 检查结果
    """
    return (kvs.get('check_hardware_alarm', '') == 'true') and _check_port_usage_exists()


def exe(kvs):
    return [[('', '', CheckHardwareAlarm(kvs).execute_task_check())]]
