﻿#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
# -*- coding: utf-8 -*-
"""检查当前SUSE系统内核版本"""

import os
import time
from .common_utils.check_task import OSCheckTask, OSCheckConfig

_config_params = {
    # 检查项名称
    "task_name": 'task_check_suse_kernel',
    # sudo包相对路径
    "sudo_pkg_name": "es_check_sudobin_pkg.tar",
    # sudo包临时解压目录
    "sudo_tmp_path": f"/opt/clean_audit_{int(time.time())}",
    # 当前目录
    "upload_src_path": os.path.split(os.path.realpath(__file__))[0],
    # 上传目录
    "upload_trg_path": "/opt/oss/tmp",
}


def execute_task_check(kvs):
    """
    执行检查任务
    :param kvs: 全局字典
    :return: bool 检查项是否通过
    """
    sudo_script = "clean_audit.sh"
    check_result = "/var/log/clean_audit_result"
    task = OSCheckTask(OSCheckConfig(_config_params), kvs)
    # 如果Suse版本属于安全版本，则不需要检查
    if task.check_suse_kernel():
        task.finish()
        return True
    # 上传提权包到管理节点和其他节点
    if not task.upload_sudo_pkg():
        return False
    # 提权执行检查脚本
    if not task.execute_sudo_script(sudo_script):
        return False
    if not task.check_results(check_result, "clean"):
        return False
    else:
        msg = "The current SUSE system version has a soft lock problem. When the system load is heavy, the soft lock " \
              "may be triggered and the system is suspended.Audit rules have been automatically cleared for the " \
              "current check item to reduce the triggering probability. You are advised to upgrade the kernel OS " \
              "version to 4.12.14-122.159 or later."
        if task.check_kvs_contain('id_upgrade_os', 'upgrade_item'):
            msg = "The SUSE kernel is checked successfully. " + msg
            task.logger.easysuite_info(msg)
        else:
            task.logger.easysuite_error(msg)
            return False

    # 清理上传sudo包
    if not task.rm_sudo_pkg():
        return False
    task.finish()
    return True


def check(kvs):
    """
    功能描述：判断此任务是否需要执行
    :param kvs: 全局字典
    :return: bool 是否需要执行
    """
    return 'suse' in kvs.get('os_type').lower()


def exe(kvs):
    """
    功能描述：任务执行入口
    :param kvs: 全局字典
    :return: list 检查结果
    """
    return [[('', '', execute_task_check(kvs))]]
