# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
# -*- coding: utf-8 -*-
"""
检查项配置类与任务类
"""
import os
import re
import logging
from collections import Counter
from utils.ssh_util import Host, SSHClient
from utils.thread_util import MultiThreadPool
from func.upgrade.common.upgrade_local_ssh import UpgradeLocalSsh
from func.upgrade.upload_pkg_mgr.dao.upload_package_dao import UploadPackageDao
from .check_decorator import decorator_multithread
from .check_node import MasterNode, StandbyNode, ArbitrateNode
from distutils.version import LooseVersion
from .check_exception import CheckResultError, InconsistentOSTypeError, InvalidOSKernelError, UploadFileError, \
    InvalidOSTypeError, NoInvolvedSuseOSError, InconsistentRPMList, InvalidRPMVersion, InvalidPatchError

_local_ip = "127.0.0.1"
_upload_path_tmp = "/tmp"
_unknown_os_type = "Unknown"
_ssh_para = "-o ConnectTimeout=30 " \
            "-o StrictHostKeyChecking=no " \
            "-o ConnectionAttempts=5 " \
            "-o ServerAliveInterval=3 " \
            "-o ServerAliveCountMax=20"
_os_type_dict = {
    "euler": "Euler",
    "centos": "Centos",
    "kylin": "Kylin",
    "suse": "Huawei_Suse, Customer_Suse"
}
_os_path_map = {
    "suse-12sp5": "rpm_list/sles/12.5/rpm_list.txt"
}
_local_logger = logging.getLogger(__name__)


class CheckOperationUtils:
    """
    检查项操作函数类
    """

    @staticmethod
    def cp_file_on_node(node, file_name, src_path, trg_path):
        """
        在节点上复制文件
        :param node: 节点信息
        :param file_name: 复制文件名
        :param src_path: 复制文件源目录
        :param trg_path: 复制文件目标目录
        :return: None
        """
        mkdir_cmd = f"[ -d {trg_path}] || mkdir -p {trg_path}"
        node.execute_su_cmd(mkdir_cmd)
        cp_cmd = f"cp -af {os.path.join(src_path, file_name)} {trg_path}"
        node.execute_su_cmd(cp_cmd)

    @staticmethod
    def mv_file_on_node(node, src_file, src_path, trg_file, trg_path):
        """
        在节点上移动文件
        :param node: 节点信息
        :param src_file: 移动源文件名称
        :param src_path: 移动源文件目录
        :param trg_file: 移动目标文件名称
        :param trg_path: 移动目标文件目录
        :return: None
        """
        mkdir_cmd = f"[ -d {trg_path}] || mkdir -p {trg_path}"
        node.execute_su_cmd(mkdir_cmd)
        cp_cmd = f"mv -f {os.path.join(src_path, src_file)} {os.path.join(trg_path, trg_file)}"
        node.execute_su_cmd(cp_cmd)

    @staticmethod
    def rm_file_on_node(node, file_full_path):
        """
        在节点上删除文件
        :param node: 节点信息
        :param file_full_path: 待删除文件绝对路径
        :return: None
        """
        rm_cmd = f"rm -f {file_full_path}"
        node.execute_su_cmd(rm_cmd)

    @staticmethod
    @decorator_multithread
    def remove_nodes_file(node, ip, file_full_path):
        """
        删除所有节点上sudo包
        :param node: 操作节点信息
        :param ip: 操作ip
        :param file_full_path: 待删除文件绝对路径
        :return: None
        """
        # ip为空跳过
        if not ip or not node:
            return
        cmd = f"rm -f {file_full_path}"
        if ip != _local_ip:
            cmd = f"ssh {_ssh_para} {ip} \"{cmd}\""
        if ip == _local_ip and UpgradeLocalSsh.is_es_node(node.login_ip):
            node.execute_cmd(cmd)
        else:
            node.execute_su_cmd(cmd)

    @staticmethod
    def upload_file_to_all_nodes(nodes_info, src_file, src_path, trg_file, trg_path):
        """
        把sudo包上传到所有节点
        :param nodes_info: 所有节点信息
        :param src_file: 上传源文件名称
        :param src_path: 上传源文件目录
        :param trg_file: 上传目标文件名称
        :param trg_path: 上传目标文件目录
        :return: None
        """
        # 将包上传到管理节点
        CheckOperationUtils.upload_file_to_managers(nodes_info, src_file, src_path, trg_file, trg_path)
        # 由管理节点将包上传到其他节点
        CheckOperationUtils.upload_file_to_other_nodes(nodes_info, trg_file, trg_path, trg_path)

    @staticmethod
    def upload_file_to_managers(nodes_info, src_file, src_path, trg_file, trg_path):
        """
        将提权包从ES管理面上传到其他站点的管理节点
        :param nodes_info: 所有节点信息
        :param src_file: 上传源文件名称
        :param src_path: 上传源文件目录
        :param trg_file: 上传目标文件名称
        :param trg_path: 上传目标文件目录
        :return: None
        """
        for _, node in nodes_info.items():
            if not node or not node.login_ip:
                continue
            if UpgradeLocalSsh.is_es_node(node.login_ip):
                # 本端节点，只需复制文件
                mkdir_cmd = f"mkdir -p {trg_path}"
                cp_cmd = f"cp -af {src_path}/{src_file} {trg_path}/{trg_file}"
                node.execute_cmd(f"{mkdir_cmd} && {cp_cmd}")
            elif node.node_type == "arbitrate":
                # 上传文件到三方站点节点管理面，需借助/tmp目录中转
                CheckOperationUtils.upload_file_to_node(node, src_file, src_path, trg_file, _upload_path_tmp)
                # 再把sudo包移到upload目录
                CheckOperationUtils.cp_file_on_node(node, trg_file, _upload_path_tmp, trg_path)
                CheckOperationUtils.rm_file_on_node(node, os.path.join(_upload_path_tmp, trg_file))
            else:
                # 上传文件到备站点节点
                CheckOperationUtils.upload_file_to_node(node, src_file, src_path, trg_file, trg_path)

    @staticmethod
    @decorator_multithread
    def upload_file_to_other_nodes(node, ip, file_name, src_path, trg_path):
        """
        在站点内将提权包从管理节点上传到其他节点
        :param node: 执行节点
        :param ip: 执行ip
        :param file_name: 上传源文件名称
        :param src_path: 上传源文件目录
        :param trg_path: 上传目标文件目录
        :return: None
        """
        # ip为空跳过
        if not ip or not node:
            return
        # 仲裁节点，或者ip为本节点ip时跳过，不需要进行传包操作
        if node.node_type == "arbitrate" or ip == _local_ip:
            return
        mkdir_cmd = f"ssh {_ssh_para} {ip} \"[ -d {trg_path} ] || mkdir -p {trg_path}\""
        node.execute_su_cmd(mkdir_cmd)
        clear_cmd = f"ssh {_ssh_para} {ip} \"[ ! -f {os.path.join(trg_path, file_name)} ] || " \
                    f"rm -f {os.path.join(trg_path, file_name)}\""
        node.execute_su_cmd(clear_cmd)
        upload_cmd = f"scp {_ssh_para} {src_path}/{file_name} [{ip}]:{trg_path}/{file_name}"
        node.execute_su_cmd(upload_cmd)

    @staticmethod
    def upload_file_to_node(node, src_file, src_path, trg_file, trg_path):
        """
        上传文件到特点节点上
        :param node: 全局节点信息
        :param src_file: 上传源文件名称
        :param src_path: 上传源文件目录
        :param trg_file: 上传目标文件名称
        :param trg_path: 上传目标文件目录
        :return: None
        """
        mkdir_cmd = f"[ -d {trg_path}] || mkdir -p {trg_path}"
        node.execute_su_cmd(mkdir_cmd)
        clear_cmd = f"[ ! -f {os.path.join(trg_path, trg_file)} ] || rm -f {os.path.join(trg_path, trg_file)}"
        node.execute_su_cmd(clear_cmd)
        if node.node_type == "arbitrate":
            # 三方仲裁节点，用sopuser登录
            host = Host(hostname=node.login_ip, username=node.login_user,
                        password=node.login_pwd, port=node.sftp_port)
        else:
            # 对端节点，用ossadm登录
            host = Host(hostname=node.login_ip, username=node.su_user,
                        password=node.su_pwd, port=node.sftp_port)
        result = SSHClient(host).put(os.path.join(src_path, src_file),
                                     remote=os.path.join(trg_path, trg_file), preserve_mode=True)
        if not result:
            _local_logger.error("Upload sudo package to %s failed", node.login_ip)
            raise UploadFileError(f"Upload sudo package to {node.login_ip} failed")

    @staticmethod
    @decorator_multithread
    def execute_sudo_script(node, ip, script_with_params, pkg_full_path, tmp_path):
        """
        在节点上执行sudo检查脚本
        :param node: 执行节点
        :param ip: 执行ip
        :param script_with_params: 待执行的脚本名以及传入参数
        :param pkg_full_path: sudo包绝对路径
        :param tmp_path: sudo包临时解压目录
        :return: None
        """
        # ip为空跳过
        if not ip or not node:
            return
        sudobin_home_path = CheckOSInfoUtils.get_sudobin_file(node)
        cmd = f"sudo {sudobin_home_path}/execute.sh {pkg_full_path} {tmp_path} {script_with_params} 2>>/dev/null"
        # 三方站点，降权至ossadm来提权执行
        if node.node_type == "arbitrate":
            cmd = f"su - ossadm -c \"{cmd}\""
        if ip != _local_ip:
            cmd = f"ssh {_ssh_para} {ip} \"{cmd}\""
        # 执行检查
        node.execute_su_cmd(cmd)

    @staticmethod
    @decorator_multithread
    def execute_script(node, ip, script_with_params):
        """
        在节点上执行检查脚本
        :param node: 执行节点
        :param ip: 执行ip
        :param script_with_params: 待执行脚本的绝对路径，以及传入参数
        :return: None
        """
        # ip为空跳过
        if not ip or not node:
            return
        cmd = f"bash {script_with_params} 2>>/dev/null"
        if ip != _local_ip:
            cmd = f"ssh {_ssh_para} {ip} \"{cmd}\""
        # 执行检查
        if UpgradeLocalSsh.is_es_node(node.login_ip):
            node.execute_cmd(cmd)
        else:
            node.execute_su_cmd(cmd)

    @staticmethod
    @decorator_multithread
    def check_script_results(node, ip, result_full_path, task_name, check_keyword='check'):
        """
        检查执行脚本的结果
        :param node: 执行节点
        :param ip: 执行ip
        :param result_full_path: 脚本结果文件的绝对路径
        :param task_name: 检查项任务名字
        :param check_keyword: 判断结果的关键字
        :return: None
        """
        # ip为空跳过
        if not ip or not node:
            return
        cmd = f"cat {result_full_path} 2>>/dev/null"
        if ip != _local_ip:
            cmd = f"ssh {_ssh_para} {ip} \"{cmd}\""
        # 获取检查结果
        stdout = node.execute_su_cmd(cmd)
        if not stdout or check_keyword not in stdout:
            if ip == _local_ip:
                ip = node.login_ip
            _local_logger.error("Failed to obtain the check result on %s. Cmd: %s Stdout: %s", ip, cmd, stdout)
            raise CheckResultError(f"Failed to obtain the check result on {ip}. Cmd: {cmd} Stdout: {stdout}")
        if f"{check_keyword} failed" in stdout:
            if ip == _local_ip:
                ip = node.login_ip
            _local_logger.error("The %s on node %s is checked failed. "
                                "Cmd: %s Stdout: %s", task_name, ip, cmd, stdout)
            raise CheckResultError(f"The {task_name} on node {ip} is checked failed."
                                   f"Cmd: {cmd} Stdout: {stdout}")


class CheckOSInfoUtils:
    """
    检查项查询与检查函数类
    """

    @staticmethod
    def get_sudobin_file(node):
        """
        获取节点中sudobin绝对路径
        :param node: 执行命令时所在节点对象
        :return: str sudobin绝对路径
        """
        # 获取sudobin安装位置，普通环境/usr/local/uniepsudobin，企业环境/opt/sudobin
        cmd = "[ -f /opt/oss/manager/bin/engr_profile.sh ] && . /opt/oss/manager/bin/engr_profile.sh;" \
              "if [ -z ${SUPER_ROOT} ];then echo '/usr/local/uniepsudobin'; else echo ${SUPER_ROOT};" \
              "fi 2>>/dev/null"
        stdout = node.execute_su_cmd(cmd)
        return stdout.replace("\n", "").replace("\r", "")

    @staticmethod
    def get_node_connect_info(kvs):
        """
        获取节点登录信息
        :param kvs: 全局字典
        :return: dict(str: Node) 三种类型(主节点、备节点、三方站点)中所有节点的信息
        """
        nodes_dict = {
            "master": MasterNode(kvs),
            "standby": StandbyNode(kvs) if kvs.get("protection") == 'id_protection_hot' else None,
            "arbit": ArbitrateNode(kvs) if "id_upgrade_arbitration" in kvs.get("upgrade_item") else None,
        }
        return nodes_dict

    @staticmethod
    def get_node_os_type_info(node):
        """
        获取节点的操作系统类型信息
        :param node: 节点信息
        :return: str 操作系统类型信息
        """
        global _unknown_os_type
        global _os_type_dict

        os_type = _unknown_os_type
        cmd = "cat /proc/version && lsblk -lo NAME"
        stdout = node.execute_cmd(cmd)
        _local_logger.info("The command 'cat /proc/version && lsblk -lo NAME' on node %s output is as follows :%s",
                           node.login_ip, stdout)
        for key, value in _os_type_dict.items():
            if key in stdout.lower():
                if key != "suse":
                    os_type = value
                    break
                if "root_bak" in stdout.lower():
                    os_type = value.split(',')[0]
                else:
                    os_type = value.split(',')[1]
                break
        _local_logger.info("The OS type of the node %s is %s.", node.login_ip, os_type)
        return os_type

    @staticmethod
    def check_nodes_os_type_consistence(nodes_info):
        """
        在所有节点上检查操作系统一致性
        :param nodes_info: 所有节点信息
        :return: None
        """
        global _unknown_os_type
        result = []
        master_node = nodes_info.get('master')
        standby_node = nodes_info.get('standby')
        arbitration_node = nodes_info.get('arbit')
        master_os_type = CheckOSInfoUtils.get_node_os_type_info(master_node)
        standby_os_type = CheckOSInfoUtils.get_node_os_type_info(standby_node)
        if not master_os_type or not standby_os_type:
            raise InvalidOSTypeError("Can not obtain os type of master node or standby node.")
        result.append((master_node.login_ip, master_os_type))
        result.append((standby_node.login_ip, standby_os_type))
        if master_os_type != standby_os_type:
            raise InconsistentOSTypeError("The OS types of the master and standby nodes are inconsistent. "
                                          "The OS upgrade is not supported.")
        elif master_os_type == _unknown_os_type:
            raise InvalidOSTypeError("The OS types of the master and standby nodes are unknown. "
                                     "The OS upgrade is not supported.")
        # 带仲裁场景
        if arbitration_node:
            arbitration_os_type = CheckOSInfoUtils.get_node_os_type_info(arbitration_node)
            if not arbitration_os_type:
                raise InconsistentOSTypeError(f"Failed to obtain arbitration os type. "
                                              f"Arbitration node ip: {arbitration_node.login_ip}")
            result.append((arbitration_node.login_ip, arbitration_os_type))
            # 仲裁节点操作系统类型与主备不一致
            if arbitration_os_type != master_os_type:
                raise InconsistentOSTypeError("The OS types of the master and arbitration nodes are inconsistent. "
                                              "The OS upgrade is not supported.")
        _local_logger.info("The os type is checked successfully.")
        return result

    @staticmethod
    def get_suse_kernel(nodes_info):
        """
        在所有节点上检查Suse操作系统内核
        :param nodes_info: 所有节点信息
        :return: dict(node_login_ip: kernel_version) 节点ip对应内核版本
        """
        # 检查节点内核是否有问题，默认一套环境内版本一致，只检查主备站点，三方仲裁不检查
        results = {}
        for site, node in nodes_info.items():
            if not node or not node.login_ip or site == 'arbit':
                continue
            cmd = "uname -r"
            stdout = node.execute_cmd(cmd)
            _local_logger.info("%s kernel is %s", node.login_ip, stdout)
            kernel_version = stdout.split("-default")[0]
            _local_logger.info("kernel_version: %s", kernel_version)
            problem_begin_version = "4.12.14-122.150"
            problem_end_version = "4.12.14-122.158"
            if problem_begin_version <= kernel_version <= problem_end_version:
                raise InvalidOSKernelError(f"Current kernel is {kernel_version}. Advised to install the "
                                           f"4.12.14-122.159 kernel patch or a later kernel patch.")
            else:
                results[node.login_ip] = kernel_version
                _local_logger.info("%s kernel is %s.", node.login_ip, kernel_version)
        return results

    @staticmethod
    def get_nodes_os_version(node, ip):
        """
        查询节点的OS版本
        :param node: 节点信息
        :param ip: 节点IP
        :return: str os版本信息
        """
        cmd = "uname -r"
        if ip != _local_ip:
            cmd = f"ssh {_ssh_para} {ip} \"{cmd}\""
        stdout = node.execute_su_cmd(cmd)
        return stdout

    @staticmethod
    def get_sites_os_consistence(nodes_info):
        sites = []
        for site, node in nodes_info.items():
            if not node:
                continue
            thread_pool = MultiThreadPool()
            for ip in node.ip_list:
                thread_pool.add_thread(CheckOSInfoUtils.get_nodes_os_version, node, ip)
            thread_pool.add_thread(CheckOSInfoUtils.get_nodes_os_version, node, _local_ip)
            results = thread_pool.execute_with_result()
            if "" in results:
                raise InvalidOSTypeError(f"Failed to obtain euler version on {site}.")
            if len(set(results)) == 1:
                sites.append(site)
        _local_logger.info("Sites in consistent os version: %s.", sites)
        return sites

    @staticmethod
    def get_suse_os(node):
        """
        获取节点Suse操作系统信息
        :return str Suse操作系统信息
        """
        cmd = "[ -f /etc/SuSE-release ] && echo suse || echo other"
        stdout = node.execute_cmd(cmd)
        _local_logger.info("check suse: %s", stdout)
        return stdout

    @staticmethod
    def get_suse_type(node):
        """
        获取节点Suse操作系统类型
        :return str Suse操作系统类型
        """
        cmd = "lsblk | grep root_bak &>>/dev/null || echo suse_customer"
        stdout = node.execute_cmd(cmd)
        _local_logger.info("Check suse type: %s", stdout)
        return stdout

    @staticmethod
    def get_suse_version(node):
        """
        获取节点Suse操作系统版本
        :param node: 节点信息
        :return: str Suse操作系统版本
        """
        cmd = "cat /etc/os-release | grep  'VERSION_ID=' | awk -F '=' '{print $2}' | sed 's/\"//g'"
        stdout = node.execute_cmd(cmd)
        _local_logger.info("Check suse version: %s", stdout)
        return stdout

    @staticmethod
    def get_suse_os_type(kvs):
        """
        获取客户自备Suse操作系统版本
        :param kvs: 全局节点信息
        :return: str Suse操作系统版本
        """
        nodes_info = CheckOSInfoUtils.get_node_connect_info(kvs)
        for _, node in nodes_info.items():
            if not node or not node.login_ip:
                continue
            _local_logger.info("Check os type in %s", node.login_ip)
            suse_os = CheckOSInfoUtils.get_suse_os(node)
            if "suse" not in suse_os:
                raise NoInvolvedSuseOSError(f"NO-INVOlVED-OS: {suse_os}")
            suse_type = CheckOSInfoUtils.get_suse_type(node)
            if "suse_customer" not in suse_type:
                raise NoInvolvedSuseOSError(f"NO-INVOlVED-OS: {suse_type}")
            suse_version = CheckOSInfoUtils.get_suse_version(node)
            if not ("12.4" in suse_version or "12.5" in suse_version):
                raise NoInvolvedSuseOSError(f"NO-INVOlVED-OS: suse{suse_version}")
            ver_main, ver_sub = suse_version.strip().split('.')
            return f"suse-{ver_main}sp{ver_sub}"
        # 可能节点都是非Suse系统，不能判定为异常。返回空字符串交给上层调用处理
        return ""

    @staticmethod
    def check_suse_os_type(kvs):
        """
        检查客户自带的Suse操作系统类型是否为支持类型
        :param kvs: 全局字典
        :return: str: OS类型信息
        """
        os_type = CheckOSInfoUtils.get_suse_os_type(kvs)
        if os_type not in _os_path_map:
            raise InvalidOSTypeError(f"The os type is {os_type}, which is unsupported os type.")
        return os_type

    @staticmethod
    def check_nodes_rpm(kvs, nodes_info, cur_dir):
        """
        检查所有节点组件是否齐全
        :param kvs: 全局字典
        :param nodes_info: 所有节点信息
        :param cur_dir: 检查项当前目录
        :return: None
        """
        os_type = CheckOSInfoUtils.get_suse_os_type(kvs)
        rpm_path = os.path.join(cur_dir, _os_path_map.get(os_type))
        default_rpm_set = CheckOSInfoUtils.get_default_rpm_set(rpm_path)
        thread_pool = MultiThreadPool()
        for site, node in nodes_info.items():
            if site == "arbit" or not node or not node.login_ip:
                continue
            for ip in node.ip_list:
                thread_pool.add_thread(CheckOSInfoUtils.check_node_rpm, node, ip, default_rpm_set)
            thread_pool.add_thread(CheckOSInfoUtils.check_node_rpm, node, _local_ip, default_rpm_set)
        results = thread_pool.execute_with_result()
        error_message_list = [r for r in results if r]
        if error_message_list:
            error_message = '\n'.join(error_message_list)
            error_message = f"Path of the component check list file: {rpm_path}\n" + error_message
            raise InconsistentRPMList(error_message)

    @staticmethod
    def check_node_rpm(node, ip, default_rpm_set):
        """
        检查节点rpm组件
        :param ip: 节点ip
        :param node: 节点信息
        :param default_rpm_set: 默认rpm表
        :return: str rpm差异信息
        """
        if not node or not ip:
            return ""
        env_rpm_set = CheckOSInfoUtils.get_env_rpm_set(node, ip)
        not_installed_set = default_rpm_set - env_rpm_set
        msg = ""
        if ip == _local_ip:
            ip = node.login_ip
        if not_installed_set:
            _local_logger.error("Some rpm packages is not installed in %s.", ip)
            _local_logger.error("The not installed rpm packages list: %s.", not_installed_set)
            msg = f"Some rpm packages is not installed in {ip}. " \
                  f"The not installed rpm packages list: {not_installed_set}."
        _local_logger.info("OS components check successfully in %s", ip)
        return msg

    @staticmethod
    def get_env_rpm_set(node, ip):
        """
        获取节点环境中rpm信息
        :param ip: 节点ip
        :param node: 节点信息
        :return: set 环境rpm表集合
        """
        if not node or not ip:
            return True
        cmd = "rpm -qa --qf='%{name},'"
        if ip == _local_ip:
            query_cmd = f"result=$({cmd} 2>>/dev/null);echo $result"
        else:
            query_cmd = f"result=$(ssh {_ssh_para} {ip} \"{cmd} 2>>/dev/null\");echo $result"
        if UpgradeLocalSsh.is_es_node(node.login_ip):
            stdout = node.execute_cmd(query_cmd)
        else:
            stdout = node.execute_su_cmd(query_cmd)
        return set(stdout.strip().split(","))

    @staticmethod
    def get_default_rpm_set(rpmlist_full_path):
        """
        获取默认rpm表
        :param rpmlist_full_path: rpm_list文件绝对路径
        :return: set 默认rpm表集合
        """
        default_rpm_set = set()
        if not os.path.isfile(rpmlist_full_path):
            _local_logger.error("The %s not found.", rpmlist_full_path)
            raise FileNotFoundError(f"The {rpmlist_full_path} not found.")
        with open(rpmlist_full_path, mode='r') as f:
            for line in f.read().splitlines():
                default_rpm_set.add(line)
        return default_rpm_set

    @staticmethod
    def get_rpm_version(node, rpm_name):
        """
        获取节点中特定rpm组件版本
        :param node: 节点信息
        :param rpm_name: 特定rpm名
        :return: str rpm组件版本
        """
        cmd = f"rpm -q {rpm_name}"
        stdout = node.execute_cmd(cmd)
        return stdout.split(f"{rpm_name}-")[1].split("x86_64")[0]

    @staticmethod
    def check_nodes_nscd_version(nodes_info):
        """
        检查所有节点中nscd组件版本
        :param nodes_info: 所有节点信息
        :return: None
        """
        msgs = []
        nscd_fail = False
        os_version_fail_msgs = []
        os_version_fail = False
        for _, node in nodes_info.items():
            if not node or not node.login_ip:
                continue
            ip = node.login_ip
            _local_logger.info("Check nscd in %s", ip)
            nscd_version = CheckOSInfoUtils.get_rpm_version(node, 'nscd')
            _local_logger.info("Version num: %s", nscd_version)
            suse_version = CheckOSInfoUtils.get_suse_version(node)
            _local_logger.info("Version: %s", suse_version)
            if "12.4" in suse_version:
                os_version = "SUSE 12SP4"
                target_version = "2.22-114.8.3"
            elif "12.5" in suse_version:
                os_version = "SUSE 12SP5"
                target_version = "2.22-109.2"
            else:
                os_version_fail = True
                os_version_fail_msgs.append(f"Failed to obtain the suse version on node {ip}")
            error_info = f"The version:{nscd_version} of the nscd component of the {os_version} system on node " \
                         f"{node.login_ip} is lower than {target_version}. During the upgrade process, there is a " \
                         f"low probability that a known bug in nscd will be triggered, which will cause nscd to deny " \
                         f"service and cause the upgrade to fail. It is recommended to upgrade nscd to: " \
                         f"{target_version} or above, please contact the SUSE vendor for support."
            success_info = f"The version:{nscd_version} of the nscd component in {os_version} system on node " \
                           f"{node.login_ip} meet the requirements."
            if LooseVersion(nscd_version) < LooseVersion(target_version):
                _local_logger.error(error_info)
                msgs.append(error_info)
                nscd_fail = True
            else:
                _local_logger.info(success_info)
                msgs.append(success_info)
        if os_version_fail:
            raise InvalidOSTypeError('\n'.join(os_version_fail_msgs))
        if nscd_fail:
            raise InvalidRPMVersion('\n'.join(msgs))
        return '\n'.join(msgs)

    @staticmethod
    def check_os_patch_consistence(kvs):
        """
        检查操作系统补丁一致性
        :param kvs: 全局字典
        :return: None
        """
        task_id = kvs.get('easysuite.task_id')
        os_type = kvs.get('os_type')
        all_packages = UploadPackageDao.query_upload_all_package(task_id, 'VERIFY')

        patch_version = []
        patchtool_version = []
        for package in all_packages:
            if os_type == "Euler":
                rest_patch = re.match(r"(iMasterNCE.*)(_EulerOS[0-9.]+Patch)(-[a-zA-Z0-9-]+)(.*\..*)",
                                      package.real_pkg_name)
                rest_patchtool = re.match(r"(iMasterNCE.*)(_OSPatchtool_EulerOS[0-9.]+)(-[a-zA-Z0-9-]+)(.*\..*)",
                                          package.real_pkg_name)
            else:
                rest_patch = re.match(r"(iMasterNCE.*)(_suse[0-9.]+Patch)(-[a-zA-Z0-9-]+)(.*\..*)",
                                      package.real_pkg_name)
                rest_patchtool = re.match(r"(iMasterNCE.*)(_OSPatchtool_suse[0-9.]+)(-[a-zA-Z0-9-]+)(.*\..*)",
                                          package.real_pkg_name)
            if rest_patch:
                patch_version.append(f'{rest_patch.group(1)}{rest_patch.group(3)}')
            if rest_patchtool:
                patchtool_version.append(f'{rest_patchtool.group(1)}{rest_patchtool.group(3)}')

        _local_logger.info("OSPatch version: %s", patch_version)
        _local_logger.info("OSPatchtool version: %s", patchtool_version)

        if not patch_version:
            _local_logger.error("The OSPatch package version information was not found.")
            raise InvalidPatchError("The OSPatch package version information was not found.")
        # 适配只有补丁包的企业场景
        if not patchtool_version:
            _local_logger.info("No OSPatchtool package of the matching version was found. Skip this check.")
            return
        # 判断补丁包版本一致性
        if Counter(patch_version) != Counter(patchtool_version):
            _local_logger.error("The OSPatch version does not match the OSPatchtool version. "
                                "The upgrade is not allowed.")
            raise InvalidPatchError("The OSPatch version does not match the OSPatchtool version. "
                                    "The upgrade is not allowed.")
        first_version = patch_version[0].strip().split('-', 1)[0]
        for version in patch_version + patchtool_version:
            curr_version = version.strip().split('-', 1)[0]
            if curr_version != first_version:
                _local_logger.error("The patch package has an inconsistent version: %s %s."
                                    "The upgrade is not allowed.", curr_version, first_version)
                raise InvalidPatchError(f"The patch package has an inconsistent version: "
                                        f"{curr_version} {first_version}. The upgrade is not allowed.")
