# encoding=utf-8
"""
功 能：记录日志
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
"""
import os
import sys
import json
import time
import queue
import inspect
import threading
import subprocess
from json.decoder import JSONDecodeError

from util import httpclient

NODELISTS_JSON = "/opt/oss/manager/etc/sysconf/nodelists.json"
QUERY_DBM_DB_URL = '/rest/plat/dbmgr/v1/main/instances?region=&stage=&state='
QUERY_DEPLOY_URL = '/rest/plat/sysmgr/v1/main/swmgr/product?product-name=%s'


class UpgradeCheckUtil:
    """
    功能说明:公共方法
    """

    @staticmethod
    def check_dr_info_valid(response):
        """
        功能说明:根据请求返回检查异地容灾逻辑和物理站点角色是否一致
        :params response
        :return False、dr_info
        """
        if not response:
            return False
        try:
            dr_info = json.loads(response)
        except JSONDecodeError as e:
            return False
        except TypeError as e1:
            return False
        if not isinstance(dr_info, list):
            return False
        if not dr_info:
            return False
        dr_product_info = dr_info[0]
        if dr_product_info.get('primary') == 'active':
            return True
        return False

    @staticmethod
    def check_is_dr():
        """
        功能说明:检查是否包含容灾服务
        返回 :True、False
        """
        dr_service_app = "/opt/oss/manager/apps/DRMgrService/"
        if os.path.isdir(dr_service_app):
            return True
        return False

    @staticmethod
    def send_post_request(rest_url, params):
        """
        功能说明:封装OMP post接口
        参数：rest_url, params
        返回 :True、False
        """
        status = 404
        response = ''
        for _ in range(0, 2):
            status, response = httpclient.IRHttpClient().post(rest_url, params)
            if status == 200:
                break
            # rest接口查询失败重试2次
            time.sleep(1)
        if status != 200:
            return False, ''
        return True, response

    @staticmethod
    def send_get_request(rest_url):
        """
        功能说明:封装OMP post接口
        参数：rest_url
        返回 :True、False
        """
        status = 404
        response = ''
        for _ in range(0, 2):
            status, response = httpclient.IRHttpClient().get(rest_url)
            if status == 200:
                break
            # rest接口查询失败重试2次
            time.sleep(1)
        if status != 200:
            return False, ''
        return True, response

    @staticmethod
    def get_db_info():
        """
        功能描述:检查环境升级后是否会有管理面和产品节点合设,影响管理面备份以及管理面数据库升级等操作
        参数：result
        返回 :True、False
        """
        request_result, response = UpgradeCheckUtil.send_get_request(QUERY_DBM_DB_URL)
        if not request_result:
            return []
        db_data = [i for i in json.loads(response).get('entity') if i.get('dbType') != 'redis']
        manager_gauss_dbs = [i.get('nodeID') for i in db_data if i.get('tenantName') == 'manager'
                             and i.get('dbType') == 'gauss']
        manager_zenith_dbs = [i.get('nodeID') for i in db_data if i.get('tenantName') == 'manager'
                              and i.get('dbType') == 'zenith']
        product_gauss_dbs = [i.get('nodeID') for i in db_data if i.get('tenantName') != 'manager'
                             and i.get('dbType') == 'gauss']
        product_zenith_dbs = [i.get('nodeID') for i in db_data if i.get('tenantName') != 'manager'
                              and i.get('dbType') == 'zenith']
        return [manager_gauss_dbs, manager_zenith_dbs, product_gauss_dbs, product_zenith_dbs]

    @staticmethod
    def check_extra_product_db_deployment(mgr_db_list: list, product_db_list: list) -> bool:
        """
        功能说明:检查两个集合是否满足:  1.管理面集合属于产品子集  2.有产品集合不在管理面集合上
        :param mgr_db_list: list
        :param product_db_list: list
        :return:
        """
        mgr_db_set = set(mgr_db_list)
        product_db_set = set(product_db_list)

        if (not bool(mgr_db_set)) or (not bool(product_db_set)):
            return False
        return mgr_db_set.issubset(product_db_set) and bool(product_db_set.difference(mgr_db_set))

    @staticmethod
    def deploy_pkgs_from_url():
        """
        功能说明:部署软件包
        :return:
        """
        deploy_pkgs = dict()
        product_list = UpgradeCheckUtil.get_products()
        for product in product_list:
            status, response = UpgradeCheckUtil.send_get_request(QUERY_DEPLOY_URL % product)
            packages = json.loads(response).get("entity")[0].get("packages")
            deploy_pkgs.update({product: [x.get('name') for x in packages]})
        return deploy_pkgs


    @staticmethod
    def get_products():
        """
        功能说明:获取产品列表
        :return:
        """
        product_list = []
        with open(NODELISTS_JSON, 'r') as file_obj:
            file_data = json.load(file_obj)
            for node in file_data.get("nodeList").values():
                if node.get("assignedToTenancy") not in product_list:
                    product_list.append(node.get("assignedToTenancy"))
        if "manager" in product_list:
            product_list.remove("manager")
        return product_list

    @staticmethod
    def check_dr_status():
        """
        功能说明:检查容灾系统状态
        """
        query_dr_url = "/rest/plat/drmgrservice/v1/main/sitestatus"
        query_dr_params = {"productname": []}
        if not UpgradeCheckUtil.check_is_dr():
            return False, '{}'
        ret_status, response = UpgradeCheckUtil.send_post_request(query_dr_url, query_dr_params)
        if not ret_status:
            return False, response
        return True, response

    @staticmethod
    def check_arb_exist():
        """
        功能说明:检查环境是否带仲裁
        """
        url = "/rest/drservice/v1/main/drmgr/custom/queryParam"
        param = 'true'

        ret_status, response = UpgradeCheckUtil.send_post_request(url, param)
        if not ret_status or not response:
            return False
        try:
            decode_response = response.decode()
            json_response = json.loads(decode_response)
            if json_response.get("arbiterType") == "AAMonitor" and json_response.get("autoswitch_status") == "ON":
                return True
            return False
        except Exception:
            return False


class UpgradeCheck:
    """
    功能描述：任务列表
    """

    @staticmethod
    def check_dr_consistent(result):
        """
        功能说明:检查异地容灾逻辑和物理站点角色是否一致
        参数：result, check_kvs=""
        返回 :True、False
        """
        query_dr_url = "/rest/plat/drmgrservice/v1/main/sitestatus"
        query_dr_params = {"productname": []}
        default_result = {"ret_code": 0, "response": {"dr_consistent": "false"}}
        if not UpgradeCheckUtil.check_is_dr():
            result.put(default_result)
            return False
        ret_status, response = UpgradeCheckUtil.send_post_request(query_dr_url, query_dr_params)
        if not ret_status:
            result.put(default_result)
            return False
        if UpgradeCheckUtil.check_dr_info_valid(response):
            default_result.update({"response": {"dr_consistent": "true"}})
        result.put(default_result)
        return True

    @staticmethod
    def check_is_float(result):
        """
        功能描述: 检查站点是否为异地容灾本地场景
        参数：result, check_kvs=""
        返回 :True、False
        """
        default_result = {"ret_code": 0, "response": {"with_ha_float_ip": "false",
                                                      "dist_with_ha_float_ip": "false"}}
        if not os.path.isfile(NODELISTS_JSON):
            default_result.update({"ret_code": 1})
            result.put(default_result)
            return False
        with open(NODELISTS_JSON, "r") as read_file:
            nodes_data = json.load(read_file)
        for node in nodes_data.get("nodeList").values():
            for node_ip in node.get("IPAddresses"):
                if "LB-DR-L2-ACTIVE-VIP" in node_ip.get("usage") \
                        or "LB-DR-L3-ACTIVE-VIP" in node_ip.get("usage"):
                    default_result.get("response").update({"dist_with_ha_float_ip": "true"})
                    break
                if "DRFLOATINGIP" in node_ip.get("usage"):
                    default_result.get("response").update({"with_ha_float_ip": "true"})
                    break
        result.put(default_result)
        return True

    @staticmethod
    def gen_os_key(result):
        """
        功能描述: 提供OS软件包定制key
        参数：result, check_kvs=""
        返回 :True、False
        """
        default_result = {"ret_code": 0, "response": {"is_euler_add_pkg": {},
                                                      "is_suse_add_pkg": {}}}
        product_list = UpgradeCheckUtil.get_products()
        response = default_result.get("response")
        suse = "FALSE"
        euler = "FALSE"
        return_code, _ = subprocess.getstatusoutput('cat /etc/os-release | grep -qi "suse"')
        if return_code == 0:
            suse = "TRUE"
        return_code, _ = subprocess.getstatusoutput('cat /etc/os-release | grep -qi "euler"')
        if return_code == 0:
            euler = "TRUE"
        for one_product in product_list:
            response.get("is_euler_add_pkg").update({one_product: euler})
            response.get("is_suse_add_pkg").update({one_product: suse})
        result.put(default_result)
        return True

    @staticmethod
    def gen_del_key(result):
        """
        功能描述: 提供common软件包定制key
        参数：result, check_kvs=""
        返回 :True、False
        """
        default_result = {"ret_code": 0, "response": {"is_common_del_pkg": {},
                                                      "is_common_add_pkg": {}}}
        product_list = UpgradeCheckUtil.get_products()
        response = default_result.get("response")
        for one_product in product_list:
            response.get("is_common_del_pkg").update({one_product: "TRUE"})
            response.get("is_common_add_pkg").update({one_product: "TRUE"})
        result.put(default_result)
        return True

    @staticmethod
    def check_is_assigned_to_mgr(result):
        """
        功能描述: 检查管理面节点是否托管到产品节点, 通过判断assignedToTenancy参数值是否有 manager 来实现
        参数：result, check_kvs=""
        返回 :True、False
        """
        default_result = {"ret_code": 0, "response": {"is_assigned_to_mgr": "false"}}
        if not os.path.isfile(NODELISTS_JSON):
            default_result.update({"ret_code": 1})
            result.put(default_result)
            return False
        with open(NODELISTS_JSON, "r") as read_file:
            nodes_data = json.load(read_file)
            for node in nodes_data.get("nodeList").values():
                if node.get("assignedToTenancy").lower() == "manager":
                    default_result.get("response").update({"is_assigned_to_mgr": "true"})
                    break
        result.put(default_result)
        return True

    @staticmethod
    def check_is_like_nms_server(result):
        """
        功能说明:检查环境是否为单管场景，所有数据库所在nodeID = 0
        :param result:
        :return:
        """
        default_result = {"ret_code": 0, "response": {"is_manager_domain": "false"}}
        func_result = UpgradeCheckUtil.get_db_info()
        if not func_result:
            default_result.update({"ret_code": 1})
            result.put(default_result)
            return False

        # 合并所有数据库节点ID
        func_result[0].extend(func_result[1])
        func_result[0].extend(func_result[2])
        func_result[0].extend(func_result[3])

        # 数据库节点只有0号节点,数据库节点唯一情况下,只会部署在0号节点,管理节点必定有数据库
        if len(set(func_result[0])) == 1:
            default_result.get("response").update({"is_manager_domain": "true"})
        result.put(default_result)
        return True

    @staticmethod
    def check_is_pre_upgrade_co_deployment(result):
        """
        功能说明:检查环境升级前管理面和产品是否合设,是否合设影响升级前管理面备份
        :param result:
        :return:
        """
        # 管理面和运维面同库同节点,合设
        default_result = {"ret_code": 0, "response": {"is_pre_upgrade_co_deployment": "false"}}
        func_result = UpgradeCheckUtil.get_db_info()
        if not func_result:
            default_result.update({"ret_code": 1})
            result.put(default_result)
            return False
        if list(set(func_result[0]).intersection(set(func_result[2]))) or \
                list(set(func_result[1]).intersection(set(func_result[3]))):
            default_result.get("response").update({"is_pre_upgrade_co_deployment": "true"})
            result.put(default_result)
            return True
        result.put(default_result)
        return True

    @staticmethod
    def check_is_pre_upgrade_extra_db_deployment(result):
        """
        功能说明:检查环境升级前管理面和产品是否有合设节点, 并且产品DB有额外节点不在管理面节点上
        :param result:
        :return:
        """
        # 管理面和运维面同库同节点,合设
        default_result = {"ret_code": 0, "response": {"is_pre_upgrade_extra_db_deployment": "false"}}
        func_result = UpgradeCheckUtil.get_db_info()
        if not func_result:
            default_result.update({"ret_code": 1})
            result.put(default_result)
            return False

        if UpgradeCheckUtil.check_extra_product_db_deployment(func_result[0], func_result[2]) or \
                UpgradeCheckUtil.check_extra_product_db_deployment(func_result[1], func_result[3]):
            default_result.get("response").update({"is_pre_upgrade_extra_db_deployment": "true"})
        result.put(default_result)
        return True

    @staticmethod
    def check_is_co_deployment(result):
        """
        功能描述:检查环境升级后是否会有管理面和产品节点合设,影响管理面备份以及管理面数据库升级等操作
        参数：result
        返回 :True、False
        """
        default_result = {"ret_code": 0, "response": {"is_co_deployment": "false"}}
        func_result = UpgradeCheckUtil.get_db_info()
        if not func_result:
            default_result.update({"ret_code": 1})
            result.put(default_result)
            return False
        func_result[0].extend(func_result[1])
        func_result[2].extend(func_result[3])
        if list(set(func_result[0]).intersection(set(func_result[2]))):
            default_result.get("response").update({"is_co_deployment": "true"})
            result.put(default_result)
            return True
        result.put(default_result)
        return True

    @staticmethod
    def get_deploy_pkgs(result):
        """
        功能说明:补齐软件包部署包key值
        :param result:
        :return:
        """
        default_result = {"ret_code": 0, "response": {"A": {"NCE": "TRUE"}}}
        deploy_pkgs = UpgradeCheckUtil.deploy_pkgs_from_url()
        response = default_result.get("response")
        for one_product, pkgs in deploy_pkgs.items():
            for pkg_name in pkgs:
                if response.get(pkg_name):
                    response.get(pkg_name).update({one_product: "TRUE"})
                else:
                    response[pkg_name] = {one_product: "TRUE"}
                response.get(pkg_name).update({x: "FALSE" for x in deploy_pkgs.keys()
                                               if x not in response.get(pkg_name).keys()})
        result.put(default_result)
        return True

    @staticmethod
    def check_protection_hot_and_arb_exist(result):
        """
        功能说明:检查当前环境是否是容灾带仲裁
        """
        default_result = {"ret_code": 0, "response": {"site_passwd_same": "no"}}
        is_protection_hot, _ = UpgradeCheckUtil.check_dr_status()
        if is_protection_hot and UpgradeCheckUtil.check_arb_exist():
            default_result.get("response").update({"site_passwd_same": "yes"})
        result.put(default_result)
        return True


class GetExtend:
    """
    功能描述：要求所有操作能够并发执行
    """

    @staticmethod
    def get_check_list():
        """
        功能描述：并发执行所有升级任务
        参数：无
        返回：fun_list
        """
        fun_list = []
        for name, obj in inspect.getmembers(UpgradeCheck):
            if inspect.isfunction(obj):
                fun_list.append(name)
        return fun_list

    @staticmethod
    def mul_thread_check():
        """
        功能描述：并发执行所有升级任务,不允许有互相依赖。有依赖的合并为一个操作,key重复直接覆盖
        参数：check_kvs
        返回：True、False, ret_dict
        """
        check_list = GetExtend.get_check_list()

        # 并发下任务
        check_result = queue.Queue()
        child_task_threads = []
        for one_check in check_list:
            child_task_threads.append(
                threading.Thread(target=getattr(UpgradeCheck(), one_check),
                                 args=(check_result,)))
        for child_task_thread in child_task_threads:
            child_task_thread.start()
        for child_task_thread in child_task_threads:
            child_task_thread.join()

        # 解析执行后返回的键值
        result = True
        ret_dict = {}
        for _ in range(0, check_result.qsize()):
            q_result = check_result.get()
            if q_result:
                if q_result.get("ret_code") == 0:
                    ret_dict.update(q_result.get("response"))
                else:
                    result = False
        if not result:
            return False, ret_dict
        return True, ret_dict

    @staticmethod
    def write_ret_data():
        """
        功能描述：执行结果写入json文件
        参数：无
        返回值：True、False
        """
        script_path = sys.path[0]
        file_path = os.path.join(script_path, "extendinfo_NCE_Common.json")
        result, data = GetExtend.mul_thread_check()
        if not data:
            return False

        # 写入json文件
        with os.fdopen(os.open(file_path,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o660), "w", encoding="utf-8") as file_obj:
            file_obj.write(json.dumps(data))

        if not result:
            return False
        return True


def main():
    """
    功能说明:主入口
    """
    if not GetExtend.write_ret_data():
        return False
    return True


if __name__ == '__main__' and not main():
    sys.exit(1)
