import json
import os

from exception.exception import UnclearedAlarmCheckException
from utils.common import http_get, http_post, common_http_post
from utils.manager import Manager
from utils.log_utils import get_logger

LOGGER = get_logger()
ERROR_CODE = 50


class UnclearedHardwareAlarmCheck(object):
    # 查询管理面告警接收参数接口
    QUERY_ALARM_IP_REST = "/rest/plat/engrnotify/v1/ui/e9000nodes"
    # 查询未清除告警接口
    QUERY_UNCLEARED_ALARM_REST = '/rest/fault/v1/current-alarms/scroll'
    # 备站点硬件告警IP存储文件
    STANDBY_HARDWARE_IP = "/opt/oss/tmp/manager/HardwareAlarmCheck/standby_hardware_ip.json"

    def get_ip_info_list(self) -> list:
        """
        获取硬件IP信息
        """
        result = http_get(self.QUERY_ALARM_IP_REST)
        web_alarm_info = result.get("resultMapList", [])
        if result.get("retcode") != 0 or not web_alarm_info:
            LOGGER.error("Get alarm ip info by rest failed")
            raise UnclearedAlarmCheckException("Failed to get alarm ip info by restful request")
        return [node.get("nodeIP", "").strip() for node in web_alarm_info]

    def get_standby_ip_list(self) -> list:
        if not os.path.exists(self.STANDBY_HARDWARE_IP):
            return []
        with open(self.STANDBY_HARDWARE_IP, mode="r", encoding="utf-8") as file_obj:
            content = file_obj.read()
            json_content = json.loads(content)
        return [node_ip.strip() for node_ip in json_content.get("standby_hardware_ip").split(",") if node_ip]

    def get_uncleared_alarm(self) -> list:
        """
        获取未清除告警列表
        """
        params = {
            "query": {
                "filters": [{
                    "name": "cleared",
                    "field": "cleared",
                    "operator": "=",
                    "values": ["0"]
                }, {
                    "name": "merged",
                    "field": "merged",
                    "operator": "=",
                    "values": [1, 0]
                }],
                "expression": "and"
            }
        }
        result = http_post(self.QUERY_UNCLEARED_ALARM_REST, params)
        # 低版本NCE无法调用IR端口，切换32018进行post request
        if not result:
            result = common_http_post(self.QUERY_UNCLEARED_ALARM_REST, params, Manager().service_ip, 32018)
        uncleared_alarm_info = result.get('hits', [])
        if result.get("resCode") != 1:
            LOGGER.error("Get uncleared alarm info by rest failedd.")
            raise UnclearedAlarmCheckException("Failed to get uncleared alarm info by restful request")
        if not uncleared_alarm_info:
            LOGGER.info('Uncleared alarm list is emtpy.')
        return uncleared_alarm_info

    def check_uncleared_alarm(self) -> list:
        """
        执行未清除告警检查
        """
        LOGGER.info('Start to check uncleared alarm.')
        check_result = []
        ip_info_list = self.get_ip_info_list()
        LOGGER.info("Hardware IP: %s", ip_info_list)
        standby_ip_list = self.get_standby_ip_list()
        LOGGER.info("Standby hardware IP: %s", standby_ip_list)
        uncleared_alarm_list = self.get_uncleared_alarm()
        for node_ip in set(ip_info_list + standby_ip_list):
            if not node_ip:
                continue
            for alarm in uncleared_alarm_list:
                ip_regex = "=" + node_ip + ','
                moi = alarm.get('moi', '')
                if ip_regex in moi:
                    msg = self.get_message(node_ip, alarm)
                    check_result.append(msg)
        LOGGER.info("Uncleared alarm are: %s", check_result)
        return check_result

    def get_message(self, ip, alarm) -> str:
        """
        打印未清除告警信息
        :param ip: 硬件IP
        :param alarm: 告警信息
        """
        alarm_id = alarm.get('alarmId', '')
        alarm_name = alarm.get('alarmName', '')
        return f"Error code is {ERROR_CODE}; Error information: Alarm ID:{alarm_id};IP:{ip}; Alarm name:{alarm_name}."
