# -*- coding: utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import json
import shlex
import subprocess
import time
import types
from functools import wraps

from constant.constant import HTTP_SUCCESS_CODE
from utils.log_utils import get_logger
from util import httpclient

LOGGER = get_logger()

# 平台调用rest接口客户端
HTTP_CLIENT = httpclient.IRHttpClient()

class LazyProperty(object):
    """ 懒加载装饰器 """

    def __init__(self, func):
        self.func = func  # lazy的实例属性func等于instance方法对象

    def __get__(self, instance, cls):
        value = self.func(instance)  # 调用func方法计算出结果
        instance.__setattr__(self.func.__name__, value)  # 将结果设置给instance的func属性
        return value


def retry(times=3, interval=3, success_condition: types.FunctionType = lambda _: _):
    """
    重试装饰器
    :param times: 重试次数
    :param interval: 重试间隔：秒
    :param success_condition: 判断成功条件
    :return: func result
    """

    def decorator(func):
        @wraps(func)
        def wrap(*args, **kwargs):
            result = None
            for _ in range(times):
                result = func(*args, **kwargs)
                if success_condition(result):
                    return result
                time.sleep(interval)
            return result

        return wrap

    return decorator


@retry()
def http_get(url) -> dict:
    """
    通过平台 httpclient 工具，调用rest get接口
    :param url: get接口url
    :return: dict get接口信息
    """
    LOGGER.info(f"start to get get rest result: {url}")
    try:
        response = HTTP_CLIENT.get(url)
        if not response or response[0] != HTTP_SUCCESS_CODE or len(response) < 2:
            LOGGER.error(f"get get rest result: {url} failed, response: {response}")
            return {}
        return json.loads(response[1])
    except Exception as exp:
        LOGGER.error(f"get {url} catch an exception: {exp}")
        return {}


@retry()
def http_post(url, body) -> dict:
    """
    通过平台 httpclient 工具，调用rest post接口
    :param url: post接口url
    :param body: post接口body体
    :return: dict post接口信息
    """
    LOGGER.info(f"start to get post rest result: {url}")
    try:
        response = HTTP_CLIENT.post(url, body)
        if not response or response[0] != HTTP_SUCCESS_CODE or len(response) < 2:
            LOGGER.error(f"get post rest result: {url} failed, response {response}")
            return {}
        return json.loads(response[1])
    except Exception as exp:
        LOGGER.error(f"get post {url} catch an exception: {exp}")
        return {}


@retry()
def common_http_post(url, body, ip, port) -> dict:
    """
    通过平台 httpclient 工具，调用common rest post接口
    :param url: post接口url
    :param body: post接口body体
    :param ip: post接口对端IP
    :param port: post接口对端端口
    :return: dict post接口信息
    """
    LOGGER.info(f"start to get post rest result by common http request: {url}")
    try:
        LOGGER.info(f"common http request, ip is {ip}, and port is {port}")
        common_http_client = httpclient.CommonHttpClient(ip, port, True, False)
        response = common_http_client.post(url, body)
        if not response or response[0] != HTTP_SUCCESS_CODE or len(response) < 2:
            LOGGER.error(f"get post rest result: {url} failed, response {response}")
            return {}
        return json.loads(response[1])
    except Exception as exp:
        LOGGER.error(f"get post {url} catch an exception: {exp}")
        return {}


@retry(times=2, success_condition=lambda result: result[0] == 0)
def exec_cmd(cmd: str, pwd: str = "", timeout: int = 30, is_log = True) -> tuple:
    """
    执行shell命令
    :param cmd:待执行的命令（带参数）
    :param pwd:涉及密码的命令，单独传入
    :param timeout:执行超时时间
    :param is_log:是否记录执行日志
    :return: tuple: result, stdout, stderr
    """
    LOGGER.info(f"start to exec cmd {cmd}")
    try:
        stream = subprocess.Popen(shlex.split(cmd), stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE, shell=False, universal_newlines=True)
        if pwd:
            stdout, stderr = stream.communicate(pwd, timeout=timeout)
        else:
            stdout, stderr = stream.communicate(timeout=timeout)
        if is_log:
            LOGGER.info(f"exec {cmd} return code {stream.returncode} stdout {stdout}, stderr: {stderr}")
        return stream.returncode, stdout, stderr
    except Exception as _:
        LOGGER.error(f"exec cmd {cmd} catch an exception")
        return -1, "", ""


@retry(success_condition=lambda result: result[0] == 0)
def exec_pipe_cmd(cmd: str, cmd02: str = "", timeout: int = 30) -> tuple:
    """
    执行shell带管道的命令
    :param cmd: 管道 '|' 之前执行的命令
    :param cmd02: 管道 '|' 之后执行的命令
    :param timeout:执行超时时间
    :return: tuple: result, stdout, stderr
    """
    LOGGER.info(f"start to exec cmd {cmd} {cmd02}")
    try:
        stream01 = subprocess.Popen(shlex.split(cmd), stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE, shell=False, universal_newlines=True)
        stream02 = subprocess.Popen(shlex.split(cmd02), stdin=stream01.stdout, stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE, shell=False, universal_newlines=True)
        stdout, stderr = stream02.communicate(timeout=timeout)
        LOGGER.info(f"exec {cmd} {cmd02} return code {stream02.returncode} stdout {stdout}, stderr: {stderr}")
        return stream02.returncode, stdout, stderr
    except Exception as _:
        LOGGER.error(f"exec cmd {cmd} {cmd02} catch an exception")
        return -1, "", ""


def check_ping(source_ip: str = "", target_ip: str = "") -> bool:
    """
    通过shell的ping命令，判断从源IP到目标IP是否可达
    :param source_ip: 源IP地址
    :param target_ip: 目标IP地址
    :return: bool 可达 True,不可达 False
    """
    cmd = f"ping -c 3 -I {source_ip} {target_ip}"
    ret, stdout, stderr = exec_cmd(cmd)
    return ret == 0
