# -*- coding: utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import os
import json

from data.service_info import ServiceInfo, get_service_info
from utils.common import LazyProperty, exec_pipe_cmd, exec_cmd, http_get, http_post
from constant.constant import InstallType, DEFAULT_PAGESIZE, DEFAULT_PAGE, LOG_DIR, SQL_CMD
from exception.exception import HealthCheckException
from utils.log_utils import get_logger
from utils.singleton import Singleton
from util import ossext

LOGGER = get_logger()


class Manager(Singleton):
    """ 管理面信息统一管理 单实例 """

    # nodelists.json路径
    NODE_LIST_FILE = "/opt/oss/manager/etc/sysconf/nodelists.json"
    # 平台管理面告警监听IP的默认网口用途，若未找到该网口用途则认为是私有云场景
    ALARM_USAGE = "MGMTALM_SBI"
    MAINTENANCE_USAGE = "maintenance"
    # 告警查询脚本
    QUERY_ALARM_SCRIPT = "/opt/oss/manager/bin/queryengrnotifyconfig.sh"
    # 告警查询输出结果
    OUTPUT_FILE = os.path.join(LOG_DIR, "alarm.json")
    # 查询管理面告警接收参数接口
    QUERY_ALARM_REST = "/rest/plat/engrnotify/v1/ui/e9000nodes"
    # 校验管理面告警接受参数接口
    VERIFY_ALARM_REST = "/rest/plat/engrnotify/v1/service/ir/devices/validator"
    # 健康检查信息查询接口
    HEALTH_CHECK_REST = "/rest/smpmanagerservice/v1/healthcheck/queryhardwaredetails"
    # 健康检查微服务配置文件
    DFS_SERVICE_CONFIG_FILE = "/opt/oss/manager/apps/SMPManagerService/envs/env.properties"
    # the key of database
    DATABASES_KEY = "databases"
    # the key of smp manager service db
    SMP_MANAGER_SERVICE_DB_KEY = "smpmanagerservicedb"
    # the key of pass
    PASS_WD_KEY = "passwd"
    # 健康检查参数list
    HEALTH_CHECK_SERVICE_LIST = []

    def __init__(self):
        super().__init__()

    @LazyProperty
    def node_list_content(self) -> dict:
        """ 读取产品的nodelists.json内容并以dict返回 """
        if not os.path.exists(self.NODE_LIST_FILE):
            raise HealthCheckException("node list json file is not exist")
        with open(self.NODE_LIST_FILE, mode="r", encoding="utf-8") as file_obj:
            node_list_content = json.load(file_obj)
        LOGGER.info(f"get node list json is {node_list_content}")
        return node_list_content

    @LazyProperty
    def install_type(self) -> InstallType:
        """
        返回安装的类型：单管，小型化，分布式 三种类型
        :return: InstallType
        """
        manager_nodes = 0
        for _, value in self.node_list_content.get("nodeList", {}).items():
            if value.get("assignedToTenancy") == "manager":
                manager_nodes += 1
        if manager_nodes == InstallType.SINGLE.value:
            return InstallType.SINGLE
        if manager_nodes == InstallType.MINI.value:
            return InstallType.MINI
        if manager_nodes == InstallType.DISTRIBUTED.value:
            return InstallType.DISTRIBUTED
        return InstallType.UNKNOWN

    @LazyProperty
    def alarm_listening_ip_list(self) -> list:
        """
        获取当前节点管理面告警监听IP地址列表
        :return: list 告警监听IP
        """
        LOGGER.info("start to get local alarm listening ip")
        alarm_ip_list = []
        for _, value in self.node_list_content.get("nodeList", {}).items():
            ip_address_list = value.get("IPAddresses", [])
            alarm_ip = [ip_address.get("IP", "") for ip_address in ip_address_list if
                        self.ALARM_USAGE in ip_address.get("usage", [])]
            alarm_ip_list.extend(alarm_ip)
        LOGGER.info(f"get alarm listening ip list is {alarm_ip_list}")
        return alarm_ip_list

    @LazyProperty
    def alarm_listening_ip(self) -> str:
        """
        获取当前节点管理面告警监听IP地址
        :return: str 本地告警监听IP(分布式场景下，返回当前OMP节点下的告警监听IP地址)
        """
        for alarm_ip in self.alarm_listening_ip_list:
            ret, stdout, stderr = exec_pipe_cmd(cmd="ifconfig", cmd02=f"grep -w {alarm_ip}")
            if ret == 0:
                return alarm_ip
        return ""

    @LazyProperty
    def alarm_listening_port(self) -> str:
        """ 告警监听端口，默认为30085 """
        return "30085"

    @LazyProperty
    def alarm_receive_info(self) -> list:
        """ 获取管理面配置的告警接收参数信息 """
        if os.path.exists(self.OUTPUT_FILE):
            os.remove(self.OUTPUT_FILE)
        cmd = f"bash {self.QUERY_ALARM_SCRIPT} -output {self.OUTPUT_FILE}"
        ret, stdout, stderr = exec_cmd(cmd)
        if ret != 0:
            LOGGER.error(f"get alarm info by shell {cmd} failed, stdout: {stdout}, stderr: {stderr}")
            return []
        if not os.path.join(self.OUTPUT_FILE):
            raise HealthCheckException("query error")
        with open(self.OUTPUT_FILE, mode="r", encoding="utf-8") as file_obj:
            alarm_info = json.load(file_obj)
        result = http_get(self.QUERY_ALARM_REST)
        web_alarm_info = result.get("resultMapList", [])
        if result.get("retcode") != 0 or not web_alarm_info:
            LOGGER.error("get alarm info by rest failed")
            return []
        for alarm in alarm_info:
            for web_alarm in web_alarm_info:
                if alarm.get("nodeIP", "") == web_alarm.get("nodeIP", ""):
                    alarm.update({
                        "notifyID": web_alarm.get("notifyID", ""),
                        "snmpAuth": web_alarm.get("snmpAuth", "")
                    })
                    break
        return alarm_info

    @LazyProperty
    def service_ip(self) -> str:
        """ 获取业务面IP地址 """
        for _, value in self.node_list_content.get("nodeList", {}).items():
            if value.get("assignedToTenancy") != "NCE":
                continue
            ip_address_list = value.get("IPAddresses", [])
            service_ip = [ip_address.get("IP", "") for ip_address in ip_address_list if
                          self.MAINTENANCE_USAGE in ip_address.get("usage", [])]
            if service_ip:
                return service_ip[0]
        return ""

    def verify_alarm_receive_param(self, body: dict) -> bool:
        """ 使用平台接口，校验配置的告警接收参数 """
        result = http_post(self.VERIFY_ALARM_REST, [body])
        return result and result[0].get("result", "-1") == "0"

    def get_service_info_list_v2(self, service_type: str) -> list:
        """
        通过DFS的接口，获取管理面健康检查中配置的告警信息
        :param service_type: ServiceType 服务器、FC、交换机
        :return: list[ServiceInfo]
        """
        LOGGER.info(f"start to get {service_type} info from dfs rest interface")
        params = []
        page_size, current_page, total_count = DEFAULT_PAGESIZE, DEFAULT_PAGE, DEFAULT_PAGESIZE
        while total_count >= page_size:
            url = f"{self.HEALTH_CHECK_REST}?" \
                  f"type={service_type}&pageSize={str(page_size)}&currentPage={str(current_page)}"
            result = http_get(url)
            if not result:
                raise HealthCheckException("get service info from dfs interface failed")
            for data in result.get("data", []):
                if data.get("serviceType") != service_type:
                    continue
                params.append(get_service_info(data))
            page_size = result.get("pageSize", DEFAULT_PAGESIZE)
            total_count = result.get("totalCount", 0)
            current_page += 1
        LOGGER.info(f"query {service_type} info from dfs interface success")
        return params

    def get_db_connect_info(self, config_file) -> tuple:
        """
        获取数据库连接信息
        :param config_file: 配置文件
        :return: (数据库连接IP，数据库连接端口，数据库连接密码)
        """
        if not os.path.isfile(config_file):
            raise HealthCheckException(f"Get APP_CONF_FILE failed, for {config_file} is not file")
        with open(os.path.realpath(config_file)) as app_cfg_file:
            app_cfg_data = json.load(app_cfg_file)
            if self.DATABASES_KEY in app_cfg_data:
                if self.SMP_MANAGER_SERVICE_DB_KEY in app_cfg_data.get(self.DATABASES_KEY):
                    db_list = app_cfg_data.get(self.DATABASES_KEY).get(self.SMP_MANAGER_SERVICE_DB_KEY)
                    if isinstance(db_list, list) and len(db_list) >= 0:
                        host = db_list[0].get('accessHost')
                        port = db_list[0].get('accessPort')
                        password = ossext.Cipher.decrypt(db_list[0].get(self.PASS_WD_KEY))
                        return host, port, password
        return "", "", ""

    def get_service_info_list_v1(self) -> list:
        """
        老版本NCE（R22C10之前），连接数据库获取健康检查配置信息
        :return: list[ServiceInfo]
        """
        LOGGER.info("enter get_service_info_list_v1, get service info by db")
        if Manager.HEALTH_CHECK_SERVICE_LIST:
            return Manager.HEALTH_CHECK_SERVICE_LIST
        # 获取SMPManagerService配置文件
        ret, stdout, stderr = exec_pipe_cmd(f"cat {self.DFS_SERVICE_CONFIG_FILE}", "grep APP_CONF_FILE")
        if ret != 0:
            raise HealthCheckException("Get APP_CONF_FILE failed from file")
        app_config_file_list = [out.strip() for out in stdout.strip().split("=")]
        if len(app_config_file_list) != 2:
            raise HealthCheckException("Get APP_CONF_FILE failed")
        app_config_file = app_config_file_list[1]
        LOGGER.info(f"Get app config file is {app_config_file}")
        # 从配置文件中获取数据库连接信息
        host, port, password = self.get_db_connect_info(app_config_file)
        # 执行命令，获取信息
        ret, stdout, stderr = exec_cmd(f"bash {SQL_CMD} SMPMANAGERSERVICEDB {host} {port}", pwd=password, is_log=False)
        if ret != 0:
            raise HealthCheckException("get hardware service info failed")
        start = False
        for line in stdout.split("\n"):
            if "----" in line:
                start = True
                continue
            if start and line:
                line_list = [cont.strip() for cont in line.split(" ") if cont]
                if len(line_list) < 4:
                    break
                Manager.HEALTH_CHECK_SERVICE_LIST.append(
                    ServiceInfo(service_type=line_list[0], service_ip=line_list[1], service_user=line_list[2],
                                service_pwd=line_list[3]))
        return Manager.HEALTH_CHECK_SERVICE_LIST

    def get_service_info_list(self, service_type: str) -> list:
        """
        获取管理面健康检查中配置的信息
        :return: list[ServiceInfo]
        """
        if Manager.HEALTH_CHECK_SERVICE_LIST:
            return [service for service in self.get_service_info_list_v1() if service.service_type == service_type]
        try:
            return self.get_service_info_list_v2(service_type)
        except HealthCheckException as _:
            return [service for service in self.get_service_info_list_v1() if service.service_type == service_type]

    def clean_tmp_file(self) -> None:
        """ 清理生成的中间文件 """
        if os.path.exists(self.OUTPUT_FILE):
            os.remove(self.OUTPUT_FILE)
