# encoding=utf-8
"""
功 能：上传备份产品数据包
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2022-05-21 23:17 创建
"""
import sys
import os
import json
import time

from datetime import datetime
from commonlog import Logger
from taskmgr_util import Taskmgrutil
from uniep_taskmgr import Unieptaskmgr

# 日志级别枚举
INFO = "INFO"
ERROR = "ERROR"
LOGGER = Logger().getinstance(sys.argv[0])

# 上传备份包接口
UPLOAD_BACKUP_PACKAGE_URL = f"/rest/plat/brmgr/v1/backupforspec/backupdataaction?action=upload"
# 备份时间戳文件
TIMESTAMP_FILE = '/opt/upgrade/backup/backup_timestamp.properties'
# 升级步骤名
STEP_NAME = "upload_backup_package"

# 任务管理状态
FAILURE_STATE = ["FAILURE", "PARTIAL_SUCCESS", "TERMINATING", "TERMINATED"]
RUNNING_STATE = ["INIT", "RUNNING", "WAITING"]

# 查询进度状态枚举
FINISH = 0
RUNNING = 1
FAULT = 2


def build_upload_param(product_name, timestamp):
    """
    上传参数
    """
    param = dict(
        action="upload",
        mode="upgrade",
        timestamp=timestamp,
        productName=product_name,
        mutexLevel="none",
        backupPath=f"/opt/backup/backuptmp/{product_name}/upgrade",
        resource={
            "mode": {
                "en_US": "Upgrade",
                "zh_CN": "升级"
            }
        }
    )
    return param
	
	
class UploadBackupPackage:
    """
    上传备份产品数据包
    """

    def __init__(self, args):
        LOGGER.info("Init upload backup product package.")
        self.script_id = args.get("script_id", "")
        self.upgrade_path = args.get("upgrade_path", "")
        self.product_name = args.get("product_name", "")
        self.action = args.get("action", "")
        self.backup_product_diff = args.get("backup_product_diff", "")

        e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
        self.path = os.path.join(e_taskmgr_path, self.script_id)
        self.rest = Unieptaskmgr()
        self.taskmgr = Taskmgrutil()
        self.taskmgr.init_e_taskmgr(self.path)
        self.msg_list = []

    def record_log(self, msg, level=""):
        """
        更新任务process、status、和日志
        :param msg:
        :param level:
        :return:
        """
        now_time = datetime.strftime(datetime.now(), '%Y-%m-%d %H:%M:%S')
        str_msg = "[%s] [%s] %s | %s" % (now_time, os.getpid(), level, msg)
        self.msg_list.append(str_msg)
        self.taskmgr.set_e_taskmsg(self.path, "\n".join(self.msg_list))
        if level == ERROR:
            self.taskmgr.set_e_taskstatus(self.path, "error")
            self.taskmgr.set_e_taskprogress(self.path, "100")
            LOGGER.error(msg)
        else:
            LOGGER.info(msg)
        return True

    def check_progress(self, url):
        """
        检查上传进度
        :param url:查询接口
        :return:
        """
        try:
            status, result = self.rest.send_get_request(url)
            if status:
                result = json.loads(result.decode())[0]
                execute_result = result.get("currentState")
                progress = result.get("progress", "0")
                if not execute_result or execute_result in FAILURE_STATE:
                    return FAULT, progress
                if execute_result in RUNNING_STATE:
                    return RUNNING, progress
                return FINISH, progress
            return FAULT, "0"
        except (TypeError, ValueError) as e_msg:
            self.record_log("Failed to query task msg. Exception:{}".format(e_msg), ERROR)
            return FAULT, "0"

    def query_progress(self, url, timestamp):
        """
        查询上传任务进度
        :param url:查询接口
        :param timestamp:备份时间戳
        :return:
        """
        # 升级只创建任务，不查询任务结果
        if self.action == "upgrade":
            self.record_log(f"Finish to create upload task for {timestamp}.", INFO)
            return True

        # 查询上传任务进度
        self.record_log(f"Start to query upload task progress for {timestamp}.", INFO)
        max_wait_seconds = 14400
        delay_seconds = 10
        end_time = int(time.monotonic()) + max_wait_seconds
        status = RUNNING
        while status != FINISH:
            status, progress = self.check_progress(url)
            if status == FAULT:
                self.record_log(f"Failed to query upload task progress for {timestamp}.", ERROR)
                return False
            self.record_log("Check upload process is running, execution progress is %s%%" % progress, INFO)
            if status == FINISH:
                self.record_log(f"Finish to execute upload task for {timestamp}.", INFO)
                return True
            time.sleep(delay_seconds)
            if int(time.monotonic()) >= end_time:
                self.record_log("Check upload progress timeout, waiting for %d seconds" % max_wait_seconds, ERROR)
                return False
        return True

    def upload_backup_package(self):
        """
        上传备份包
        :return:
        """
        # 查询备份产品数据时间戳
        timestamp_list = self.query_timestamp_list()
        for timestamp in timestamp_list:
            self.record_log(f"Start to upload backup product package for {timestamp}.", INFO)
            # 构造上传接口参数
            upload_param = build_upload_param(self.product_name, timestamp)
            status, response = self.rest.send_post_request(UPLOAD_BACKUP_PACKAGE_URL, upload_param)
            if not status:
                self.record_log(f"Failed to upload backup product package for {timestamp}.", ERROR)
                return False

            # 解析上传接口返回
            response = json.loads(response.decode())
            task_url = response.get("url", "")
            if not task_url:
                task_result = response.get("result", "")
                task_entity = response.get("entity", None)
                if task_result == "ERROR" and task_entity is not None:
                    self.record_log(
                        "Failed to create task. Error msg:{}".format(task_entity.get("details", "None")), ERROR)
                else:
                    self.record_log("Failed to upload backup product package.", ERROR)
                return False

            # 查询上传进度
            if not self.query_progress(task_url, timestamp):
                return False
        self.taskmgr.set_e_taskstatus(self.path, "finish")
        self.taskmgr.set_e_taskprogress(self.path, "100")
        return True

    def query_timestamp_list(self):
        """
        查询备份时间戳
        """
        # 待上传的产品数据备份时间戳列表
        timestamp_list = []
        # 备份类型（包含增量备份）
        backup_type_list = ["backup_product"]
        if self.backup_product_diff == "true":
            backup_type_list = ["backup_product_full", "backup_product_diff"]
        # 解析时间戳列表
        for backup_type in backup_type_list:
            timestamp = ""
            timestamp_str = f"{backup_type}_{self.product_name}_{self.upgrade_path}"
            # 备份时间戳记录
            timestamp_file_content = open(TIMESTAMP_FILE, "r", encoding="utf-8")
            # 取最近的一条记录
            for line in timestamp_file_content:
                if line.find(timestamp_str) > -1:
                    timestamp = line.split(":")[1].strip()
            timestamp_list.append(timestamp)
        self.record_log(f"timestamp_list is {timestamp_list}", INFO)
        return timestamp_list

    def check_finish(self):
        """
        根据script_id检查上次任务是否成功
        :return:
        """
        if STEP_NAME not in self.script_id:
            return False
        cmd = "grep %s %s" % (self.script_id, LOGGER.logfile)
        ret_code = os.system(cmd)
        if ret_code == 0:
            self.record_log("Finish to execute upload backup product package task.", INFO)
            self.taskmgr.set_e_taskstatus(self.path, "finish")
            self.taskmgr.set_e_taskprogress(self.path, "100")
            return True
        return False


def main(argv):
    """
    上传产品数据备份包
    :return:
    """
    script_id = argv[1]
    upgrade_path = argv[2]
    product_name = argv[3]
    action = argv[4]
    backup_product_diff = argv[5]
    args = {}
    args.update({
        "script_id": script_id,
        "upgrade_path": upgrade_path,
        "product_name": product_name,
        "action": action,
        "backup_product_diff": backup_product_diff
    })

    upload_function = UploadBackupPackage(args)
    try:
        if not upload_function.check_finish():
            return upload_function.upload_backup_package()
        return True
    except Exception as e:
        upload_function.record_log(f"Failed to upload backup product package. Exception:{e}", ERROR)
        return False


if __name__ == '__main__':
    main(sys.argv)
