#!/bin/bash
# 脚本功能：作为维护工具,跳过DFS巡检NOGO项

# 脚本路径
SCRIPT_PATH=$(cd $(dirname $0);pwd)
# 脚本自身
SELF_FILE=$(basename $0)
# ES数据库文件所在位置
ES_DB="/opt/oss/easysuite/easysuite/easysuite/db/easysuite.sqlite3"
# 不满足前提条件
PRE_ERROR_CODE=100
# 数据库状态更新失败
POST_ERROR_CODE=101


##################################################
# 功能：检查脚本执行用户是否合法
##################################################
function check_user()
{
    # 3001为ossadm用户ID
    if [ "$(whoami)" != "$(id -nu 3001)" ]
    then
        echo "User have no permission to run this script"
        return 1
    fi
}

########################
# 功能: 脚本调用方式
########################
function usage()
{
    echo "Usage:
    ${SELF_FILE} -task_id task_id
command:
    task_id                  : ."
}

########################
# 功能: 脚本调用方式
########################
function log_and_print()
{
    local level="${1}"
    local msg="${2}"
    case ${level} in
    "ERROR")
        logger -p local0.err "${msg}"
        echo "${msg}"
    ;;
    "INFO")
        logger -p local0.err "${msg}"
        echo "${msg}"
    ;;
    esac
    return
}

########################
# 功能: 参数校验
# @param task_id: 任务ID
########################
function check_params()
{
    # 参数数量
    [ $# != 2 ] && return 1
    # 参数合规校验
    [[ "${2}" =~ ^[0-9]+$ ]] || return 1
}

########################
# 功能: 获取数据库查询结果
# @param task_id: 任务ID
########################
function query_db_status()
{
    local task_id="${1}"
    local status="${2}"
    sqlite3 "${ES_DB}" <<END | grep -q ${status}
select status from task_main where id="${1}";
.quit
END

    # 刷新成功返回0,不成功非零
    return $?
}

########################
# 功能: 更新数据库任务状态
# @param task_id: 任务ID
########################
function modify_db_status()
{
    local task_id="${1}"
    sqlite3 "${ES_DB}" <<END
update task_main set status='upgrade_pre_check_finish' where id="${task_id}" and status="upgrade_pre_check_error";
.quit
END
}

########################
# 功能: 更新产品配置文件关闭nogo项开关
# @param task_id: 任务ID
########################
function modify_upgrade_config()
{
    local task_id="${1}"
    find /opt/oss/easysuite/easysuite/var/task_config/${task_id}/*/website/data/ -type f -name "upgrade_params_config.json" | xargs grep -E 'health_check_set": *"true"'

    # 未打开nogo项开关,无需更新直接跳过
    if [ $? -ne 0 ]
    then
        return 0
    fi

    # 打开nogo项开关,先关闭nogo项开关
    find /opt/oss/easysuite/easysuite/var/task_config/${task_id}/*/website/data/ -type f -name "upgrade_params_config.json" | xargs sed -i 's/"health_check_set": *"true"/"health_check_set": "false"/g'
}

########################
# 功能: 跳过DFS巡检项
# @param task_id: 任务ID
# @return 0: 成功, 100: 不满足前提条件, 101: 状态更新失败
########################
function skip_dfs_step()
{
    local task_id="${2}"

    # 检查数据库状态,如总任务状态已是成功状态则仅更新nogo项开关
    query_db_status "${task_id}" "upgrade_pre_check_finish"
    if [ $? -eq 0 ]
    then
        modify_upgrade_config "${task_id}" && return 0 || return ${POST_ERROR_CODE}
    fi

    # 检查数据库状态,非检查失败状态不更新
    query_db_status "${task_id}" "upgrade_pre_check_error" || return ${PRE_ERROR_CODE}

    # 更新数据库任务状态
    modify_db_status "${task_id}" && query_db_status "${task_id}" "upgrade_pre_check_finish"
    [ $? -ne 0 ] && return ${POST_ERROR_CODE}

    # 更新产品配置文件中的nogo项开关
    modify_upgrade_config "${task_id}" || return ${POST_ERROR_CODE}
    return 0
}

########################
# 功能: 主入口
# @param task_id: 任务ID
########################
function main()
{
    # 检查脚本执行用户
    check_user || return

    # 校验参数合规性
    check_params "$@"
    if [ "$?" -ne 0 ]
    then
        usage
        log_and_print "ERROR" "${SELF_FILE}:Invalid parameter:$@"
        return
    fi

    # 跳过DFS巡检
    skip_dfs_step "$@"
    result="$?"

    # 按错误码报错
    case ${result} in
    "${PRE_ERROR_CODE}")
        log_and_print "ERROR" "${SELF_FILE}:The task is not in upgrade_pre_check_error status and cannot be skipped.ErrorCode:${result}"
        return
    ;;
    "${POST_ERROR_CODE}")
        log_and_print "ERROR" "${SELF_FILE}:The NCE Health Check is skipped failed.${result}"
        return
    ;;
    esac

    # 执行成功
    log_and_print "INFO" "${SELF_FILE}:The NCE Health Check is skipped successfully."
}


main "$@"