#!/bin/bash

function fn_get_log_file_path()
{
    echo "${current_path}/../../cmd.rollback_local_process.log"
}
function fn_get_all_log_file_path()
{
    echo "${current_path}/../../cmd.rollback_local_process_all.log"
}

function fn_init()
{
    readonly current_path=$(dirname $(readlink -f $0 ))

    # Loading Modules
    source ${current_path}/../utils/load_all_util.sh
    readonly c_root_path=$(cd ${current_path}/../../ && pwd && cd - &>/dev/null)

    # Initialization Log
    fn_sys_set_logging_verbose "$SYS_LOGGING_INFO"
    readonly c_log_file="$(fn_get_log_file_path)"

    # save history log file
    readonly all_log_file="$(fn_get_all_log_file_path)"
    fn_sys_create_log_file "$all_log_file"
    [ -f "${c_log_file}" ]  && cat "${c_log_file}" >> ${all_log_file}

    [[ -f ${c_log_file} ]] && rm -f ${c_log_file}
    fn_sys_create_log_file "$c_log_file"
    fn_sys_set_log_file "$c_log_file"

    source "${c_root_path}/shscript/profile/upgrade_profile.sh"
    rm -rf "${c_root_path}/python" >/dev/null 2>&1;
    return 0
}

function fn_prepare()
{
    fn_sys_log_info "Start to execute preparing."
    fn_clear_step_flag

    fn_kill_useless_ndp_process

    ps -ef | grep /opt/huawei/Bigdata | grep "^omm" >/dev/null 2>&1
    if [[ $? != 0 ]]; then
        fn_sys_log_info "Start to kill process of hadoop-yarn."
        fn_kill_cgroup_procs
    fi

    # 生成两个集合 meta_dir_arr:需要压缩的meta目录集合; skip_dir_arr:既不需要视为meta目录,也不视为数据目录,无需备份.
    final_dirs=""
    OLD_IFS="$IFS"
    IFS=","
    meta_dir_arr=(${need_tar_dir})
    for dir_str in ${meta_dir_arr[@]}
    do
        [[ -d ${dir_str} ]] && final_dirs="${final_dirs} ${dir_str}"
    done

    # 不备份的目录。
    skip_final_dirs=""
    skip_dir_arr=(${skip_dir})
    for dir_str in ${skip_dir_arr[@]}
    do
        [[ -d ${dir_str} ]] && skip_final_dirs="${skip_final_dirs} ${dir_str}"
    done
    IFS="${OLD_IFS}"

    fn_sys_log_info "End to execute preparing."
    return 0
}

function fn_check()
{
    fn_sys_log_info "Start to execute checking."

    fn_sys_assert_current_user_matched "root" || return 1

    fn_sys_log_info "End to execute checking."
    return 0
}

# 0.0 恢复omm用户的定时任务
function fn_precess_crontab()
{
    fn_sys_log_info "Start to rollback crontab for omm..."
    [[ -f ${g_backup_path}/crontab_backup_fi.txt ]] && crontab -u omm ${g_backup_path}/crontab_backup_fi.txt
    rm -f ${g_backup_path}/crontab_backup_fi.txt
    fn_sys_log_info "End to rollback crontab for omm."
    return 0
}

# 1.0 主节点如果namenode的目录是从别的节点拷贝过来的，需要删除 √
function fn_remove_namenode_from_other()
{
    fn_sys_log_info "Start to remove files copy from other node."
    [[ -f ${g_namenode_copy_flag} ]] && rm /srv/BigData/namenode -rf

    return 0;
}

# 2.0 恢复FI备份的srv目录 -- 元数据
function fn_rollback_srv_meta_data()
{
    fn_sys_log_info "Start to rollback Metadata."

    src_tar_file=${g_backup_path}/${g_fi_backup_file%.*}
    if [[ -f ${src_tar_file} ]];
    then

        OLD_IFS="$IFS"
        IFS=","
        meta_dir_arr=(${need_tar_dir})
        for dir_str in ${meta_dir_arr[@]}
        do
            [[ -d ${dir_str} ]] && rm -rf "${dir_str}"
        done
        IFS="${OLD_IFS}"

        # 恢复备份过的元数据
        tar -zxvf ${src_tar_file} -C / > /dev/null 2>&1
        [[ $? != 0 ]] && fn_sys_log_error "Failed to rollback Metadata."
    fi

    fn_sys_log_info "End to rollback Metadata."
    return 0;
}

# 2.1 恢复备份的os文件 -- ldap
function fn_rollback_os_config()
{
    fn_sys_log_info "Start to rollback Metadata."

    [[ -f ${g_backup_path}/report.json ]] && fn_sys_log_info "The backup of report.json is restored." && cp ${g_backup_path}/report.json /tmp/report.json

    [[ ! -f ${g_backup_path}/sssd.conf ]] && fn_sys_log_error "The backup file [sssd.conf] does not exist." && return 1
    cp ${g_backup_path}/sssd.conf /etc/sssd/sssd.conf

    [[ ! -f ${g_backup_path}/ldap.conf ]] && fn_sys_log_error "The backup file [ldap.conf] does not exist." && return 1
    cp ${g_backup_path}/ldap.conf /etc/openldap/ldap.conf
    cp ${g_backup_path}/ldap.conf /etc/ldap.conf

    [[ ! -f ${g_backup_path}/hosts ]] && fn_sys_log_error "The backup file [hosts] does not exist." && return 1
    cp ${g_backup_path}/hosts /etc/hosts

    chown root:root /etc/ldap.conf
    chmod 640 /etc/ldap.conf
    chown omm:wheel /etc/openldap/ldap.conf
    chmod 600 /etc/openldap/ldap.conf
    chown omm:wheel /tmp/report.json
    chmod 600 /tmp/report.json
    chown omm:wheel /etc/openldap/schema/kerberos.schema
    chmod 600 /etc/openldap/schema/kerberos.schema
    chown root:root /etc/hosts
    chmod 644 /etc/hosts
    if [[ -d /sys/fs/cgroup/cpu,cpuacct/hadoop-yarn ]];then
        fn_sys_log_info "Start to change owner of hadoop-yarn."
        chown -R omm:wheel /sys/fs/cgroup/cpu,cpuacct/hadoop-yarn
        [[ $? != 0 ]] && fn_sys_log_error "Can not change owner of hadoop-yarn." && return 1
        chmod 750 /sys/fs/cgroup/cpu,cpuacct/hadoop-yarn
    fi
    sed -i '/^ossuser/d' /etc/cron.allow

    fn_sys_log_info "End to rollback Metadata."
    return 0;
}

# 3.1 清理/srv/Bigdata下面升级后新生成的数据
function fn_clear_new_business_data()
{
    dir_path=$1
    fn_sys_log_info "Start to clear backup file in ${dir_path}."

    local dn_data_dirs=`ls -d ${dir_path}/*`
    [ -z "${dn_data_dirs}" ] && fn_sys_log_info "There is no data dir in ${dir_path},Skip" && return 0
    dn_data_dirs=(${dn_data_dirs})

    # 声明数组，里面是所有的元数据目录。遍历非元数据目录的时候进行跳过即可。
    for dir in "${dn_data_dirs[@]}"; do
        fn_sys_log_info "The subdirectory is: ${dir}."
        if [ "${dir}" == "${dir_path}/backup" ]; then
            fn_sys_log_info "Backup directory found, skip."
            continue
        fi

        if [[ -f ${dir} ]];then
            fn_sys_log_info "It is a file, skip."
            continue
        fi

        dir_name=$(basename ${dir})
        if [ "${dir_name}" == "lost+found" ];then
            fn_sys_log_info "this dir is lost+found, skip."
            continue
        fi
        fn_sys_log_info "Start to clear backup file: ${dir_name}."
        mv ${dir} "${dir_path}/backup/${dir_name}_backup"
        mv "${dir_path}/backup/${dir_name}" "${dir_path}/"
        if [[ $? -ne 0 ]]; then
            fn_sys_log_error "Failed to rollback ${dir_name}."
            return 1
        fi
    done

    fn_sys_log_info "End to clear business data by clear /srv/BigData dirs."
    return 0;
}
rollback_nm_dir()
{
    local nmDir="$1"

    local nmDir_backup="${nmDir}_backup"
    [ ! -d "${nmDir_backup}" ] && fn_sys_log_warn "backup dir is not found: ${nmDir_backup}" && return 0
    fn_sys_log_info "backup dir is existed , mv ${nmDir_backup} ${nmDir}"

    rm -rf "${nmDir}"
    mv "${nmDir_backup}" "${nmDir}"

    fn_sys_log_info "Success to rollback dir: ${nmDir}"
}
# 3.0 删除FI备份的srv目录下的备份数据目录 -- 非元数据
function fn_rollback_srv_business_data()
{
    fn_sys_log_info "Start to delete backup business data in /srv/BigData."

    source ${g_upgrade_param}
    if [[ "X${dn_data_dirs}" == "X" ]]; then
        local dn_data_dirs=`ls -d /srv/BigData/*`
    fi
    [ -z "${dn_data_dirs}" ] && fn_sys_log_info "There is no datanode dir,Skip" && return 0
    dn_data_dirs=(${dn_data_dirs})

    # 声明数组，里面是所有的元数据目录。遍历非元数据目录的时候进行跳过即可。
    for dir in "${dn_data_dirs[@]}"; do

        fn_sys_log_info "Start to rollback ${dir}"
        if [[ -L ${dir} ]];then
            fn_sys_log_info "It is a file which is soft connection file, skip."
            continue
        fi

        if [[ -f ${dir} ]];then
            fn_sys_log_info "It is a file which is not business data, skip."
            continue
        fi

        if [[ "/srv/BigData/hadoop" == "${dir}" ]]; then
            fn_sys_log_info "Found subdirectory hadoop."
            rollback_nm_dir "${dir}/data1/nm/containerlogs"
            rollback_nm_dir "${dir}/data1/nm/localdir"
            if [[ ! -d "${dir}/data1/dn/" ]] ; then
                fn_sys_log_info "No dn subdirectory found in /srv/BigData/hadoop"
                continue
            fi
            if [ ! -f "${dir}/data1/dn/backup/${g_source_version}_SUCCESS_FLAG" ]; then
                fn_sys_log_info "Success flag not found, skip."
                continue
            fi
            fn_clear_new_business_data "${dir}/data1/dn/"
            [[ $? == 0 ]] && rm -rf "${dir}/data1/dn/backup/${g_source_version}_SUCCESS_FLAG"
            continue
        fi

        if [ ! -f "${dir}/backup/${g_source_version}_SUCCESS_FLAG" ]; then
            fn_sys_log_info "Success flag not found, skip."
            continue
        fi

        fn_clear_new_business_data "${dir}"

        [[ $? == 0 ]] && rm -rf "${dir}/backup/${g_source_version}_SUCCESS_FLAG"
    done

    fn_sys_log_info "The duration of this action depends on the disk I/O status."
    fn_sys_log_info "Start to remove the backup directory."
    find /srv/BigData/ -maxdepth 4 -name backup | xargs -i rm -rf {}

    fn_sys_log_info "Start to check whether the backup directory had been removed."
    count_find=$(find /srv/BigData/ -maxdepth 4 -name backup | wc -l)
    [[ ${count_find} != 0 ]] && fn_sys_log_error "Cleanup is not complete." && return 1

    fn_sys_log_info "End to delete backup business data in /srv/BigData."
    return 0
}

# 4.0 rollback /srv/BigData下面的目录控制权限
function fn_process_srv_facl()
{
    fn_sys_log_info "Start to deal with the facl."
    local result_code=0
    local dn_data_dirs=`ls -d /srv/BigData/*`
    [ -z "${dn_data_dirs}" ] && fn_sys_log_info "There is no subdir,Skip" && return 0
    dn_data_dirs=(${dn_data_dirs})
    fn_sys_log_info "Start to restores the facl."
    for dir in "${dn_data_dirs[@]}"; do
        fn_sys_log_info "Start to deal with ${dir}."
        if [[ "${skip_final_dirs[@]}" =~ "${dir}" ]];then
            fn_sys_log_info "It is a skip dir, skip."
            continue
        fi

        if [[ -L ${dir} ]];then
            fn_sys_log_info "It is a file which is soft connection file, skip."
            continue
        fi
        dir_name=$(basename ${dir})
        if [[ "/srv/BigData/hadoop" == "${dir}" ]]; then
            if [[ -d /srv/BigData/hadoop/data1/dn ]] && [[ -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag ]]; then
                find /srv/BigData/hadoop -maxdepth 2 | xargs -i chown omm:wheel {}
                fn_sys_log_info "The duration of this action depends on the disk I/O status."
                fn_sys_log_info "Start to modify the permission on the ${dir}."
                chown -R omm:wheel /srv/BigData/hadoop/data1/dn
                rm -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag
            fi
            chown omm:wheel /srv/BigData/hadoop
            fn_sys_log_info "End to deal with ${dir}."
            continue
        fi
        if [[ "/srv/BigData/hadoop" =~ "${dir}" ]] && [[ -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag ]]; then
            fn_sys_log_info "The duration of this action depends on the disk I/O status."
            fn_sys_log_info "Start to modify the permission on the ${dir}."
            chown -R omm:wheel ${dir}
            rm -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag
            fn_sys_log_info "End to deal with ${dir}."
            continue
        fi

        fn_sys_log_info "Start to test file: ${g_backup_path}/${dir_name}_backup_srv_facl.conf."
        [[ ! -f "${g_backup_path}/${dir_name}_backup_srv_facl.conf" ]] && continue
        [[ ! -f "${g_backup_path}/${dir_name}_backup_srv_facl.flag" ]] && continue

        cd /
        setfacl --restore=${g_backup_path}/${dir_name}_backup_srv_facl.conf
        if [[ $? != 0 ]]; then
            fn_sys_log_error "Failed to change the facl of ${dir}."
            return 1
        fi
        rm -f ${g_backup_path}/${dir_name}_backup_srv_facl.conf
        rm -f ${g_backup_path}/${dir_name}_backup_srv_facl.flag
        fn_sys_log_info "End to deal with ${dir}."
    done
    chmod 770 /srv/BigData
    chmod o+t /srv/BigData
    chown omm:ficommon /srv/BigData
    fn_sys_log_info "End to deal with the facl."
    return 0;
}


# 5.0 删除备份信息。
function fn_process_backup_file()
{
    fn_sys_log_info "Start to remove backup file."
    [[ -f ${g_backup_srv_facl_path} ]] && rm -f ${g_backup_srv_facl_path}
    [[ -f ${g_backup_path}/${g_fi_backup_file%.*} ]] && rm -f ${g_backup_path}/${g_fi_backup_file%.*}
    fn_sys_log_info "End to remove backup file."

    fn_sys_log_info "Start to clear MapReduce file create by NdpService."
    [[ -f /srv/BigData/mapred/ ]] && rm -rf /srv/BigData/mapred/
    fn_sys_log_info "End to clear MapReduce file create by NdpService"

    return 0
}

# 6.0 删除升级目录，删除锁住的升级标志位。
function fn_process_upgrade_file()
{
    fn_sys_log_info "Start to remove upgrade flag."
    [ -f "${g_upgrade_flag}" ] &&  chattr -i ${g_upgrade_flag} && rm -rf ${g_upgrade_flag}
    [ -f "${g_upgrade_replace_flag}" ] &&  chattr -i ${g_upgrade_replace_flag} && rm -rf ${g_upgrade_replace_flag}
    fn_sys_log_info "End to remove upgrade directory."

    # 支持FI已启动情况下的回滚重试。
    ps -ef | grep ldap | grep "^omm"
    if [[ $? != 0 ]]; then
        fn_sys_log_info "Start to remove ldap directory."
        rm -f /srv/BigData/ldapData/ldapserver/data/*
        rm -f /srv/BigData/ldapData/ldapserver/init_flag
        [[ -d /srv/BigData/ldapData/ ]] && mkdir -p /srv/BigData/ldapData/ldapserver/data/
        [[ -d /srv/BigData/ldapData/ ]] && chown -R omm:wheel /srv/BigData/ldapData/ldapserver
        [[ -d /srv/BigData/ldapData/ ]] && chmod -R 700 /srv/BigData/ldapData/ldapserver
        fn_sys_log_info "End to remove ldap directory."
    fi

    su - ossuser -c "rm -f /tmp/redis_aof_flag"
    rm -f ${aof_flag_path}
    rm /opt/backup/ncendp -rf
    rm ${g_upgrade_Dir} -rf
    [ -d "${g_upgrade_Dir}" ] && fn_sys_log_error "Failed to remove upgrade directory." && return 1
    rm /export -rf
    rm ${g_backup_path}/* -rf
    fn_sys_log_info "End to remove upgrade directory."
    return 0
}

###########################################################
#
# 7.0 删除开机启动项
# 先停止服务
# 删除软连接以及源文件
#
############################################################
function fn_remove_load_service()
{
    fn_sys_log_info "Start to remove rc.d .."
    systemctl stop load_dn_socket
    systemctl disable load_dn_socket
    chkconfig --del load_dn_socket
    find /etc -name load_dn_socket | xargs -i rm -f {}

    systemctl stop load_cgroup
    systemctl disable load_cgroup
    chkconfig --del load_cgroup
    find /etc -name load_cgroup | xargs -i rm -f {}
    fn_sys_log_info "End to remove rc.d ."

    rm -f /tmp/flag_changing_param

    count_file=$(find /etc -name load_dn_socket | wc -l)
    [[ $count_file != 0 ]] && fn_sys_log_error "Failed to remove load_dn_socket." && return 1
    count_file=$(find /etc -name load_cgroup | wc -l)
    [[ $count_file != 0 ]] && fn_sys_log_error "Failed to remove load_cgroup." && return 1

    return 0
}

###########################################################
#
# 8.0 启动omm互信。如果没有omm互信，会导致suse环境FI的OMS启动失败
# suse升级场景;
# 不是所有节点都有此脚本;
# 脚本重复执行无问题，无需增加保护。
#
############################################################
function fn_process_ssh_agent()
{
    fn_sys_log_info "Start the omm mutual trust process.."

    local l_suse_version="$(fn_sys_get_suse_version)"
    [[ -z "$l_suse_version" ]] && fn_sys_log_info "In Euler scenarios, skip this step." && return 0

    [[ ! -f ${g_fi_install_path}/om-server/om/sbin/ssh-agent-monitor-startup.sh ]] && fn_sys_log_info "Skip if no SSH script exists on the current node." && return 0
    su - omm -c "nohup bash ${g_fi_install_path}/om-server/om/sbin/ssh-agent-monitor-startup.sh &"

    fn_sys_log_info "The OMM mutual trust process successfully."
    return 0
}

###########################################################
#
# 9.0 增加保护，删除/srv/BigData目录下的ossuser属主文件和目录
#
############################################################
function fn_remove_ossuser_in_srv()
{
    fn_sys_log_info "Start to remove ossuser file in srv ..."
    find /srv/BigData/ -user ossuser | xargs -i rm -rf {}
    fn_sys_log_info "End to remove ossuser file in srv ..."
    return 0
}

function fn_do()
{

    # 0.0 恢复omm用户的定时任务
    fn_precess_crontab || return 1

    # 2.0 恢复FI备份的srv目录 -- 元数据
    fn_rollback_srv_meta_data || return 1

    # 2.1 恢复备份的os文件 -- ldap
    fn_rollback_os_config || return 1

    # 3.0 删除FI备份的srv目录下的备份数据目录 -- 非元数据
    fn_rollback_srv_business_data || return 1

    # 1.0 主节点如果namenode的目录是从别的节点拷贝过来的，需要删除
    fn_remove_namenode_from_other || return 1

    # 4.0 rollback /srv/BigData下面的目录控制权限
    fn_process_srv_facl || return 1

    # 5.0 删除备份信息。
    fn_process_backup_file || return 1

    # 6.0 删除升级目录，删除锁住的升级标志位。
    fn_process_upgrade_file || return 1

    # 7.0 删除开机启动项。
    fn_remove_load_service || return 1

    # 8.0 启动omm互信。
    fn_process_ssh_agent || return 1

    # 9.0 删除/srv/bigdata下面的ossuser文件
    fn_remove_ossuser_in_srv || return 1

    return 0
}

function exit_create_flag()
{
    err_msg=$1
    fn_sys_log_error "${err_msg}"
    touch ${g_process_rollback_flag}
    return 1
}

function fn_main()
{
    c_start_time="$(date +%s)"
    fn_init || exit_create_flag "Start to create the flag file and exit." || return 1
    fn_check || exit_create_flag "Start to create the flag file and exit." || return 1
    fn_prepare || exit_create_flag "Start to create the flag file and exit." || return 1
    fn_do
    local l_result="$?"
    rm -f ${g_namenode_copy_flag}
    touch ${g_process_rollback_flag}
    local l_elapsed_seconds="$(fn_sys_get_elapsed_time "$c_start_time")"
    fn_sys_log_info "Leave post_install with status [${l_result}]. It takes [${l_elapsed_seconds}] seconds"
    [[ ${l_result} -ne 0 ]] && fn_sys_log_error "Failed to run rollback_local script."
    return "$l_result"
}

###############################################################################################
fn_main
result="$?"
exit "$result"
