#!/bin/bash
#
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
#

###################################################################################
# 执行节点： 各FI节点
# 执行依赖： 无
# 参数：
#     无
# 执行返回值：
#     0 ：执行检查成功
#     非0 ：执行检查失败
###################################################################################

function fn_log_screen() {
    local log_level="$1"
    local l_message="$2"
    echo "[$(date -d today +"%Y-%m-%d %H:%M:%S")][${log_level}] ${l_message}"
}

function get_osname() {
    local sles_release="/etc/SuSE-release"
    local sles15_release="/etc/os-release"
    local os_release="/etc/os-release"
    local name=""

    #获取SuSE的版本
    if [[ -f "${sles_release}" ]]; then
        echo "suse"
        return 0
    fi

    #获取SuSE15的版本
    if [[ -f "${sles15_release}" ]]; then
        if [[ "suse" = "$(cat "${sles15_release}" | grep "^ID_LIKE=" | awk -F "\"" '{print $2}')" ]]; then
            echo "suse"
            return 0
        fi
        raw=$(cat "${os_release}")
        eval $raw
        name="$ID"
        echo "$name"
        return 0
    fi

    return 1
}

################################################
#
# 获取系统中缺失的rpm列表
#
################################################
function check_rpms() {
    fn_log_screen "INFO" "start checking rpms."
    local must_rpms=""    #系统中必须有的rpm包列表
    local missing_rpms="" #用于保存缺失rpm列表的文件
    local osname=""       #os名称(suse/redhat)
    local line=""         #文件每行的内容
    local newline=""      #转换后的行的内容(去空格)
    #获取OS名称
    osname="$(get_osname)"
    if [[ $? -ne 0 ]]; then
        fn_log_screen "ERROR" "get_osname(${osname}) failed."
        fn_log_screen "ERROR" "Failed to obtain the rpm package."
    fi

    #获取列表文件 （在脚本位置的下级rpm_lists目录中）
    if [[ $(cat /etc/os-release | grep "^ID=" | awk -F= '{print $2}' | sed 's/"//g') == "sles" ]]; then
        must_rpms="$(dirname "$0")/rpm_lists/${osname}_must-rpms-x86.lst"
        missing_rpms="$(dirname "$0")/rpm_lists/${osname}__missing-rpms-x86.lst"
    else
        fn_log_screen "INFO" "This OS is not SUSE.Skip"
		return 0
    fi

    #文件检查
    if [[ ! -f "${must_rpms}" ]]; then
        fn_log_screen "ERROR" "${must_rpms} is not a file in me."
        fn_log_screen "ERROR" "Failed to obtain the rpm package."
        return 1
    fi

    #直接写入文件,如果写入失败,直接返回
    echo "# Missing rpms list of host " >"${missing_rpms}"
    if [[ $? -ne 0 ]]; then
        fn_log_screen "ERROR" "${missing_rpms} is invalid, write failed."
        fn_log_screen "ERROR" "Failed to obtain the rpm package."
        return 1
    fi

    echo "# This list was generated on $(date)" >>"${missing_rpms}"
    echo "# os: ${osname}" >>"${missing_rpms}"

    while read line; do
        #排除非字母或数字开头的行
        if [[ -z "$(echo "$line" | grep "^[[:alnum:]]")" ]]; then
            continue
        fi

        #删除行中空格
        newline="$(echo "$line" | tr -d '[:space:]')"
		newline_prefix="${newline%.*}" #获取前缀
		newline_suffix="${newline##*.}" #获取后缀
		#处理前缀名称符号，带符号自动添加转义
		newline_name=$(echo "$newline_prefix" | sed 's/\+/\\+/g')
		#判断系统中是否包含需要的rpm包
		version=$(rpm -qa|egrep ^${newline_name}-[0-9]+\.[0-9]+\.[0-9]+.*\.${newline_suffix}$)
        if [[ $? -ne 0 ]]; then
            echo "${newline}" >>"${missing_rpms}"
            fn_log_screen "ERROR" "need to install : ${newline}"
        else
            echo "INFO" "[RPM_VER]#${newline}#${version}"
        fi

    done <"${must_rpms}"

    echo "" >>"${missing_rpms}"
    #读取$missing_rpms中的内容，并输出缺失的rpm包
    local missing_content=$(cat $missing_rpms | grep -v "^#" | sed '/^\s*$/d') #从@missing_rpms文件中读取缺失的rpm包
    if [[ -z $missing_content ]]; then
        fn_log_screen "INFO" "No RPM package is missing."
        return 0
    else
        fn_log_screen "ERROR" "The following RPM packages are missing.Please install them: $missing_content"
        return 1
    fi
}

check_rpms
exit $?
