#!/bin/bash

#
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
#
###################################################################################
# 执行节点： FI管理节点
# 目前只针对装Hbase产品适用
# 执行依赖： 依赖使用omm用户和安装Hbase
# 参数：
#     无
# 执行返回值：
#     0 ：执行检查成功
#     非0 ：执行检查失败

###################################################################################

function fn_log_screen() {
    local log_level="$1"
    local l_message="$2"
    echo "[$(date -d today +"%Y-%m-%d %H:%M:%S")][${log_level}] ${l_message}"
}


function check_all_fi_hosts()
{
    fn_log_screen "INFO" "Start checking hostname in shell."
    # 判断当前用户是否是OMM，如果不是omm打印日志，终止后面执行
    local current_user=$(whoami)
    [[ "${current_user}" != "omm" ]] && fn_log_screen "Warn" "The current user is not an omm user." && return 1

    local fi_hbase_xml_path=$(realpath /opt/huawei/Bigdata/FusionInsight_Current/*RegionServer/etc/hbase-site.xml)
    # 判断是否安装hbase，如果没有安装直接跳过
    [[ -z "${fi_hbase_xml_path}" ]] && fn_log_screen "ERROR" "HBase is not installed." && return 1
    # 根据regionserver路径获取FI节点IP通信信息
    local fi_ips_str=$(cat "${fi_hbase_xml_path}" | grep "hbase.regionserver.list" -A1 | grep -oP '(?<=<value>).*?(?=<\/value>)')
    # 将ip字符串根据逗号转换成列表
    local fi_ips=$(echo "${fi_ips_str}" | tr ',' ' ')
    # 判断结果
    [[ -z "${fi_ips}" ]] && fn_log_screen "ERROR" "Check whether 'hbase.regionserver.list' in the hbase-site.xml file is configured." && return 1
    local result=0

    if [ $(echo "${fi_ips}" | awk '{print NF}') -eq 1 ]; then
       #检查/etc/hosts中是否配置127.0.0.1 localhost，单机场景要求一定要有
       local checkResult=$(cat /etc/hosts | grep -v '^#' | awk '{if ($1 == "127.0.0.1" && $2 == "localhost"){print $0}}')
       [[ -z "${checkResult}" ]] && fn_log_screen "ERROR" "'127.0.0.1 localhost' is not configured in '/etc/hosts'." && result=1

       check_etc_hosts "${fi_ips}" "$(hostname)" "$(cat /etc/hosts)" || result=1
       check_hostname "${fi_ips}" "$(hostname)" || result=1
       return ${result}
    fi

    for fi_ip in ${fi_ips} ; do
        # 所有FI节点的IP 主机名 ip
        local mapping_info=$(ssh -o StrictHostKeyChecking=no omm@${fi_ip} 'hostname && hostname -i')
        # 如果命令执行报错，需要提示检查网络互通行
        [[ $? -ne 0 ]] && result=1 && fn_log_screen "ERROR" "Check whether the communication between the node IP address ${fi_ip} is normal." && return ${result}

        # 判断是否存一个主机名映射多个IP
        local mapping_ip_count=$(echo "${mapping_info}" |  grep -oE "\b([0-9]{1,3}\.){3}[0-9]{1,3}\b" | wc -l)
        [[ "${mapping_ip_count}" -ne 1 ]] && fn_log_screen "ERROR" "${fi_ip}: the hostname corresponds to more than one ip in '/etc/hosts'." && result=1

        local mapping_hostname=$(echo "${mapping_info}" | head -1)
        # 需要判断主机名非空
        [[ -z "${mapping_hostname}" ]] && result=1 && fn_log_screen "ERROR" "the hostname of the current node ${fi_ip} is not configured."
        # 遍历所有FI节点的/etc/hosts
        for remote_ip in ${fi_ips} ; do
            check_etc_hosts_mapping "${fi_ip}" "${remote_ip}" "${mapping_hostname}"
            [[ $? -ne 0 ]] && result=1
        done
    done
    return ${result}
}

function check_etc_hosts()
{
    local host_ip=$1
    local hostname_str=$2
    local etc_host_info=$3
    # 校验主机名和ip是否存在映射，主机名和IP全部匹配
    local checkhostip=$(echo "${etc_host_info}" | grep -v '^#' | awk '{if ($1 == "'${host_ip}'" && $2 == "'${hostname_str}'"){print $0}}')
    [[ -z "${checkhostip}" ]] && fn_log_screen "ERROR" "The IP address and hostname in the '/etc/hosts' do not match: ${host_ip} ${hostname_str}, hosts: ${etc_host_info}" && return 1
    return 0
}

function check_hostname()
{
    local host_ip=$1
    local node_hostname=$2
    local etc_host_info=$3
    # 检查不能只有数字的主机名，平台不支持，保留校验
    local formatCheck=$(echo ${node_hostname} | egrep ^"[[:digit:]]+$")
    if [[ -n "${formatCheck}" ]]; then
        fn_log_screen "ERROR" "${host_ip}: The hostname cannot contain only digits."
        return 1
    fi

    # 检查hostname格式
    formatCheck=$(echo ${node_hostname} | egrep "^[a-zA-Z]([-a-zA-Z0-9]*[a-zA-Z0-9])*$")
    if [[ -z "${formatCheck}" ]]
    then
        fn_log_screen "ERROR" "${host_ip}: hostname '${node_hostname}' format error, correct format is '^[a-zA-Z]([-a-zA-Z0-9]*[a-zA-Z0-9])*\$'"
        return 1
    fi
    fn_log_screen "INFO" "${host_ip}: Success checking hostname in shell."
    return 0
}

function check_etc_hosts_mapping()
{
    local fi_ip=$1
    local remote_ip=$2
    local mapping_hostname=$3
    # 获取FI远程节点的/etc/hosts信息
    local hosts_info=$(ssh -o StrictHostKeyChecking=no omm@${remote_ip} 'cat /etc/hosts')
    # 需要校验其他节点是否跟当前节点主机名重复，忽略大小写
    if [ "${fi_ip}" != "${remote_ip}" ]
    then
       local remote_hostname=$(ssh -o StrictHostKeyChecking=no omm@${remote_ip} 'echo $(hostname)')
       if [ "$(echo ${remote_hostname}| tr A-Z a-z)" = "$(echo ${mapping_hostname} | tr A-Z a-z)" ]
       then
           fn_log_screen "ERROR" "The hostname of this IP:${fi_ip} address is the same as the hostname of the remote IP:${remote_ip} address."
           return 1
       fi
    else
        check_hostname "${fi_ip}" "${mapping_hostname}" "${hosts_info}"
        [[ $? -ne 0 ]] && return 1
    fi
    check_etc_hosts "${fi_ip}" "${mapping_hostname}" "${hosts_info}"
    [[ $? -ne 0 ]] && fn_log_screen "ERROR" "Please login to node ${remote_ip} and check the mapping of the /etc/hosts configuration file." && return 1
    return 0
}

check_all_fi_hosts

exit $?