#!/bin/bash

USER=`whoami`
OPERIP=`who am i | awk '{print $NF}' | sed 's/[()]//g'`

[ -z "${WORK_USER}" ] && WORK_USER='ossadm'

if [ "${USER}" != "${WORK_USER}" ]
then
    echo "Only ${WORK_USER} can run this command."
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Only ${WORK_USER} can run this command."
    exit 1
fi

#Note:set global parameter
LOGPATH="${ABSOLUATELY_PATH}/logs"
GLOBAL_BASH_PID=$$
CONSTANT_LOGFILE="${LOGPATH}/${MODULE_NAME}.log"
TMP_CONFIG_PATH="${LOGPATH}/tmp"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmpcontent_${MODULE_NAME}_${RANDOM}.log"
mkdir -p ${LOGPATH} &> /dev/null
mkdir -p ${TMP_CONFIG_PATH} &> /dev/null
chmod 750 "${LOG_PATH}" >/dev/null 2>&1

function FUNC_CLEAN_TMP_FILE()
{
    [ -f ${TMP_CONTENT_FILE} ] && rm -f ${TMP_CONTENT_FILE} &> /dev/null
    [ -d ${TMP_CONFIG_PATH} ] && rm -rf ${TMP_CONFIG_PATH}/* &> /dev/null
    return 0
}

# print text with color -- green
# parameters:   $1 -- text
function fn_echo_succ()
{
    FG=32 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# print text with color -- yellow
# parameters:   $1 -- text
function fn_echo_warn()
{
    FG=33 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# print text with color -- red
# parameters:   $1 -- text
function fn_echo_fail()
{
    FG=31 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# Name        : PRINT_FILE_TO_LOG
# Description : print file content into log file
# Parameters  : $1 the file to be printed
function PRINT_FILE_TO_LOG()
{
    local file_name=$1

    LOGFILE="${LOGFILE:-${CONSTANT_LOGFILE}}"

    if [ -f "${file_name}" ]
    then
        local check_content=`cat ${file_name}`
        if [ -z "${check_content}" ]
        then
            LOG "This file <${file_name}> is empty."
            return 1
        else
            LOG "Print <${file_name}>"
            LOG "`cat \"${file_name}\" | sed 's/\r//g'`"
        fi
    else
        LOG "This file <${file_name}> does not exist."
        return 1
    fi
}

#*************************************************************#
# Name:        WARN                                           #
# Description: record the warning message into the logfile    #
#*************************************************************#
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************#
# Name:        ERROR                                          #
# Description: record the error message into the logfile      #
#*************************************************************#
function ERROR()
{
    LOG "ERROR: $*"
}

# Name        : PRINT_LOG
# Description : print log
function PRINT_LOG()
{
    local LOG_LEVEL=$1
    local LOG_INFO=$2

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "[Line-${BASH_LINENO}] ${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}

#*************************************************************#
# Name:        getSysTime                                     #
# Description: get the system time                            #
#*************************************************************#
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}

#Note: get single log file size: unit M
function fn_get_shell_log_size()
{
    local log_size=""
    log_size=$(cat /usr/local/osconfig/os/conf/common.cfg 2>/dev/null | grep "MAX_SHELL_LOG_SIZE=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_size}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_size}"
    fi
}

function fn_get_shell_log_num()
{
    local log_num=""
    log_num=$(cat /usr/local/osconfig/os/conf/common.cfg 2>/dev/null | grep "MAX_SHELL_LOG_NUM=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_num}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_num}"
    fi
}

#*************************************************************#
# Name:         LOG                                           #
# Description: record the message into the logfile            #
#*************************************************************#
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSizeM=`fn_get_shell_log_size`
    [ -z "${maxSizeM}" ] && maxSizeM="2"
    local maxSize=$((${maxSizeM} * 1024 * 1024))
    local maxRecord=`fn_get_shell_log_num`
    [ -z "${maxRecord}" ] && maxRecord="5"
    local index=`expr $maxRecord - 1`
    local content="$*"

    if [  -f ${CONSTANT_LOGFILE} ]
    then
        curSize=`ls -lk ${CONSTANT_LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${CONSTANT_LOGFILE}.${index}.gz ]
                then
                    echo y | mv ${CONSTANT_LOGFILE}.${index}.gz ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                    chmod 440 ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                fi
                let index--
            done
            echo y | mv ${CONSTANT_LOGFILE} ${CONSTANT_LOGFILE}.1
            gzip ${CONSTANT_LOGFILE}.1
            chmod 440 ${CONSTANT_LOGFILE}.1.gz
        fi
        if echo "${content}" | grep -w "^WARN:" &> /dev/null
        then
            fn_echo_warn "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        elif echo "${content}" | grep -w "^ERROR:" &> /dev/null
        then
            fn_echo_fail "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        else
            echo -e "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        fi
    else
        tmp_log_path=$(dirname ${CONSTANT_LOGFILE})
        if [ ! -d "${tmp_log_path}" ]
        then
            mkdir -p "${tmp_log_path}"
            chmod 750 "${tmp_log_path}"
        fi
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${CONSTANT_LOGFILE}
    fi
}


function fn_check_pid()
{
    local file_list="${1}"
    local times=0

    while [ "${times}" -lt 7 ]
    do
        ps -ewwf | grep -v grep | egrep -w "${file_list}" || break
        sleep 5
        let times++
    done

    return 0
}


function fn_disabl_features()
{
    local tmp_work_home="$1"
    [ -d "${tmp_work_home}/os/bin" ] || return 0
    local tmp_get_ntp_script="${tmp_work_home}/os/bin/getntpinfo.sh"
    local tmp_check_ntp_script="${tmp_work_home}/os/bin/checkntpstatus.sh"

    chmod 750 "${tmp_work_home}/os/bin"

    local tmp_file_name
    for tmp_file_name in ${tmp_get_ntp_script} ${tmp_check_ntp_script}
    do
        [ -f "${tmp_file_name}" ] || continue
        chmod 640 "${tmp_file_name}"
        mv "${tmp_file_name}" "${tmp_file_name}-bak"
    done

    fn_check_pid "getntpinfo.sh|checkntpstatus.sh"
    return 0
}

function fn_check_env()
{
    local ret=0

    local tmp_msg=$(df --output=pcent,target | awk '$1=="100%" && ($2=="/" || $2=="/tmp" || $2=="/usr") {print $0}')
    if [ ! -z "${tmp_msg}" ]
    then
        PRINT_LOG "FATAL" "content=${tmp_msg},The disk space is full."
        [[ ${ERROR_CODE} =~ "UpgradeOsconfig000001 " ]] || ERROR_CODE="${ERROR_CODE} UpgradeOsconfig000001 "
        ret=11
    fi

    local tmp_msg=$(stat -c "%U:%G %n" / /root /usr/ /usr/local/ /tmp | grep -vw "root:root")
    if [ ! -z "${tmp_msg}" ]
    then
        PRINT_LOG "FATAL" "content=${tmp_msg}, The directory owner group is incorrect."
        [[ ${ERROR_CODE} =~ "UpgradeOsconfig000002 " ]] || ERROR_CODE="${ERROR_CODE} UpgradeOsconfig000002 "
        ret=12
    fi

    local tmp_msg=$(stat -c "%A %n" / /root /usr/ /usr/local/ | cut -c9- | grep '^w')
    if [ ! -z "${tmp_msg}" ]
    then
        PRINT_LOG "FATAL" "content=${tmp_msg},The directory permission is incorrect."
        [[ ${ERROR_CODE} =~ "UpgradeOsconfig000003 " ]] || ERROR_CODE="${ERROR_CODE} UpgradeOsconfig000003 "
        ret=14
    fi

    return "${ret}"
}

function fn_get_pyscript()
{
    local py_name="$1"
    os_type=$(bash /usr/local/osconfig/os/bin/getostype.sh)
    for path in \
    "/usr/local/osconfig/os/pyscripts/action/${py_name}.pyc" \
    "/usr/local/osconfig/os/pyscripts/action/${py_name}.py" \
    "/usr/local/osconfig/os/${os_type}/pyscripts/action/${py_name}.pyc" \
    "/usr/local/osconfig/os/${os_type}/pyscripts/action/${py_name}.py"
    do
        [[ -f "${path}" ]] && { python_file=${path}; return 0; }
    done
    PRINT_LOG "WARN" "no python script find, current osconfig version is: $(cat /usr/local/osconfig/os/bin/release_version.properties | grep release_version)"
    return 1
}


function register_system_service()
{
    PRINT_LOG "INFO" "Register osconfig system service..."
    . /usr/local/osconfig/os/lbin/env_profile.sh
    python_file=""
    fn_get_pyscript "config_osconfig_service" || { PRINT_LOG "INFO" "script not found, no need to execute"; return 0; }
    PRINT_LOG "INFO" "Invoke ${python_file}..."
    ${PYTHONHOME}/bin/python ${python_file} register
    if [[ $? -ne 0 ]]
    then
        PRINT_LOG "FATAL" "register osconfig system service failure"
        return 1
    fi
    
    PRINT_LOG "INFO" "Register osconfig system service done"
    return 0
}

# config cron to pretect osconfig service
function add_cron()
{
    PRINT_LOG "INFO" "config osconfig cron ..."
    . /usr/local/osconfig/os/lbin/env_profile.sh
    python_file=""
    fn_get_pyscript "cron_action" || { PRINT_LOG "INFO" "script not found, no need to execute"; return 0; }

    PRINT_LOG "INFO" "Invoke ${python_file}..."
    ${PYTHONHOME}/bin/python ${python_file} -action add
    if [[ $? -ne 0 ]]
    then
        PRINT_LOG "FATAL" "config osconfig cron  failure"
        return 1
    fi
    
    PRINT_LOG "INFO" "config osconfig cron done"
    return 0
}
