#!/usr/bin/env bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.
# File name     : rollback.sh
# Description   : 欧方回滚脚本
###########################################################

CURRENT_DIR=$(cd $(dirname $0);pwd)
LOG_FILE="${CURRENT_DIR}"/rollback.log
VERSION=$1
OSM_PATH=$2
BUILD_FLAG=$3
WIRELESS_FLAG=$4

. "${CURRENT_DIR}/get_os_info.sh"
. "${CURRENT_DIR}/upgrade_rollback_func.sh"

############################################################
# Name:        rollback_sudo
# Description: 回滚sudoconfig
#
############################################################
function rollback_sudo() {
    print_log "INFO" "start to rollback sudoconfig."
    # 判断是否需要回滚sudoconfig
    if [ ! -f "${CURRENT_DIR}/backup_sudoconfig.flag" ];then
        print_log "WARN" "no need to rollback sudoconfig"
        return
    fi
    bash ${SUDOCONFIG_PATH}/rebuild/sudoconfig_rollback.sh -oldsudobin "${SUDOBIN_PATH}" -newsudobin "${SUDOBIN_PATH}" -version "${VERSION}"
    [ $? -ne 0 ] && print_log "ERROR" "rollback sudoconfig failed, please read log in ${CURRENT_DIR}/../sudoconfig/rebuild/logs/" && return 1
    rm -f "${CURRENT_DIR}/backup_sudoconfig.flag"
    print_log "INFO" "sucess to rollback sudoconfig."
    return 0
}


############################################################
# Name:        rollback_osconfig
# Description: 回滚osconfig
#
############################################################
function rollback_osconfig() {
    print_log "INFO" "start to rollback osconfig."
    # 判断是否需要回滚osconfig
    if [ ! -f "${CURRENT_DIR}/backup_osconfig.flag" ];then
        print_log "WARN" "no need to rollback osconfig."
        return
    fi
    bash "${OSCONFIG_PATH}"/osconfig_rollbackup.sh -backuppath ${OSM_PATH}/../bak_${VERSION} -oldinstallpath /usr/local/osconfig -newinstallpath /usr/local/osconfig -oldsudobin "${SUDOBIN_PATH}" -version ${VERSION}
    [ $? -ne 0 ] && print_log "ERROR" "rollback osconfig fail, please read log in ${CURRENT_DIR}/../osconfig/logs/" && return 1
    rm -f "${CURRENT_DIR}/backup_osconfig.flag"
    print_log "INFO" "success to rollback osconfig."
    return 0
}


############################################################
# Name:        rollback_common_osconfig
# Description: 回滚common_osconfig
#
############################################################
function rollback_common_osconfig() {
    print_log "INFO" "start to rollback common osconfig."
    # 判断是否需要回滚操作系统配置
    if [ ! -f "${CURRENT_DIR}/nce_config.flag" ];then
        print_log "WARN" "no need to rollback common osconfig."
        return
    fi
    bash /usr/local/osconfig/os/bin/commonosconfig.sh rollback "${VERSION}" "${BUILD_FLAG}" "${WIRELESS_FLAG}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade common osconfig fail." && return 1
    rm -f "${CURRENT_DIR}/nce_config.flag"
    print_log "INFO" "success to rollback common osconfig."
    return 0
}

############################################################
# Name:        pre_rollback
# Description: 回滚前置操作
#
############################################################
function pre_rollback() {
    print_log "INFO" "start to execute operation before rollback."
    # 恢复升级删除的配置
    sudo "${SUDOBIN_PATH}"/execute.sh "${CURRENT_DIR}/sudobin-upgrade_pkg.tar" "/opt/rollback_config_$(date '+%Y%m%d%H%M%S')" pre_rollback.sh
    print_log "INFO" "finish to execute operation before rollback."
    return 0
}

############################################################
# Name:        post_rollback
# Description: 回滚后置操作
#
############################################################
function post_rollback() {
    print_log "INFO" "start to execute operation after rollback."
    # 因平台回滚导致的签名工具acl权限丢失，要在回滚后重新刷新，避免回滚后验签失败
    sudo "${SUDOBIN_PATH}"/execute.sh "${CURRENT_DIR}/sudobin-upgrade_pkg.tar" "/opt/rollback_config_$(date '+%Y%m%d%H%M%S')" post_rollback.sh
    print_log "INFO" "finish to execute operation after rollback."
    return 0
}

############################################################
# Name:        main
# Description: 主函数，入口
#
############################################################
function main() {
    # 修改文件权限
    modify_file_perm
    # 解压osconfig和sudoconfig包
    unzip_cloudsop_pack || return 1

    # 回滚OS配置
    pre_rollback || return 1
    rollback_common_osconfig || return 1

    # 回滚osconfig
    rollback_osconfig || return 1

    # 回滚sudoconfig
    rollback_sudo || return 1

    # 回滚后置操作
    post_rollback || return 1

    # 清理升级路径的文件, 删除平台osconfig的备份目录
    clean_after_rollback

    print_log "INFO" "success to rollback."
    return 0
}

print_log "INFO" "start to rollback, version: ${VERSION}" >> "${LOG_FILE}" 2>&1

main >> "${LOG_FILE}" 2>&1
result=$?
if [ "${result}" -eq 0 ];then
    touch "${CURRENT_DIR}"/rollback_succeed_flag
else
    touch "${CURRENT_DIR}"/rollback_fail_flag
fi
exit "${result}"