#!/bin/bash

function execute_sudo_pkg()
{
    init_env "$@" || return $?
    verify_input_params "$@" || return $?
    decompress_inner_pkg || return $?
    execute_script || return $?
}

function init_env()
{
    ABSOLUTELY_PATH=$(cd $(dirname "$0"); pwd)
    source "${ABSOLUTELY_PATH}"/bin/osprofile &> /dev/null
    export LC_ALL=en_US.UTF-8
    WORK_SPACE=/opt/$(cat /proc/sys/kernel/random/uuid)
    INPUT_PKG_FILE="$1"
    WORK_PKG_FILE=${WORK_SPACE}/$(basename "${INPUT_PKG_FILE}")
    INNER_WORK_SPACE=/opt/$(cat /proc/sys/kernel/random/uuid)
    ENTER_SCRIPT="$2"
    PARAMETERS="${*:3}"
    #Note: if second parameter is path, need use third parameter as ENTER_SCRIPT
    [[ "$ENTER_SCRIPT" =~ "/" ]] && ENTER_SCRIPT="$3" && PARAMETERS="${*:4}"

    umask 0077
}

function verify_input_params()
{
    check_param_special_char "$@" || return $?
    verify_pkg_file || return $?
}

function check_param_special_char()
{
    [ $# -lt 2 ] && {
        echo "Error: the parameters number error, this script need at lease two parameters"
        return 1
    }
    for param in "$@"
    do
        no_blank_param=$(echo "${param}" | sed ':a;$!N;s/[\*\ \n\t\r\`]//g;ta')
        [ "${param}" == "${no_blank_param}" ] || {
            echo "Error: The param <${param}> can not contain <\*|\ |\n|\t|\r|\`>."
            return 5
        }
    done
    [[ "${ENTER_SCRIPT}" =~ ^[a-zA-Z0-9_.-]+$ ]] || return $?
}

function verify_pkg_file() {
    #Note: before cp, verify package size first
    check_pkg_size "${INPUT_PKG_FILE}" || return $?
    #Note: verify pkg under the workspace, so need copy file to workspace
    copy_input_pkg || return $?
    check_link_path || return $?
    check_pkg_size "${WORK_PKG_FILE}" || return $?
    #Note: check pkg content need get sign file name first
    INNER_PKG_NAME=$(tar tf "${WORK_PKG_FILE}" | egrep "\.tar$|\.tar\.gz$")
    check_pkg_content || return $?
    cd ${WORK_SPACE}
    #Note: verify sign need extract tar first
    verify_sign_pkg || return $?
}

function copy_input_pkg()
{
    [  -f "${INPUT_PKG_FILE}" ] || {
      echo "Error: the file not found ${INPUT_PKG_FILE}"
      return 1
    }
    mkdir -m 700 "${WORK_SPACE}" || return $?
    cp "${INPUT_PKG_FILE}" "${WORK_PKG_FILE}" || return $?
}

function check_link_path()
{
   [ "$(readlink -m "${WORK_PKG_FILE}")" == "${WORK_PKG_FILE}" ] || {
        echo "Error: the file<${WORK_PKG_FILE}> is relative path"
        return 1
     }
}

function check_pkg_size()
{
    local file="${1}"
    local sudo_pkg_file_size=$(du -sm "${file}" | awk '{print $1}')
    [ "${sudo_pkg_file_size}" -ge 10240 ] && return 6
    [ "$(df -Pm "/opt" | grep -v "^Filesystem" | awk '{print $4}')" -ge "${sudo_pkg_file_size}" ] || return 7
}

function check_pkg_content()
{
    [ "$(tar tf "${WORK_PKG_FILE}" | grep -Ec "\.tar$|\.tar\.gz$")" -ne 1 ] && return 8
    tar -tf "${WORK_PKG_FILE}" 2> /dev/null | grep -Eq "(osprofile|\.sh|\.inc)$" && return 9
    tar -tvf "${WORK_PKG_FILE}" 2> /dev/null | awk '{print $1}' | egrep -iq "s|t" && return 10
    tar -tvf "${WORK_PKG_FILE}" 2> /dev/null | egrep -q "^l" && {
        echo "Error: the ${WORK_PKG_FILE} contain soft link"
        return 12
    }
    tar -tf "${WORK_PKG_FILE}" | egrep -v "^(\.\/)*${INNER_PKG_NAME}$|^(\.\/)*${INNER_PKG_NAME}.cms$|^(\.\/)*${INNER_PKG_NAME}.crl$|^(\.\/)*crldata.crl$|^(\.\/)*${INNER_PKG_NAME}.asc$" && { echo "Error: invalid package."; return 14; }

    return 0
}

function fn_repair_missing_files()
{
    local ca_crl_file="/etc/ssl/certs/HuaweiRootCAsipCRLs.crl"
    local ca_release_crl_file="/etc/ssl/certs/HuaweiRootCAsipCRLs_Release.crl"
    local src_ca_crl_file="/usr/local/signtool/HuaweiRootCAsipCRLs.crl"
    local src_ca_release_crl_file="/usr/local/signtool/HuaweiRootCAsipCRLs_Release.crl"
    [ -f "${src_ca_crl_file}" -a ! -f "${ca_crl_file}" ] && { cp -a "${src_ca_crl_file}" "${ca_crl_file}"; }
    [ -f "${src_ca_release_crl_file}" -a ! -f "${ca_release_crl_file}" ] && { cp -a "${src_ca_release_crl_file}" "${ca_release_crl_file}"; }
}

function verify_sign_pkg()
{
    tar --no-same-owner -xf "${WORK_PKG_FILE}" -C "${WORK_SPACE}" || return $?
    fn_repair_missing_files
    verify_cms && return 0
    verify_gpg || return $?
}

function adapts_to_eulerOS_r11()
{
    local os_type=$(arch |grep "aarch")
    local os_version=$(uname -a |grep "eulerosv2r11")
    if [ -n "${os_version}" -a -n "${os_type}" ]; then
        if [ ! -f "/usr/lib64/ld-linux-aarch64.so.1" ]; then
            echo "Adapts to eulerOS R11"
            ln -s /usr/lib/ld-linux-aarch64.so.1 /usr/lib64/ld-linux-aarch64.so.1 &>/dev/null
        fi
    fi
}

function verify_cms()
{
    adapts_to_eulerOS_r11
    local file="${WORK_SPACE}/${INNER_PKG_NAME}"
    local cms_file="${file}.cms"
    local crl_file="${file}.crl"
    [ -f "${crl_file}" ] || crl_file="${WORK_SPACE}/crldata.crl"
    local ca_crl_file="/etc/ssl/certs/HuaweiRootCAsipCRLs.crl"
    local crl_file_option="--crl ${crl_file}"
    [ -f "${crl_file}" ] || crl_file_option=""
    hwcmstool --verify --source "${file}" --sign "${cms_file}" ${crl_file_option} --cacrl "${ca_crl_file}"
    if [ $? -ne 0 ]
    then
        echo "Error: Verify package ${file} failed with CMS"
        return 15
    fi
}

function verify_gpg()
{
    local gpg_file="${WORK_SPACE}/${INNER_PKG_NAME}.asc"
    
    [ -f "${gpg_file}" ] || { echo "Error: gpg sign file not exist"; return 18; }
    echo "Using the openPGP checks the file"
    check_gpg_key || return $?
    gpg --verify "${gpg_file}" 2>&1 | grep -w "Good signature" &>/dev/null
    local ret=("${PIPESTATUS[@]}")
    if [ "${ret[0]}" != 0 -o "${ret[1]}" != 0 ]
    then
        echo "Error: failed to verify ${gpg_file}"
        return 17
    fi
}

function check_gpg_key()
{
    gpg --list-keys | grep -w "OpenPGP signature" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo "GPG Error: The public key does not exist when OpenGPG verified package."
        return 16
    fi
    echo "The public key exists"
}

function decompress_inner_pkg()
{
    pre_extract_inner_pkg || return $?
    tar --no-same-owner -xf "${WORK_SPACE}/${INNER_PKG_NAME}" -C "${INNER_WORK_SPACE}" || return $?
}

function pre_extract_inner_pkg()
{
    tar -tf "${WORK_SPACE}/${INNER_PKG_NAME}" | egrep "^${ENTER_SCRIPT}$|^\./${ENTER_SCRIPT}$"
    if [ $? -ne 0 ]
    then
        echo "Error: The ${ENTER_SCRIPT} is not in ${1}."
        return 11
    fi
    #Note: create floder for extract package
    mkdir -m 700 "${INNER_WORK_SPACE}" || {
        echo "mkdir ${INNER_WORK_SPACE} fail"
        return 13
    }
}

function execute_script()
{
    cd "${INNER_WORK_SPACE}"
    bash "${INNER_WORK_SPACE}/${ENTER_SCRIPT}" ${PARAMETERS} || {
        result=$?
        echo "execute script: ${ENTER_SCRIPT} result: $result"
        return $result
    }
}

function clean_workspace()
{
    [ -d "${WORK_SPACE}" ] && rm -rf "${WORK_SPACE}"
    [ -d "${INNER_WORK_SPACE}" ] && rm -rf "${INNER_WORK_SPACE}"
}

function write_operation_Log()
{
    local user=$(id -un)
    local script=$(basename "$0")
    local terminal=$(who am i | awk '{print $NF}' | sed -e 's/[()]//g' -e 's/:.*//g')
    if [ "${1}" -eq 0 ]
    then
        logger -t "${user}" -p local0.info "${script};Successful;${terminal:-"127.0.0.1"};run ${script} result: ${result}"
    else
        logger -t "${user}" -p local0.err "${script};Failed;${terminal:-"127.0.0.1"};run ${script} result: ${result}"
    fi
}

execute_sudo_pkg "$@"
result=$?
clean_workspace
write_operation_Log $result
exit "${result}"
