#!/bin/bash
# Name             : sudoconfig_upgrade.sh
# Description      : rollback sudoconfig when upgrade UniEP
# Platform         : Linux
# Modification log :
umask 0027

FileName="`basename $0`"
RELATIVE_PATH="$(dirname $0)"
cd "${RELATIVE_PATH}"
ABSOLUATELY_PATH=`pwd`
. ./common.inc
cd - &> /dev/null

MODULE_NAME="`basename $0 | sed 's/\.sh$//'`"

USER=`whoami`
OPERIP=`who am i | awk '{print $NF}' | sed 's/[()]//g'`
if [ "${USER}" != "${WORK_USER}" ]
then
    echo "Only ${WORK_USER} can run this command."
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Only ${WORK_USER} can run this command."
    exit 1
fi

#Note:set global parameter
LOGPATH="${ABSOLUATELY_PATH}/logs"
GLOBAL_BASH_PID=$$
CONSTANT_LOGFILE="${LOGPATH}/${MODULE_NAME}.log"
TMP_CONFIG_PATH="${LOGPATH}/tmp"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmpcontent_${MODULE_NAME}_${RANDOM}.log"
mkdir -p ${LOGPATH} &> /dev/null
mkdir -p ${TMP_CONFIG_PATH} &> /dev/null
chmod 700 "${LOG_PATH}" >/dev/null 2>&1

IS_SUDO_INSTALL="false"
IS_SUCCESS="false"
oldsudobin_path=""

[ -d "${ABSOLUATELY_PATH}/script" ] && rm -rf "${ABSOLUATELY_PATH}/script" &>/dev/null

function FUNC_CLEAN_TMP_FILE()
{
    [ -f ${TMP_CONTENT_FILE} ] && rm -f ${TMP_CONTENT_FILE} &> /dev/null
    [ -d ${TMP_CONFIG_PATH} ] && rm -rf ${TMP_CONFIG_PATH}/* &> /dev/null
    return 0
}



# print text with color -- green
# parameters:   $1 -- text
function fn_echo_succ()
{
    FG=32 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# print text with color -- yellow
# parameters:   $1 -- text
function fn_echo_warn()
{
    FG=33 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# print text with color -- red
# parameters:   $1 -- text
function fn_echo_fail()
{
    FG=31 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# Name        : PRINT_FILE_TO_LOG
# Description : print file content into log file
# Parameters  : $1 the file to be printed
function PRINT_FILE_TO_LOG()
{
    local file_name=$1

    LOGFILE="${LOGFILE:-${CONSTANT_LOGFILE}}"
    
    if [ -f "${file_name}" ]
    then
        local check_content=`cat ${file_name}`
        if [ -z "${check_content}" ]
        then
            LOG "This file <${file_name}> is empty."
            return 1
        else
            LOG "Print <${file_name}>"
            LOG "`cat \"${file_name}\" | sed 's/\r//g'`"
        fi
    else
        LOG "This file <${file_name}> does not exist."
        return 1
    fi
}

#*************************************************************#
# Name:        WARN                                           #
# Description: record the warning message into the logfile    #
#*************************************************************#
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************#
# Name:        ERROR                                          #
# Description: record the error message into the logfile      #
#*************************************************************#
function ERROR()
{
    LOG "ERROR: $*"
}

# Name        : PRINT_LOG
# Description : print log
function PRINT_LOG()
{
    local LOG_LEVEL=$1
    local LOG_INFO=$2

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "[Line-${BASH_LINENO}] ${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}

#*************************************************************#
# Name:        getSysTime                                     #
# Description: get the system time                            #
#*************************************************************#
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}

#*************************************************************#
# Name:         LOG                                           #
# Description: record the message into the logfile            #
#*************************************************************#
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSize=102400
    local maxRecord=3
    local index=2
    local content="$*"
  
    if [  -f ${CONSTANT_LOGFILE} ]
    then
        curSize=`ls -lk ${CONSTANT_LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${CONSTANT_LOGFILE}.${index} ]
                then
                    mv ${CONSTANT_LOGFILE}.${index} ${CONSTANT_LOGFILE}.`expr ${index} + 1`

                fi
                let index--
            done
            mv ${CONSTANT_LOGFILE} ${CONSTANT_LOGFILE}.1
        fi
        if echo "${content}" | grep -w "^WARN:" &> /dev/null
        then
            fn_echo_warn "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        elif echo "${content}" | grep -w "^ERROR:" &> /dev/null
        then
            fn_echo_fail "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        else
            echo -e "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        fi
    else
        mkdir -p `dirname ${CONSTANT_LOGFILE}`
        chmod 750 `dirname ${CONSTANT_LOGFILE}`
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${CONSTANT_LOGFILE}
    fi
}

#*************************************************************#
# Name:         fn_install_sudo                               #
# Description:  install sudo script, use newsudobinpath       #
# Parameters:   NULL                                          #
#*************************************************************#
function fn_install_sudo()
{
    local sudobinpath="$1"
    PRINT_LOG "INFO" "Start to install sudo script"
    #Note:install sudobin_pkg
    local sudopkg="${ABSOLUATELY_PATH}/sudobin_pkg.tar"
    local sudotmpPath="/root/osconfig_${RANDOM}"

    sudo "${sudobinpath}/execute.sh" "${sudopkg}" "${sudotmpPath}" "install.sh" "${sudobinpath}" &> "${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        PRINT_LOG "FATAL" "Install sudo script failure." 
        return 1
    fi
    
    IS_SUDO_INSTALL="true"
    
    PRINT_LOG "INFO" "Install sudo script success."
    
    return 0
}

#*************************************************************#
# Name:         fn_uninstall_sudo                             #
# Description:  uninstall sudo script,use oldsudobinpath      #
# Parameters:                                                 #
#*************************************************************#
function fn_uninstall_sudo()
{    
    local old_sudobinpath="$1"
    local new_sudobinpath="$2"
    
    PRINT_LOG "INFO" "Start to uninstall sudo script"
    #Note:uninstall sudobin_pkg
    local sudopkg="${ABSOLUATELY_PATH}/sudobin_pkg.tar"
    local sudotmpPath="/root/osconfig_${RANDOM}"
    
    sudobinpath="${new_sudobinpath}"
    if [ "${old_sudobinpath}" != "${new_sudobinpath}" -a "${IS_SUCCESS}" == "true" ]
    then
        PRINT_LOG "INFO" "The sudobin path has changed." &>/dev/null
        sudobinpath="${old_sudobinpath}"
    fi
    
    PRINT_LOG "INFO" "The sudobin path is : ${sudobinpath}" &>/dev/null
    
    sudo "${sudobinpath}/execute.sh" "${sudopkg}" "${sudotmpPath}" "uninstall.sh" "${new_sudobinpath}" &> "${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        PRINT_LOG "WARN" "Uninstall sudo script failure."
        return 0
    fi
    
    PRINT_LOG "INFO" "Uninstall sudo script success."
    return 0
}

#*************************************************************#
# Name:         fn_main                                       #
# Description:  main function                                 #
# Parameters:   NULL                                          #
#*************************************************************#
function fn_main()
{
    PRINT_LOG "INFO" "bash $0 $*" &>/dev/null
    
    if [ $# -lt 6 ]
    then
        PRINT_LOG "FATAL" "Usage: $0 -oldsudobin <path> -newsudobin <path> -version <version>"
        PRINT_LOG "FATAL" "e.g:   $0 -oldsudobin /opt/sudobin -newsudobin /usr/local/uniepsudobin -version V1R6C10B100"
        return 3
    fi
    
    if [ "$1" != "-oldsudobin" -o "$3" != "-newsudobin" -o "$5" != "-version" ]
    then
        PRINT_LOG "FATAL" "Usage: $0 -oldsudobin <path> -newsudobin <path> -version <version>"
        PRINT_LOG "FATAL" "e.g:   $0 -oldsudobin /opt/sudobin -newsudobin /usr/local/uniepsudobin -version V1R6C10B100"
        return 3
    fi

    oldsudobin_path="$2"
    newsudobin_path="$4"
    version="$6"
    
    PRINT_LOG "INFO" "oldsudobin_path:<${oldsudobin_path}>, newsudobin_path:<${newsudobin_path}>, version:<${version}>."
    
    if [ -z "${oldsudobin_path}" -o -z "${newsudobin_path}" ]
    then
        PRINT_LOG "FATAL" "The old sudobin path<${oldsudobin_path}> or new sudobin path<${newsudobin_path}> is empty."
        return 3
    fi
    
    if [ -z "${version}" ]
    then
        PRINT_LOG "FATAL" "The version<${version}> is empty."
        return 3
    fi
    
    echo "${version}" | egrep -v "^[a-zA-Z0-9]+$" &>/dev/null
    if [ $? -eq 0 ]
    then
        PRINT_LOG "FATAL" "The ${version} is invalid, the version only contain the chars [a-zA-Z0-9]."
        return 1
    fi

    fn_check_env "${newsudobin_path}" || return $?

    #Note:install sudobin
    fn_install_sudo "${newsudobin_path}" || return $?
    
    #Note: rollback sudo and signtool
    PRINT_LOG "INFO" "Start to rollback sudo and signtool."
    sudo "${newsudobin_path}/sudoconfig_sudo.sh" "rollback" "${version}" &> "${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        PRINT_LOG "FATAL" "Rollback sudo and signtool failure."
        return 1
    fi
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
    PRINT_LOG "INFO" "Rollback sudo and signtool success."
    
    return 0
}


# main entrance

fn_main "$@"
MAIN_RET=$?
FUNC_CLEAN_TMP_FILE
[ -d "${TMP_CONFIG_PATH}" ] && rm -rf "${TMP_CONFIG_PATH}"/tmp*

if [ ${MAIN_RET} -eq 0 ]
then
    IS_SUCCESS="true"
fi

if [ "${IS_SUDO_INSTALL}" == "true" ]
then
    fn_uninstall_sudo "${oldsudobin_path}" "${newsudobin_path}"
fi

PRINT_LOG "INFO" "MAIN_RET=${MAIN_RET}" &> /dev/null
[ ! -z "$ERROR_CODE" ] &&  ERROR_CODE=$(echo ${ERROR_CODE} | xargs -n1 | sort -u | xargs) && PRINT_LOG "INFO" "Error code: ${ERROR_CODE}"
record_syslog "${MAIN_RET}" "${FileName}"
exit ${MAIN_RET}
