#!/usr/bin/env bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.
# File name     : upgrade.sh
# Description   : 欧方升级脚本
###########################################################

CURRENT_DIR=$(cd $(dirname $0);pwd)
LOG_FILE="${CURRENT_DIR}"/upgrade.log
VERSION=$1
OSM_PATH=$2
BUILD_FLAG=$3
WIRELESS_FLAG=$4

. "${CURRENT_DIR}/get_os_info.sh"
. "${CURRENT_DIR}/upgrade_rollback_func.sh"


############################################################
# Name:        backup_sudo
# Description: 备份sudoconfig，升级前需备份
#
############################################################
function backup_sudo() {
    print_log "INFO" "start to backup sudoconfig."
    bash "${SUDOCONFIG_PATH}"/rebuild/sudoconfig_backup.sh -oldsudobin "${SUDOBIN_PATH}" -newsudobin "${SUDOBIN_PATH}" -version "${VERSION}"
    [ $? -ne 0 ] && print_log "ERROR" "backup sudoconfig failed, please read log in ${CURRENT_DIR}/../sudoconfig/rebuild/logs/" && return 1
    # 是否可以回滚sudoconfig的标志
    touch "${CURRENT_DIR}"/backup_sudoconfig.flag
    print_log "INFO" "sucess to backup sudoconfig."
    return 0
}


############################################################
# Name:        upgrade_sudo
# Description: 升级sudoconfig
#
############################################################
function upgrade_sudo() {
    print_log "INFO" "start to upgrade sudoconfig."
    bash "${SUDOCONFIG_PATH}"/rebuild/sudoconfig_upgrade.sh -oldsudobin "${SUDOBIN_PATH}" -newsudobin "${SUDOBIN_PATH}" -version "${VERSION}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade sudoconfig failed, please read log in ${CURRENT_DIR}/../sudoconfig/rebuild/logs/" && return 1
    print_log "INFO" "sucess to upgrade sudoconfig."
    return 0
}


############################################################
# Name:        backup_osconfig
# Description: 备份osconfig，升级前备份
#
############################################################
function backup_osconfig() {
    print_log "INFO" "start to backup osconfig."
    mkdir -p "${OSM_PATH}/../bak_${VERSION}"
    [ $? -ne 0 ] && print_log "ERROR" "create osconfig backup path fail." && return 1
    bash "${OSCONFIG_PATH}"/osconfig_backup.sh -backuppath "${OSM_PATH}/../bak_${VERSION}" -oldinstallpath /usr/local/osconfig -newinstallpath /usr/local/osconfig -version "${VERSION}"
    [ $? -ne 0 ] && print_log "ERROR" "backup osconfig fail, please read log in ${CURRENT_DIR}/../osconfig/logs/" && return 1
    # 是否可以回滚osconfig的标志
    touch "${CURRENT_DIR}"/backup_osconfig.flag
    print_log "INFO" "success to backup osconfig."
    return 0
}


############################################################
# Name:        upgrade_osconfig
# Description: 升级osconfig
#
############################################################
function upgrade_osconfig() {
    print_log "INFO" "start to upgrade osconfig."
    bash "${OSCONFIG_PATH}"/osconfig_upgrade.sh -oldinstallpath /usr/local/osconfig -newinstallpath /usr/local/osconfig -newsudobin "${SUDOBIN_PATH}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade osconfig fail, please read log in ${CURRENT_DIR}/../osconfig/logs/" && return 1
    print_log "INFO" "success to upgrade osconfig"
    return 0
}


############################################################
# Name:        upgrade_common_osconfig
# Description: 升级common_osconfig
#
############################################################
function upgrade_common_osconfig() {
    print_log "INFO" "start to upgrade common osconfig."
    # 是否可以执行系统配置回滚的标志
    touch "${CURRENT_DIR}"/nce_config.flag
    bash /usr/local/osconfig/os/bin/commonosconfig.sh upgrade "${VERSION}" "${BUILD_FLAG}" "${WIRELESS_FLAG}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade common osconfig fail." && return 1
    print_log "INFO" "success to upgrade common osconfig."
    return 0
}

############################################################
# Name:        pre_upgrade
# Description: 升级前置操作
#
############################################################
function pre_upgrade() {
    print_log "INFO" "start to execute operation before upgrade."
    # 解锁华为suse lock服务锁住的目录，其他os不涉及
    sudo "${SUDOBIN_PATH}"/execute.sh "${CURRENT_DIR}/sudobin-upgrade_pkg.tar" "/opt/upgrade_config_$(date '+%Y%m%d%H%M%S')" pre_upgrade.sh
    print_log "INFO" "finish to execute operation before upgrade."
    return 0
}


############################################################
# Name:        main
# Description: 主函数，入口
#
############################################################
function main() {
    # 修改文件权限
    modify_file_perm
    # 升级前置操作
    pre_upgrade || return 1

    # 解压osconfig和sudoconfig包
    unzip_cloudsop_pack || return 1

    # 备份sudoconfig
    backup_sudo || return 1

    # 升级sudoconfig
    upgrade_sudo || return 1

    # 备份osconfig
    backup_osconfig || return 1

    # 升级osconfig
    upgrade_osconfig || return 1

    # 升级OS配置
    upgrade_common_osconfig || return 1

    # 清理升级路径的目录及文件
    clean_after_upgrade

    print_log "INFO" "success to upgrade."
    return 0
}

print_log "INFO" "start to upgrade, version: ${VERSION}" >> "${LOG_FILE}" 2>&1

main >> "${LOG_FILE}" 2>&1
result=$?
if [ "${result}" -eq 0 ];then
    touch "${CURRENT_DIR}"/upgrade_succeed_flag
else
    touch "${CURRENT_DIR}"/upgrade_fail_flag
fi
exit "${result}"






