# -*- coding: UTF-8 -*-

"""
Perform repconf config for logicrep tool.
Copyright © Huawei Technologies Co., Ltd. 2010-2018. All rights reserved.
"""

import os
import shutil
import sys
import re
import xml.etree.ElementTree as ET
from xml.etree.ElementTree import ElementTree, Element


# Read and parse the xml file
# in_path: xml file path
# return: ElementTree
def read_xml(in_path):
    tree_temp = ElementTree()
    tree_temp.parse(in_path)
    return tree_temp


# Write out the xml file
# tree_temp: xml tree info
# out_path: The path of the xml file to be written
def write_xml(tree_temp, out_path):
    tree_temp.write(out_path, encoding="utf-8", xml_declaration=True)


# Determine whether a node contains all incoming parameter attributes
# node: xml node
# kv_map: the map of node attributes and node attribute values
def if_match(node, kv_map):
    for key in kv_map:
        if node.get(key) != kv_map.get(key):
            return False
    return True


# Find all nodes that match a path
# tree_temp: xml tree
# path: Node path
# return: nodes
def find_nodes(tree_temp, path):
    return tree_temp.findall(path)


# Create a new node
# tag: Node label name
# property_map:the map of node attributes and node attribute values
# content: Text content in node closure label
# return: new node
def create_node(tag, property_map, content):
    element = Element(tag, property_map)
    element.text = content
    return element


# Add child nodes to a node
# nodelist: Node list
# element: Child node element
def add_child_node(nodelist, element):
    for node in nodelist:
        node.append(element)


# Locate and delete a node based on attributes and attribute values
# nodelist: Parent node list,
# tag:Node label name,
# kv_map: the map of node attributes and node attribute values
def del_node_by_tagkeyvalue(nodelist, tag, kv_map):
    for parent_node in nodelist:
        children = [i for i in parent_node]
        for child in children:
            if child.tag == tag and if_match(child, kv_map):
                parent_node.remove(child)


# Amend xml format
def prettyXml(element, indent, newline, level=0):
    if len(element) != 0:
        if element.text is None or element.text.isspace():
            element.text = newline + indent * (level + 1)
        else:
            element.text = newline + indent * (level + 1) \
                           + element.text.strip() \
                           + newline + indent * (level + 1)
    temp = list(element)
    for subelement in temp:
        # If it is not the last element, the next line is an element at the same level,
        # and the indentation should be consistent
        if temp.index(subelement) < (len(temp) - 1):
            subelement.tail = newline + indent * (level + 1)
        # If it is the last element, the next line is the end of the parent element,
        # and the indentation should be one less
        else:
            subelement.tail = newline + indent * level
        prettyXml(subelement, indent, newline, level=level + 1)
    return element


# backup file
def copy_files(file):
    bakname = os.path.basename(file)
    shutil.copy(file, "conf/" + bakname + "_bak")


# Delete the tableMapping node in the modelMapping section
def init_repconf_db(file):
    del_parent_nodes = find_nodes(tree, "modelMapping")
    for parent_node in del_parent_nodes:
        children = [i for i in parent_node]
        for child in children:
            if child.tag == "tableMapping":
                parent_node.remove(child)
    while len(find_nodes(tree, "modelMapping/tableMapping")) != 0:
        init_repconf_db(file)
    write_xml(tree, file)


# Delete the repNameInfo node in the repName section
def init_repNameInfo(file):
    del_parent_nodes = find_nodes(tree, "repName")
    for parent_node in del_parent_nodes:
        children = [i for i in parent_node]
        for child in children:
            if child.tag == "repNameInfo":
                parent_node.remove(child)
    while len(find_nodes(tree, "repName/repNameInfo")) != 0:
        init_repNameInfo(file)
    write_xml(tree, file)


# Delete the userInfo node in the filteredUser section
def init_userInfo(file):
    del_parent_nodes = find_nodes(tree, "filteredUser")
    for parent_node in del_parent_nodes:
        children = [i for i in parent_node]
        for child in children:
            if child.tag == "userInfo":
                parent_node.remove(child)
    while len(find_nodes(tree, "filteredUser/userInfo")) != 0:
        init_userInfo(file)
    write_xml(tree, file)


# delete the modelMapping section
def delete_node_modelMapping(srcTable, file):
    if len(srcTable) == 0:
        print("modelMapping is null, please check the input")
        exit()
    del_parent_nodes = find_nodes(tree, "modelMapping")
    del_node_by_tagkeyvalue(del_parent_nodes, "tableMapping",
                            {"srcTable": srcTable})
    write_xml(tree, file)


# delete the userInfo section
def delete_node_userInfo(userInfo, file):
    if len(userInfo) == 0:
        print("modelMapping is null, please check the input")
        exit()
    del_parent_nodes = find_nodes(tree, "filteredUser")
    del_node_by_tagkeyvalue(del_parent_nodes, "userInfo",
                            {"userName": userInfo})
    write_xml(tree, file)


# add child node of modelMapping
def add_node_modelMapping(srcTable, srcSchema, dstTable, dstSchema, columinfo,
                          file):
    add_parent_nodes = find_nodes(tree, "modelMapping")
    a = create_node("tableMapping",
                    {"srcTable": srcTable, "srcSchema": srcSchema,
                     "dstTable": dstTable, "dstSchema": dstSchema}, "")
    add_child_node(add_parent_nodes, a)
    if columinfo == "":
        pass
    else:
        key_list = columinfo.split(',')
        for key in key_list:
            if '(' in key and ')' in key:
                key = re.findall(r"[(](.*)[)]", key)[0]
            key = '"' + key + '"'
            b = create_node("column", {"srcColumn": key, "dstColumn": key},
                            "")
            c = ET.Element("ignoreUpdatecolumns")
            c.append(b)
            a.append(c)
    write_xml(tree, file)


# add repNameInfo node in repName section
def add_repNameInfo(repNameInfo_name, file):
    add_parent_nodes = find_nodes(tree, "repName")
    a = create_node("repNameInfo", {"name": repNameInfo_name}, "")
    add_child_node(add_parent_nodes, a)
    write_xml(tree, file)


# add userInfo node in filteredUser section
def add_userInfo(userInfo_userName, file):
    add_parent_nodes = find_nodes(tree, "filteredUser")
    a = create_node("userInfo", {"userName": userInfo_userName}, "")
    add_child_node(add_parent_nodes, a)
    write_xml(tree, file)


# add column of table mapping node
def add_column(srcColumn, dstColumn, isKey, file):
    add_nodes = find_nodes(tree, "modelMapping/tableMapping")
    if len(isKey) == 0:
        a = create_node("column", {"srcColumn": srcColumn,
                                   "dstColumn": dstColumn}, "")
        add_child_node(add_nodes, a)
    else:
        a = create_node("column", {"srcColumn": srcColumn,
                                   "dstColumn": dstColumn, "isKey": isKey},
                        "")
        add_child_node(add_nodes, a)
    write_xml(tree, file)


# Generate new xml file
def write_new_file(file):
    tree_new = read_xml(file)
    root_new = tree_new.getroot()
    root_new = prettyXml(root_new, '\t', '\n')
    tree_new = ET.ElementTree(root_new)
    tree_new.write(file, encoding="utf-8", xml_declaration=True)


def show_help_imfomation():
    print("python repconf_db_confige.py xml_path init/add_modelMapping"
          "/delete_modelMapping/add_column parameters")


if __name__ == "__main__":
    current_path = os.path.dirname(os.path.realpath(__file__))
    src_xml_file = os.path.join(current_path, sys.argv[1])
    tree = read_xml(src_xml_file)
    if len(sys.argv) < 2:
        show_help_imfomation()
        exit()
    elif len(sys.argv) == 3 and sys.argv[2] == "init":
        copy_files(src_xml_file)
        init_repconf_db(src_xml_file)
    elif len(sys.argv) == 4 and sys.argv[2] == "delete_modelMapping":
        delete_node_modelMapping(sys.argv[3], src_xml_file)
    elif len(sys.argv) == 4 and sys.argv[2] == "delete_userInfo":
        delete_node_userInfo(sys.argv[3], src_xml_file)
    elif sys.argv[2] == "add_modelMapping" and len(sys.argv) == 8:
        print("add_modelMapping")
        add_node_modelMapping(sys.argv[3], sys.argv[4], sys.argv[5],
                              sys.argv[6], sys.argv[7], src_xml_file)
    elif sys.argv[2] == "add_column" and len(sys.argv) == 5:
        add_column(sys.argv[3], sys.argv[4], "", src_xml_file)
    elif sys.argv[2] == "add_column" and len(sys.argv) == 6:
        add_column(sys.argv[3], sys.argv[4], sys.argv[5], src_xml_file)
    elif sys.argv[2] == "init_repNameInfo" and len(sys.argv) == 3:
        init_repNameInfo(src_xml_file)
    elif sys.argv[2] == "init_userInfo" and len(sys.argv) == 3:
        init_userInfo(src_xml_file)
    elif sys.argv[2] == "add_repNameInfo" and len(sys.argv) == 4:
        add_repNameInfo(sys.argv[3], src_xml_file)
    elif sys.argv[2] == "add_userInfo" and len(sys.argv) == 4:
        add_userInfo(sys.argv[3], src_xml_file)
    else:
        print("please check input, only init/add_modelMapping/"
              "delete_modelMapping/add_column allows to be modified")
        exit()
    write_new_file(src_xml_file)
    exit()
