# coding=utf-8
'''
update_service_cert
'''
import copy
import os
import sys
import zipfile
import shutil

from utils import set_log
from utils import utils

LOG = set_log.set_log(stream=True)
SUCCESS = 0
REQ_FAILED = 1

ZTP_CERT_PATH = "/opt/pub/software/repository/Services/Common/ZtpAppCerts"
CERTS_NAME = ["TRUST_Huawei_Enterprise_Network_Product_CA.cer", "TRUST_Huawei_Equipment_Root_CA.der"]

INPUT_PARAM = {
    "serviceId": "ZTP_CA_886822bb-fb85-48a2-92be-c4296d6a30b3",
    "uploadType": "allinone",
    "tenantId": None,
    "certFiles": []
}

CERT_FILES = {
    "certName": "",
    "certType": "TRUST",
    "fileFormat": "PEM",
    "content": "",
}


def remove_dir(dir_path):
    if os.path.exists(dir_path):
        files = os.listdir(dir_path)
        for file in files:
            filepath = os.path.join(dir_path, file).replace("\\", '/')
            if os.path.isdir(filepath):
                remove_dir(filepath)
            else:
                os.remove(filepath)
        os.rmdir(dir_path)


def check_zip(zip_path):
    '''
    解压前检查，防止zip炸弹
    :param zip_path:
    :return:
    '''
    # 限制解压后大小不能超过10M(10 * 1024 * 1024单位为b)，文件个数不能超过20个
    max_size = 10 * 1024 * 1024
    max_file_count = 20
    zip_file = zipfile.ZipFile(zip_path, 'r')

    # 检查点1：检查文件个数，文件个数大于预期值时上报异常退出
    file_count = len(zip_file.infolist())
    if file_count >= max_file_count:
        raise IOError(f'zipfile({zip_file}) contains {file_count} files exceed max file count')

    # 检查点2：检查第一层解压文件总大小，总大小超过设定的上限值(返回的单位大小是b)
    total_size = sum(info.file_size for info in zip_file.infolist())
    if total_size >= max_size:
        raise IOError(f'zipfile({zip_file}) size({total_size}) exceed max limit')

    # 检查点3：检查第一层解压文件总大小，总大小超过磁盘剩余空间(返回的单位大小是b)
    _, _, free_size = shutil.disk_usage(ZTP_CERT_PATH)
    if total_size >= free_size:
        raise IOError(f'zipfile({zip_file}) size({total_size}) exceed remain target disk space')


def get_ztp_cert():
    res = []
    zip_files = []
    temp_directory = os.path.join(ZTP_CERT_PATH, "tmp")
    if os.path.exists(temp_directory):
        remove_dir(temp_directory)
    os.makedirs(temp_directory)
    for _, _, files in os.walk(ZTP_CERT_PATH):
        for file in files:
            if file.endswith("zip"):
                zip_files.append(file)
    zip_name = sorted(zip_files)[-1]
    zip_path = os.path.join(ZTP_CERT_PATH, zip_name)
    check_zip(zip_path)
    zip_file = zipfile.ZipFile(zip_path)
    zip_file.extractall(temp_directory)
    for cert_name in CERTS_NAME:
        zip_file_path = os.path.join(temp_directory, "etc/ssl/ztpapp", cert_name)
        if os.path.isfile(zip_file_path):
            req = copy.deepcopy(INPUT_PARAM)
            cert_files = copy.deepcopy(CERT_FILES)
            cert_files["certName"] = cert_name
            with open(zip_file_path, "r") as f:
                cert_content = f.read()
            cert_files["content"] = cert_content \
                .lstrip("-----BEGIN CERTIFICATE-----") \
                .rstrip("-----END CERTIFICATE-----") \
                .strip("\n") \
                .replace("\n", "")
            req["certFiles"].append(cert_files)
            res.append(copy.deepcopy(req))
    remove_dir(temp_directory)
    return res


def update_service_cert(input_data):
    _ip = utils.get_local_ip()
    port = utils.get_ir_port()
    url = "/rest/snbcertmgmt/v1/auto/service-certs"
    result, ret = utils.send_rest(_ip, port, url, mode="POST", log=LOG,
                                  body=input_data, sleep=6, retry_times=30,
                                  content="update service cert")
    return result, ret


def main():
    """
    主函数
    :return: result
    """

    res_code = SUCCESS
    req = get_ztp_cert()
    print(req)
    print("-" * 150)
    for input_data in req:
        result, ret = update_service_cert(input_data)
        print(result)
        print(ret)
        if result == 1:
            res_code = REQ_FAILED
    return res_code


sys.exit(main())
