import re
import os
import sys
import json
import datetime
import subprocess
import platform
import multiprocessing
from concurrent.futures import ProcessPoolExecutor, wait

sys.path.append(os.path.split(os.path.abspath(__file__))[0])
from pyscripts.utils import set_log


class OneHopUpgrade:
    def __init__(self, upgrade_path, upgrade_json_file, shell_log_file, tenant_name, step, filter_condition_dict, log, variable_dict, record_name=""):
        log_dir = os.path.split(shell_log_file)[0]
        self.log = log
        self.log.info("params is : %s", locals())
        self.upgrade_tasks_after_filter = []
        self.upgrade_path = upgrade_path
        self.tenant_name = tenant_name
        self.shell_log_file = shell_log_file
        self.upgrade_json_file = upgrade_json_file
        self.step = step
        self.success = 0
        self.fail = 1
        self.split_str = "^^##&&^^"
        self.result_str = "result_str"
        self.variable_dict = variable_dict
        self.common = "common"
        self.variable = "variable"
        self.is_execute_after_success = "is_execute_after_success"
        self.params = "params"
        self.is_to_screen = "is_to_screen"
        self.is_to_screen_after_success = "is_to_screen_after_success"
        self.responsibility_field = "responsibility_field"
        self.script = "script"
        self.description_zh = "description_zh"
        self.description_en = "description_en"
        self.filter_condition = "filter_condition"
        self.order = "order"
        self.fail_stop = "fail_stop"
        self.version = "version"
        self.src = "src"
        self.des = "des"
        if record_name:
            self.record = os.path.join("record", record_name)
        else:
            self.record = "record"
        self.format_time = "%Y-%m-%d %H:%M:%S"
        self.record_dir = os.path.join(log_dir, self.record, self.step)
        self.cost_time_record = os.path.join(log_dir, self.record, "cost_time.log")
        self.src_version, self.des_version = self.deal_upgrade_path()
        self.make_record_dir()
        self.filter_condition_dict = filter_condition_dict
        self.upgrade_json_dir = self.get_upgrade_json_dir()

    def get_upgrade_json_dir(self):
        '''
        获取 json_file所在的路径，json_file的传入有两种场景，
        1、不传入，默认为upgrade.json，此时 upgrade.json 和 __file__同级目录
        2、另一种，调用secomanager相关脚本，传入 upgrade.json 的绝对路径
        :return:
        '''
        # 记录当前路径
        now_dir = os.getcwd()

        # 首先获取 python所在目录，并切换到当前目录
        python_dir = os.path.split(os.path.abspath(sys.argv[0]))[0]
        os.chdir(python_dir)

        # 然后获取upgade.json的绝对路径，如果传入的是 upgrade.json，则转化为绝对路经；
        # 否则 upgrade.json本身就是绝对路径，值不变
        upgrade_json_dir = os.path.split(os.path.abspath(self.upgrade_json_file))[0]
        self.log.info("the upgrade_json_dir is %s ", upgrade_json_dir)

        # 路径切回去，以免之前的调用有问题
        os.chdir(now_dir)
        return upgrade_json_dir

    def make_record_dir(self):
        '''
        每个步骤的记录日志
        :return:
        '''
        if not os.path.exists(self.record_dir):
            os.makedirs(self.record_dir)

    def deal_upgrade_path(self):
        self.log.info("deal_upgrade_path")
        src_version, des_version = -1, -1
        upgrade_path = self.upgrade_path.upper()
        version_re = re.compile("R(\d+)C(\d+)")
        src_version_group = re.search(version_re, upgrade_path.split("-")[0])
        if src_version_group:
            src_version = self.deal_version_group(src_version_group)
        des_version_group = re.search(version_re, upgrade_path.split("-")[1])
        if des_version_group:
            des_version = self.deal_version_group(des_version_group)
        self.log.info("src_version: %s, des_version: %s", src_version, des_version)
        return src_version, des_version

    def deal_version_group(self, version_group):
        '''
        处理正则查出来的
        :param version_group:
        :return: version
        '''
        r_version = version_group.group(1)
        c_version = version_group.group(2)
        version = eval("%s.%s" % (int(r_version), int(c_version)))
        return version

    def read_config(self):
        '''
        读取配置文件
        :return:
        '''
        with open(self.upgrade_json_file, "rb") as _f:
            upgrade_content = json.load(_f)
        return upgrade_content

    def get_task(self):
        '''
        获取升级任务
        :return:
        '''
        self.log.info("get_task")
        upgrade_order_dict = {}
        upgrade_content = self.read_config()
        upgrade_tasks_before_filter = upgrade_content.get(self.step, [])
        self.log.info("the %s task is %s" % (self.step, upgrade_tasks_before_filter))
        for task in upgrade_tasks_before_filter:
            self.log.info("start deal the task [%s]", task[self.description_zh])
            order = task[self.order]
            is_append = True
            if self.filter_condition in task:
                for key, value in task[self.filter_condition].items():
                    # 对version的过滤条件判断
                    if key == self.version:
                        src_filter = task[self.filter_condition][key].get(self.src, 'all')
                        des_filter = task[self.filter_condition][key].get(self.des, 'all')
                        self.log.info("version filter: src_filter is %s, des_filter is %s", src_filter, des_filter)
                        if src_filter != "all":
                            src_filter_dict = {"%s_%s" % (self.src, self.version): self.src_version}
                            self.log.info("the src_filter is %s", src_filter.format(**src_filter_dict))
                            if not eval(src_filter.format(**src_filter_dict)):
                                self.log.info("the src_filter result is False")
                                is_append = False
                        if des_filter != "all":
                            des_filter_dict = {"%s_%s" % (self.des, self.version): self.des_version}
                            self.log.info("the des_filter is %s", des_filter.format(**des_filter_dict))
                            if not eval(des_filter.format(**des_filter_dict)):
                                self.log.info("the des_filter result is False")
                                is_append = False
                    # 对其他的条件判断
                    else:
                        self.log.info("the %s filter is %s, the real value is %s",
                                      key, value, self.filter_condition_dict[key])
                        if self.filter_condition_dict[key] != value:
                            self.log.info("the %s filter result is False", key)
                            is_append = False
            # 将符合结果的步骤加入upgrade_order_dict
            if is_append:
                self.log.info("the [%s] will be executed, add to upgrade_order_dict", task[self.description_zh])
                if order in upgrade_order_dict:
                    upgrade_order_dict[order].append(task)
                else:
                    upgrade_order_dict[order] = []
                    upgrade_order_dict[order].append(task)
            else:
                self.log.info("the [%s] will not be executede", task[self.description_zh])
        self.log.info("the upgrade_order_dict is %s", upgrade_order_dict)
        self.deal_upgrade_order_dict(upgrade_order_dict)
        self.log.info("the upgrade_tasks_after_filter is %s", self.upgrade_tasks_after_filter)

    def deal_upgrade_order_dict(self, upgrade_order_dict):
        '''
        :param upgrade_order_dict:
        :return:
        '''
        keys = sorted(list(upgrade_order_dict.keys()))
        for key in keys:
            self.upgrade_tasks_after_filter.append(upgrade_order_dict[key])

    def deal_fail_stop(self, task):
        '''
        对任务失败的处理，看是否升级终止
        :param task:
        :return:
        '''
        is_fail_stop = task.get(self.fail_stop, "true")
        is_to_screen = task.get(self.is_to_screen, "false")
        self.log.info("[%s], %s: %s", task[self.script], self.is_to_screen, is_to_screen)
        # 这里要进行一个判断，因为json处理出来的is_to_screen，有可能是字符串，也可能是bool值
        if isinstance(is_to_screen, str) and is_to_screen.lower() == "true":
            print("%s%s" % (task.get(self.result_str, ""), self.split_str))
        elif isinstance(is_to_screen, bool) and is_to_screen:
            print("%s%s" % (task.get(self.result_str, ""), self.split_str))

        # 这里要进行一个判断，因为json处理出来的fail_stop，有可能是字符串，也可能是bool值
        if isinstance(is_fail_stop, str) and is_fail_stop.lower() == "true":
            # 责任田归属
            if task.get(self.responsibility_field, ""):
                print("[ ERROR ] Failed to execute the script %s.\n"
                      "The upgrade log file is %s.\n"
                      "The responsibility field is %s.\n%s"
                      % (task["script"], self.shell_log_file, task[self.responsibility_field], self.split_str))
            return self.fail
        elif isinstance(is_fail_stop, bool) and is_fail_stop:
            # 责任田归属
            if task.get(self.responsibility_field, ""):
                print("[ ERROR ] Failed to execute the script %s.\n"
                      "The upgrade log file is %s.\n"
                      "The responsibility field is %s.\n%s"
                      % (task["script"], self.shell_log_file, task[self.responsibility_field], self.split_str))
            return self.fail
        else:
            self.log.error("the task [%s] execution failed, but fail_stop is %s, upgrade will continue",
                          task[self.description_en], is_fail_stop)
        return self.success

    def deal_variable(self, task):
        '''
        对变量的处理
        :param task:
        :return:
        '''
        # 执行获取变量
        if task.get(self.variable):
            value = task.get(self.result_str)
            self.variable_dict[task.get(self.variable)] = value
        # 成功也打印信息到屏幕
        is_to_screen_after_success = task.get(self.is_to_screen_after_success, "false")
        self.log.info("[%s], %s: %s", task[self.script], self.is_to_screen, is_to_screen_after_success)
        # 这里要进行一个判断，因为json处理出来，有可能是字符串，也可能是bool值
        if isinstance(is_to_screen_after_success, str) and is_to_screen_after_success.lower() == "true":
            print("%s%s" % (task.get(self.result_str, ""), self.split_str))
        elif isinstance(is_to_screen_after_success, bool) and is_to_screen_after_success:
            print("%s%s" % (task.get(self.result_str, ""), self.split_str))

    def exec_tasks(self):
        for tasks in self.upgrade_tasks_after_filter:
            self.log.info("the variable dict is %s" % self.variable_dict)
            if len(tasks) == 1:
                child_task = tasks[0]
                result, child_task_after_exec_shell = self.exec_child_task(child_task)
                if result != self.success and self.deal_fail_stop(child_task_after_exec_shell) == self.fail:
                    return self.fail
                # 对变量的处理
                self.deal_variable(child_task_after_exec_shell)
            else:
                pool = ProcessPoolExecutor(max_workers=multiprocessing.cpu_count() // 2 + 1)
                futures = []
                for child_task in tasks:
                    futures.append(pool.submit(self.exec_child_task, child_task))
                wait(futures)
                success = True
                for future in futures:
                    future_result = future.result()
                    future_task = future_result[1]
                    if future_result[0] != self.success and self.deal_fail_stop(future_task) == self.fail:
                        success = False
                    # 对变量的处理
                    if success:
                        self.deal_variable(future_task)
                if not success:
                    return self.fail
        return self.success

    def exec_child_task(self, child_task):
        '''
        开始执行任务，列表长度为 1 的单线程执行；列表长度大于 1 的，进程池执行
        :param child_task:
        :return:
        '''
        script_mame = child_task[self.script]
        description = child_task[self.description_en].replace(" ", "_")
        is_execute_after_success = child_task.get(self.is_execute_after_success, "false")
        record_file = os.path.join(self.record_dir, description)

        # 判断是否需要再次执行(is_execute_after_success 为 false说明，成功后不再执行)
        need_execute = True
        result = self.success
        if (isinstance(is_execute_after_success, str) and is_execute_after_success.lower() == "false") or (
                isinstance(is_execute_after_success, bool) and not is_execute_after_success):
            if os.path.exists(record_file):
                with open(record_file, "r") as _f:
                    if _f.read() == str(self.success):
                        self.log.info("[%s] has been executed successfully, skip", description)
                        need_execute = False

        # 执行脚本F
        if need_execute:
            script_path = os.path.join(self.upgrade_json_dir, script_mame)
            self.log.info("start to execute [%s], script_name is %s", description, script_path)
            if not os.path.exists(script_path):
                self.log.error("the file [%s] does not exists", script_path)
                return self.fail, child_task
            start_time = datetime.datetime.now()
            command = r'sh %s "%s" "%s" "%s"' % (script_path, self.upgrade_path, self.tenant_name, self.shell_log_file)
            # 针对参数进行处理
            for param in child_task.get(self.params, []):
                if "%s:" % self.variable in str(param):
                    param = self.variable_dict[param.split("%s:" % self.variable)[1]]
                command += ' "%s"' % param
            self.log.info("command is %s", command)
            result, result_str = subprocess.getstatusoutput(command)
            child_task["result_str"] = result_str
            if result == self.success:
                end_time = datetime.datetime.now()
                cost_time = (end_time - start_time).seconds
                self.write_cost_time_record(cost_time, start_time, end_time, script_mame)
                self.log.info("successfully executed [%s], cost_time: %ss", script_path, cost_time)
                with open(record_file, "w") as _f:
                    _f.write(str(self.success))
            else:
                self.log.error("failed executed [%s]", script_path)
        return result, child_task

    def write_cost_time_record(self, cost_time, start_time, end_time, script_mame):
        '''
        记录时间
        :param cost_time:
        :param start_time:
        :param end_time:
        :param description:
        :return:
        '''
        start_time = datetime.datetime.strftime(start_time, self.format_time)
        end_time = datetime.datetime.strftime(end_time, self.format_time)
        with open(self.cost_time_record, "a+") as _f:
            _f.write("%s.%s cost time %ss, start_time: %s, end_time: %s\n" % (
                self.step, script_mame, cost_time, start_time, end_time))


def main(upgrade_path, tenant_name, shell_log_file, step, filter_condition_dict=None, upgrade_json_file="upgrade.json", record_name="", variable_dict=None):
    if filter_condition_dict is None:
        filter_condition_dict = {}
    if variable_dict is None:
        variable_dict = {}
    log_dir = os.path.split(shell_log_file)[0]
    if not os.path.exists(log_dir):
        os.makedirs(log_dir)
    log = set_log.set_log(stream=False, filename=os.path.join(log_dir, "%s.log" % os.path.basename(__file__)))
    log.info("\n\n{0}Start{0}".format("-" * 50))
    try:
        one_hop_upgrade = OneHopUpgrade(upgrade_path, upgrade_json_file, shell_log_file, tenant_name, step, filter_condition_dict, log, variable_dict, record_name)
        one_hop_upgrade.get_task()
        return one_hop_upgrade.exec_tasks()
    except:
        log.exception("Exception Logged")
        return 1


if __name__ == '__main__':
    tenant_name = "NCECOMMONE"
    shell_log_file = "/opt/oss/one_hop_upgrade/log/test.log"
    upgrade_path = "V300R019C00XXX-V100R021C00SPC101B022"
    step = "pre_custom"
    main(upgrade_path, tenant_name, shell_log_file, step)
