#!/bin/bash

execute_phase=$1
cur_path=$(cd "$(dirname "$0")"; pwd)

# 迁移工具根目录
export MIGRATE_ROOT=$(cd "${cur_path}/../"; pwd)
# 迁移工具文件

# 迁移过程中，存放中间数据
[[ -z $MIGRATE_DATA_DIR ]] && MIGRATE_DATA_DIR=/opt/migrate/data
export MIGRATE_DATA_DIR

# 迁移日志目录
mkdir -p $MIGRATE_DATA_DIR/logs

#迁移调度日志
run_log=$MIGRATE_DATA_DIR/logs/run.log

function log()
{
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] | $$ | $1" >> $run_log
}

function get_py_file()
{
    local py_file=$1
    [[ ! -f $py_file ]] && py_file="${py_file}c"
    echo ${py_file}
}

MIGRATE_TOOL=$(get_py_file ${MIGRATE_ROOT}/src/frame/migrate_data.py)

function init_db_driver()
{
    local db_type=$1
    if [[ $db_type == 'gauss' ]]
	  then
        driver_path=/opt/oss/envs
	  else
        driver_path=/opt/oss/manager/apps
	  fi
    local db_driver=$(find ${driver_path}/{UniEPAgent,DBAgent,HealthCheckAgent}/ -maxdepth 5 -type d -name dbdriver_${db_type} 2>/dev/null | xargs ls -td | head -n 1)
    if [[ ! -d ${db_driver} ]]
    then
        log "${db_type} driver not found"
        exit 1
    fi

    local db_path=${db_driver}/pyscript/${db_type}
    if [[ ${db_type} == "gauss" && ! -f ${db_driver}/pyscript/gauss/gsdb.py ]]
    then
        db_path=${MIGRATE_ROOT}/src/frame/gauss_driver
        mkdir -p ${db_path}
        find ${db_driver}/pyscript/gauss/ ! -name "*.pyc" -type f -exec cp -rf {} ${MIGRATE_ROOT}/src/frame/gauss_driver/ \; >> $run_log 2>&1
    fi
    log "set ${db_type} driver path:${db_path}"
    export PYTHONPATH=${PYTHONPATH}:${db_path}
    export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:${PYTHONPATH}:${db_path}
    return 0
}

function init_kmc_version()
{
    CIPHER_MODE="kmc_v1"
    [[ -f "${OSS_ROOT}/etc/cipher/cloudsop_cipher_mode.flag" ]] && CIPHER_MODE="cloudsop"
    [[ -f "${OSS_ROOT}/etc/cipher/domain_mapping.json" ]] && CIPHER_MODE="kmc_v2"
    export CIPHER_MODE
}

#初始化基础环境变量
function init_env()
{
    # OSS根目录
    [[ -z $OSS_ROOT ]] && OSS_ROOT=/opt/oss/manager
    [[ -z ${IPMC_ROOT} ]] && export IPMC_ROOT=${OSS_ROOT}
    APP_ROOT=${OSS_ROOT}/apps/DBAgent
    export APP_ROOT
    export OSS_ROOT

    init_kmc_version

    # 迁移源环境版本NCE-E_R19C00_xxx/NCE-E_R19C10/NCE-E_R20C00
    [[ -z $MIGRATE_SOURCE_VERSION ]] && MIGRATE_SOURCE_VERSION=NCE-E_R19C00spcxxx
    export MIGRATE_SOURCE_VERSION

    # 迁移目标环境版本
    [[ -z $MIGRATE_TARGET_VERSION ]] && MIGRATE_TARGET_VERSION=NCE-E_R20C10spcxxx
    export MIGRATE_TARGET_VERSION

    # 源和目标container_list_[old,new].json文件所在目录
    [[ -z $MIGRATE_CONTAINER_DIR ]] && MIGRATE_CONTAINER_DIR=${MIGRATE_ROOT}/conf/connections

    if [[ -d ${OSS_ROOT}/etc/cipher ]]
    then
        export CIPHER_ROOT=${OSS_ROOT}/etc/cipher
        export _APP_LOG_DIR=${MIGRATE_DATA_DIR}/logs
        log "set CIPHER_ROOT=${CIPHER_ROOT} and _APP_LOG_DIR=${_APP_LOG_DIR}"
    fi

    # python环境变量
    local deploy_agent_rtsp=${OSS_ROOT}/agent/DeployAgent/rtsp
    local mccommon_path=${deploy_agent_rtsp}/mccommon
    local python_home=${deploy_agent_rtsp}/python
    if [[ ! -d $python_home ]]
    then
        log "no python tool: $python_home"
        exit 1
    fi
    export PYTHONPATH=${MIGRATE_ROOT}/src:${MIGRATE_ROOT}/src/apps:${mccommon_path}/tools/pyscript:${mccommon_path}/tools/pyscript/cbb:${PYTHONPATH}
    export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:${python_home}/lib
    export PATH=${python_home}/bin:${PATH}

    # python db driver
    for db_type in gauss zenith
    do
        init_db_driver ${db_type}
        local db_home=/opt/${db_type}/app
        if [[ -d $db_home ]]
        then
            log "set ${db_type} client home"
            export PATH=${db_home}/bin:$PATH
            export LD_LIBRARY_PATH=${db_home}/lib:${LD_LIBRARY_PATH}
        fi
    done
}

#设置日志文件
function set_log_file_name()
{
    local process_step=$1
    export LOG_FILE="${MIGRATE_DATA_DIR}/logs/${process_step}_$(date '+%Y%m%d_%H%M%S').log"
}

#初始化源导出环境变量
function init_source_env()
{
    # 迁移源container_list.json文件
    export MIGRATE_CONTAINER_FILE=${MIGRATE_CONTAINER_DIR}/containerlist_old.json

    #清空导出数据目录
    local data_dir=${MIGRATE_DATA_DIR}/data
    if [[ -d ${data_dir} ]]
    then
        rm -rf ${data_dir}
    fi
}

#初始化目标导入环境变量
function init_target_env()
{
     # 迁移目标container_list.json文件
     export MIGRATE_CONTAINER_FILE=${MIGRATE_CONTAINER_DIR}/containerlist_new.json
}

function export_data()
{
    init_source_env
    set_log_file_name 'export_data'
    python ${MIGRATE_TOOL} export_data >> $run_log 2>&1
    result=$?
    . ${cur_path}/auth_cert_file.sh export_data
    . ${cur_path}/nbi_cert_upgrade.sh export_data
    return ${result}
}

function import_data()
{
    init_target_env
    set_log_file_name 'import_data'
    python ${MIGRATE_TOOL} import_data >> $run_log 2>&1
    result=$?
    . ${cur_path}/auth_cert_file.sh import_data
    . ${cur_path}/nbi_cert_upgrade.sh import_data
    return ${result}
}

function post_process()
{
    init_target_env
    set_log_file_name 'post_process'
    python ${MIGRATE_TOOL} post_process >> $run_log 2>&1
    return $?
}

function execute_single_task()
{
    local app_name=$1
    local step=$2

    if [[ -z "${app_name}" || -z "${step}" ]]
    then
        log "app_name:${app_name} step:${step} not found"
        return 1
    fi

    if [[ $step == "export" ]]
    then
        init_source_env
    else
        init_target_env
    fi

    set_log_file_name "${app_name}_${step}"
    python ${MIGRATE_TOOL} single ${app_name} ${step} >> $run_log 2>&1
    return $?
}

function main()
{
    init_env
    umask 0000
    log "all paramter: $*, with $(python --version),MIGRATE_ROOT=${MIGRATE_ROOT},PYTHONPATH=${PYTHONPATH},PATH=${PATH}, LD_LIBRARY_PATH=${LD_LIBRARY_PATH}}"
    local ret=0
    case "$1" in
    export_data)
        export_data
        ret=$?
        ;;
    import_data)
        import_data
        ret=$?
        ;;
    process)
        export_data && import_data
        ret=$?
        ;;
    post_process)
        post_process
        ret=$?
        ;;
    single)
        execute_single_task $2 $3
        ret=$?
        ;;
    *)
        log "usage:$0 process|post_process|[single app_name step]"
        ret=1
        ;;
    esac
    log "result=$ret"
    return ${ret}
}

main $*
exit $?
