#!/usr/bin/env python
# -*- coding:utf-8 -*-
"""
功 能：安装逻辑复制工具
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2021-07-30 14:51 创建
"""

import json
import os
import re
import shutil
import subprocess
import sys
import stat


class LogicrepInstall(object):
    """逻辑复制工具安装执行类"""

    def __init__(self, install_root_path):
        self.logicrep_root_path = install_root_path
        self.key1, self.key2 = self.get_generating_key()
        self.is_backup = None
        self.nce_name = None
        self.db_info = {}

        self.data_source = dict(
            ds_type="ds.type=gauss",
            ds_url_srcdb="ds.url=jdbc:zenith",
            ds_username="ds.username",
            ds_passwd="ds.passwd",
            dk_url_kafka="ds.url",
            dk_srcurity_protocol="security.protocol",
            dk_truststore_location="ssl.truststore.location",
            dk_truststore_password="ssl.truststore.password",
            dk_keystore_location="ssl.keystore.location",
            dk_keystore_password="ssl.keystore.password",
            dk_key_password="ssl.key.password",
            dk_batch_size="batch.size",
            dk_max_request_size="max.request.size",
            dk_optimize_batch_send_buffer="optimize.batch.send.buffer")

    @staticmethod
    def save_port(db_port, _path="/opt/software"):
        """
        保存数据库端口号

        @param db_port: db_port
        @param _path: _path
        """
        port_path = os.path.join(_path, "port")
        with open(port_path, "w", encoding="utf-8") as port_file:
            port_file.write(str(db_port))

    def get_generating_key(self):
        """
        生成密码密文，同时保存两个密码子

        @return: key work_key
        """
        try:
            key, work_key = None, None
            command = "su - dbuser -c 'zencrypt -g'"
            _, res = subprocess.getstatusoutput(command)

            for line in res.split("\n"):
                if line.startswith("Key"):
                    key = line.split(":", 1)[1].strip()
                    continue
                if line.startswith("WorkKey"):
                    work_key = line.split(":", 1)[1].strip()
            if not key or not work_key:
                raise Exception("获取key、workKey失败")

            flags = os.O_WRONLY
            modes = stat.S_IWUSR | stat.S_IRUSR
            key1 = os.path.join(self.logicrep_root_path, "logicrep/conf/sec/", "key1.properties")
            key2 = os.path.join(self.logicrep_root_path, "logicrep/conf/sec/", "key2.properties")
            with os.fdopen(os.open(key1, flags, modes), 'w') as key_file_one, \
                    os.fdopen(os.open(key2, flags, modes), 'w') as key_file_two:
                key1_data = "factory.key=%s" % key
                key_file_one.write(key1_data)
                key2_data = "local.key=%s" % work_key
                key_file_two.write(key2_data)
            return key, work_key
        except Exception as _e:
            raise Exception("生成密码因子失败") from _e

    def get_db_dcnnwcommondb(self, containerlist_path):
        """
        get db dcnnwcommondb name

        @param containerlist_path: containerlist_path
        @return: dcnnwcommondb_name
        """
        try:
            with open(containerlist_path, "r", encoding="utf-8") as db_file:
                data = json.load(db_file)
            dcnnwcommondb_name = "dcnnwcommondb-0-"
            if self.is_backup == "backup":
                dcnnwcommondb_name = "dcnnwcommondb-1-"
            for key in data.keys():
                if dcnnwcommondb_name in key:
                    return key
            raise Exception("获取数据库实例失败")
        except Exception as _e:
            raise Exception("获取数据库实例失败") from _e

    def get_db_info_dtagent(self, service_path):
        """
        获取数据库端口和已解密数据库密钥

        @param service_path: DTAgent
        @return: db_port and decrypt cipher
        """
        try:
            rtsp_list = os.listdir(service_path)
            dt_agent_service_path = None
            for rtsp_file in rtsp_list:
                if "DTAgentService" in rtsp_file and rtsp_file.endswith(".json"):
                    dt_agent_service_path = os.path.join(service_path, rtsp_file)
                    break
            if not dt_agent_service_path:
                raise Exception("获取DTAgentService数据库加密文件失败")
            with open(dt_agent_service_path, "r", encoding="utf-8") as db_file:
                data = json.load(db_file)
            encryption_key = data.get("databases").get(self.db_info.get("name"))[0].get("passwd")
            db_port = data.get("databases").get(self.db_info.get("name"))[0].get("port")
            return db_port, self.decrypt_cipher(encryption_key)
        except Exception as _e:
            raise Exception("获取数据库加密密钥失败") from _e

    def get_db_ip(self, containerlist_path):
        """
        获取数据库ip，微服务下数据库文件中数据库ip地址为127.0.0.1,不可用

        @param containerlist_path: containerlist_path
        @return: db_ip
        """
        command = "netstat -anp | grep :%s | grep LISTEN | " \
                  "grep -v 127.0.0.1" % self.db_info.get("port")
        _, res = subprocess.getstatusoutput(command.encode('utf-8'))
        ipv4_pattern = r"\b(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}" \
                       r"(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b"
        db_ip = re.findall(ipv4_pattern, res)
        if "0.0.0.0" in db_ip:
            db_ip.remove("0.0.0.0")
        if db_ip:
            return db_ip[0]

        with open(containerlist_path, "r", encoding="utf-8") as db_file:
            data = json.load(db_file)
            if not isinstance(data, dict):
                raise Exception("读取containerlist.json失败")
            db_ip = data.get(self.db_info.get("dcnnwcommondb")).get("ip")
            return db_ip

    def get_db_port(self, containerlist_path):
        """
        获取数据库端口

        @param containerlist_path: containerlist_path
        @return: decrypt cipher
        """
        with open(containerlist_path, "r", encoding="utf-8") as db_file:
            data = json.load(db_file)
            if not isinstance(data, dict):
                raise Exception("读取containerlist.json失败")
            db_port = data.get(self.db_info.get("dcnnwcommondb")).get("port")
            self.save_port(db_port)
            return db_port

    @staticmethod
    def _get_jre_path(rtsp_path="/opt/oss/rtsp"):
        """
        获取java依赖JRE的路径

        @param rtsp_path: rtsp path
        @return: jre path
        """
        rtsp_list = os.listdir(rtsp_path)
        for item in rtsp_list:
            if "jre-" in item:
                jre_path = os.path.join(rtsp_path, item)
                if os.path.isdir(jre_path):
                    return jre_path
        raise Exception("获取JRE路径失败")

    def decrypt_cipher(self, cipher):
        """
        调用ossext方法解密数据库的加密密钥

        @param cipher: 待解密的数据库密钥
        @return: 解密后密钥
        """
        try:
            command = "mkdir -p /opt/install_logic/cipher;" \
                      "cp -rf /opt/oss/%s/etc/cipher/* /opt/install_logic/cipher;" \
                      "chown -hR ossadm: /opt/install_logic/;" \
                      "chmod 700 /opt/zenith/app/admin/plugin/logicrep/ctrl" % self.nce_name
            subprocess.getstatusoutput(command)

            command = '''su - ossadm -c ". /opt/oss/manager/bin/engr_profile.sh;
            export CIPHER_ROOT=/opt/install_logic/cipher;
            python -c \\"from util import ossext;print(ossext.Cipher.decrypt(\'%s\'))\\""''' \
                      % cipher
            _, _decrypt_cipher = subprocess.getstatusoutput(command)

            if "Last login" in _decrypt_cipher or not _decrypt_cipher:
                raise Exception("密钥解密失败: " + cipher)
            return _decrypt_cipher
        except Exception as _e:
            raise Exception("密钥解密失败") from _e

    def generating_aes256_cipher(self, cipher):
        """
        生成aes256密钥，需要数据库密码

        @param cipher: cipher
        @return: aes256_cipher
        """
        try:
            # 每条语句必须独立成行，故首位需加换行符
            commands = ["\nzencrypt -e AES256 -f %s -k %s" % (self.key1, self.key2),
                        cipher, cipher + "\n"]
            command = "\n".join(commands)
            subp = subprocess.Popen("su - dbuser".split(), stdin=subprocess.PIPE,
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE)
            res, _ = subp.communicate(command.encode('utf-8'), timeout=10)
            aes256_cipher = None
            subp.kill()
            res = res.decode('utf-8')
            for line in res.split("\n"):
                if "Cipher:" in line:
                    aes256_cipher = line.split(":")[1].strip()
            return aes256_cipher
        except Exception as _e:
            raise Exception("数据库密钥解密失败") from _e

    def modify_init_configuration(self):
        """修改init文件配置"""
        init_file = os.path.join(self.logicrep_root_path, "logicrep/conf/", "init.properties")
        temp_file = os.path.join(self.logicrep_root_path, "logicrep/conf/", "temp")
        replayer_class_jdbc = \
            "replayer.class=com.huawei.gauss.logicrep.replayer.jdbc.JdbcReplayer"
        replayer_class_kafka = \
            "replayer.class=com.huawei.gauss.logicrep.plugin.replayer.kafka.KafkaReplayer"
        skip_missing_file = "skip.missing.file=false"
        new_skip_missing_file = "skip.missing.file=true"
        skip_table_without_metadata = "skip.table.without.metadata=false"
        new_skip_table_without_metadata = "skip.table.without.metadata=true"

        flags = os.O_WRONLY | os.O_CREAT
        modes = stat.S_IWUSR | stat.S_IRUSR
        with open(init_file, "r", encoding="utf-8") as file_init, \
                os.fdopen(os.open(temp_file, flags, modes), 'w', encoding="utf-8") as file_tmp:
            data = file_init.read()
            new_replayer_class_jdbc = "#%s" % replayer_class_jdbc
            old_replayer_class_kafka = "#%s" % replayer_class_kafka
            if new_replayer_class_jdbc in data:
                new_replayer_class_jdbc = replayer_class_jdbc
            new_data = data.replace(replayer_class_jdbc, new_replayer_class_jdbc) \
                .replace(old_replayer_class_kafka, replayer_class_kafka) \
                .replace(skip_missing_file, new_skip_missing_file) \
                .replace(skip_table_without_metadata, new_skip_table_without_metadata)
            file_tmp.write(new_data)
        if (new_replayer_class_jdbc not in new_data) or (replayer_class_kafka not in new_data):
            os.remove(temp_file)
            raise Exception("init.properties修改失败")
        os.remove(init_file)
        os.rename(temp_file, init_file)

    def modify_startup(self):
        """修改startup文件配置"""
        init_file = os.path.join(self.logicrep_root_path, "logicrep/", "startup.sh")
        temp_file = os.path.join(self.logicrep_root_path, "logicrep/", "temp_startup")
        JAVA_OPTS = 'JAVA_OPTS="'
        new_JAVA_OPTS = 'JAVA_OPTS="-DENCRYPTED_SSL_PASSWORD=false '
        if new_JAVA_OPTS in init_file:
            return
        flags = os.O_WRONLY | os.O_CREAT
        modes = stat.S_IWUSR | stat.S_IRUSR
        with open(init_file, "r", encoding="utf-8") as file_init, \
                os.fdopen(os.open(temp_file, flags, modes), 'w', encoding="utf-8") as file_tmp:
            data = file_init.read()
            new_data = data.replace(JAVA_OPTS, new_JAVA_OPTS)
            file_tmp.write(new_data)
        if new_JAVA_OPTS not in new_data:
            os.remove(temp_file)
            raise Exception("startup修改失败")
        os.remove(init_file)
        os.rename(temp_file, init_file)

    @staticmethod
    def _get_modify_line(line, source, purpose):
        # ds_url重复不易区分，又ds_url_srcdb < dk_url_kafka
        if line.startswith(source.get("ds_url_srcdb")):
            return purpose.get("ds_url_srcdb")
        for key, val in source.items():
            if line.startswith(val):
                return purpose.get(key)
        return None

    def _get_modify_data(self, data_list, source, purpose):
        res_list = []
        for line in data_list:
            if self._get_modify_line(line, source, purpose):
                res_list.append(self._get_modify_line(line, source, purpose))
            else:
                res_list.append(line.rstrip())
        return os.linesep.join(res_list)

    @staticmethod
    def _get_delete_dstdb(source_data):
        data_list = []
        need_delete = False
        for line in source_data:
            if line.startswith("[dstdb]"):
                need_delete = True
            elif line.startswith("["):
                need_delete = False
            if not need_delete:
                data_list.append(line)
        return data_list

    def _get_datasource_purpose(self, srcdb_ip, kafka_ip, aes256_cipher):
        """
        获取datasource目标数据

        @param srcdb_ip: srcdb_ip
        @param kafka_ip: kafka_ip
        @param aes256_cipher: aes256_cipher
        @return: purpose
        """
        manifest_file_path = os.path.join(
            "/opt/oss/%s/apps/DTAgentService/etc/ssl/internal" % self.nce_name, "manifest.json")

        with open(manifest_file_path, "r", encoding="utf-8") as manifest_f:
            manifest_data = json.load(manifest_f)
            manifest_f.close()
        server_p12 = manifest_data.get("filelist").get("server.p12").get("keyPass")
        trust_jks = manifest_data.get("filelist").get("trust.jks").get("storePass")
        server_p12_cipher = self.generating_aes256_cipher(self.decrypt_cipher(server_p12))
        trust_jks_cipher = self.generating_aes256_cipher(self.decrypt_cipher(trust_jks))
        truststore_location = os.path.join(self.logicrep_root_path, "logicrep/internal/trust.jks")
        keystore_location = os.path.join(self.logicrep_root_path, "logicrep/internal/server.p12")
        srcdb_port = self.db_info.get("port")
        data_purpose = dict(
            ds_type="ds.type=zenith",
            ds_url_srcdb="ds.url=jdbc:zenith:@%s:%s?useSSL=true" % (srcdb_ip, srcdb_port),
            ds_username="ds.username=%s" % self.db_info.get("name"),
            ds_passwd="ds.passwd=%s" % aes256_cipher,
            dk_url_kafka="ds.url=%s:9094" % kafka_ip,
            dk_srcurity_protocol="security.protocol=SSL",
            dk_truststore_location="ssl.truststore.location=%s" % truststore_location,
            dk_truststore_password="ssl.truststore.password=%s" % trust_jks_cipher,
            dk_keystore_location="ssl.keystore.location=%s" % keystore_location,
            dk_keystore_password="ssl.keystore.password=%s" % server_p12_cipher,
            dk_key_password="ssl.key.password=%s" % server_p12_cipher,
            dk_batch_size="batch.size=16384",
            dk_max_request_size="max.request.size=1048576",
            dk_optimize_batch_send_buffer="optimize.batch.send.buffer=1048576")
        return data_purpose

    def modify_datasource_configuration(self, srcdb_ip, kafka_ip, aes256_cipher):
        """
        修改 datasource 文件配置

        @param srcdb_ip: srcdb_ip
        @param kafka_ip: kafka_ip
        @param aes256_cipher: aes256_cipher
        """
        datasource_file_path = os.path.join(
            self.logicrep_root_path, "logicrep/conf/", "datasource.properties")

        data_purpose = self._get_datasource_purpose(srcdb_ip, kafka_ip, aes256_cipher)

        flags = os.O_WRONLY | os.O_CREAT
        modes = stat.S_IWUSR | stat.S_IRUSR
        with open(datasource_file_path, "r", encoding="utf-8") as file_source:
            data_list = self._get_delete_dstdb(file_source.readlines())
            file_source.close()
        os.remove(datasource_file_path)
        with os.fdopen(os.open(datasource_file_path, flags, modes), 'w', encoding="utf-8") as file_purpose:
            file_purpose.write(self._get_modify_data(data_list, self.data_source, data_purpose))

    def modify_repconf_db_xml(self):
        """修改 repconf 文件配置"""
        repconf_file_path = os.path.join(self.logicrep_root_path, "logicrep/conf/repconf",
                                         "repconf_db.xml")

        datasource_info = "		<datasourceInfo srcName=\"srcdb\" dstName=\"dstdb\"/>"
        new_datasource_info = "		<datasourceInfo srcName=\"srcdb\" dstName=\"dstkafka\"/>"

        user_info = "		 <userInfo userName=\"lrep\"/>"
        new_user_info = "		 <userInfo userName=\"lrep\"/>：\n" \
                        "		 <!--<userInfo userName=\"%s\"/>-->" % self.db_info.get("name")
        with open(repconf_file_path, "r", encoding="utf-8") as repconf_file:
            repconf_data = repconf_file.read()
            if new_datasource_info not in repconf_data:
                repconf_data = repconf_data. \
                    replace(datasource_info, new_datasource_info). \
                    replace(user_info, new_user_info)

            # 删除modelMapping的内容
            repconf_data_list = repconf_data.split("\n")
            new_repconf_data_list = []
            need_delete = False
            for line in repconf_data_list:
                if "<modelMapping>" in line:
                    need_delete = True
                    # 保留标签行
                    new_repconf_data_list.append(line)
                elif "</modelMapping>" in line:
                    need_delete = False
                if not need_delete:
                    new_repconf_data_list.append(line)
            new_data = "\n".join(new_repconf_data_list)

        flags = os.O_WRONLY | os.O_CREAT
        modes = stat.S_IWUSR | stat.S_IRUSR
        os.remove(repconf_file_path)
        with os.fdopen(os.open(repconf_file_path, flags, modes), 'w', encoding="utf-8") as repconf_file_write:
            repconf_file_write.write(new_data)

    def modify_topic_table(self, dtagent_topic_path):
        """
        从微服务取逻辑复制数据库映射

        @param dtagent_topic_path: dtagent topic path
        """
        logicrep_cert_path = os.path.join(self.logicrep_root_path,
                                          "logicrep/conf/topicconf/topic_table.properties")
        if os.path.exists(logicrep_cert_path):
            os.remove(logicrep_cert_path)
        shutil.copy(dtagent_topic_path, logicrep_cert_path)

    def copy_cert_from_dtagent(self, dtagent_cert_path):
        """
        从DTAgent复制证书

        @param dtagent_cert_path: dtagent_cert_path
        """
        logicrep_cert_path = os.path.join(self.logicrep_root_path, "logicrep/internal")
        if os.path.exists(logicrep_cert_path):
            shutil.rmtree(logicrep_cert_path)
        shutil.copytree(dtagent_cert_path, logicrep_cert_path)

    def start_logicrep(self, cipher):
        """
        通过看门狗启动逻辑复制工具

        @param cipher: DB cipher
        """
        logicrep_path = os.path.join(self.logicrep_root_path, "logicrep")
        command = "chown -hR dbuser:  %s" % logicrep_path
        subprocess.getstatusoutput(command)

        if self.is_backup == "backup":
            valid_sql = "ALTER DATABASE ENABLE_LOGIC_REPLICATION ON"
            param_dict = {"db_name": self.db_info.get("name"), "db_pass": cipher,
                          "_ip": self.db_info.get("ip"), "port": self.db_info.get("port"),
                          "valid_sql": valid_sql}
            command = '''sudo -s -u dbuser<<EOF
source /home/dbuser/.bashrc
/opt/zenith/app/bin/zsql %(db_name)s/%(db_pass)s@%(_ip)s:%(port)s -c "%(valid_sql)s"
EOF''' % param_dict
            subprocess.getstatusoutput(command)

        jre_path = self._get_jre_path()
        commands = ["\ncd %s" % logicrep_path,
                    "export JAVA_HOME=%s" % jre_path,
                    "export OSS_ROOT=/opt/oss/%s" % self.nce_name,
                    "sh watchdog_logicrep.sh -n logicrep &" + "\n\n\n"]
        command = "\n".join(commands)
        subp = subprocess.Popen("su - dbuser".split(), stdin=subprocess.PIPE)
        subp.communicate(command.encode('utf-8'), timeout=10)

    @staticmethod
    def get_nce_name():
        """
        获取nce文件名

        @return: nce name
        """
        nce = "NCE"
        ncecommone = "NCECOMMONE"
        nce_path = "/opt/oss/%s/apps/DTAgentService" % nce
        ncecommone_path = "/opt/oss/%s/apps/DTAgentService" % ncecommone
        for item in ((nce_path, nce), (ncecommone_path, ncecommone)):
            if os.path.exists(item[0]):
                return item[1]
        return ncecommone

    def run(self, is_backup=None, db_name="dcnlogicrepdb"):
        """
        run

        @param is_backup: is_backup
        @param db_name: db_name
        """
        self.is_backup = is_backup
        self.nce_name = self.get_nce_name()
        containerlist_path = "/opt/oss/manager/var/tenants/%s/containerlist.json" % self.nce_name
        dtagent_cert_path = \
            "/opt/oss/%s/apps/DTAgentService/etc/ssl/internal" % self.nce_name
        dtagent_topic_path = \
            "/opt/oss/%s/apps/DTAgentService/etc/conf/topic_table.properties" % self.nce_name
        dtagent_sysconf_path = "/opt/oss/%s/apps/DTAgentService/etc/sysconf/" % self.nce_name

        # 多节点数据库ip是南向ip
        self.db_info["name"] = db_name
        self.db_info["dcnnwcommondb"] = self.get_db_dcnnwcommondb(containerlist_path)

        _, cipher = self.get_db_info_dtagent(dtagent_sysconf_path)
        self.db_info["port"] = self.get_db_port(containerlist_path)
        # 数据库ip暂用127.0.0.1
        self.db_info["ip"] = self.get_db_ip(containerlist_path)
        aes256_cipher = self.generating_aes256_cipher(cipher)

        self.copy_cert_from_dtagent(dtagent_cert_path)
        self.modify_init_configuration()
        self.modify_datasource_configuration(
            self.db_info.get("ip"), self.db_info.get("ip"), aes256_cipher)
        self.modify_repconf_db_xml()
        self.modify_topic_table(dtagent_topic_path)
        self.modify_startup()
        self.start_logicrep(cipher)


if __name__ == '__main__':
    LOGICREP_ROOT_PATH = "/opt/zenith/app/admin/plugin"
    INSTALL = LogicrepInstall(LOGICREP_ROOT_PATH)
    INSTALL.run(sys.argv[1])
