#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: enter_account.sh
# Description: Enter account information.
###############################################################################


##################################################
# Description:  Initialize environment.
# Parameters:   null
##################################################
function init_env() {
    local workspace="$(dirname $(readlink -f $0))"

    source "${workspace}"/../../public_definition/public_definition.sh
    source "${workspace}"/../common_definition/common_definition.sh

    check_execution_user || {
    echo "Error, The current execution user is wrong, please execute to the correct user."; return 1; }

    export LOGFILE="${ACCOUNT_LOG}"
    echo -e "\n========================================\n" >> "${LOGFILE}"
    chmod 600 "${LOGFILE}"

    log info "Start initializing environment...." --nostd
}

##################################################
# Description:  Copy conf to tmp.
# Parameters:   null
##################################################
function copy_conf_to_tmp() {
    log info "Start copying configuration files to temporary files...." --nostd

    [ -s "${ACCOUNT_IP_CONF}" ] && cat "${ACCOUNT_IP_CONF}" > "${account_ip_conf_temp}" || touch "${account_ip_conf_temp}"
    [ -s "${ACCOUNT_USER_CONF}" ] && cat "${ACCOUNT_USER_CONF}" > "${account_user_conf_temp}" || touch "${account_user_conf_temp}"
    [ -s "${ACCOUNT_PASSWD_CONF}" ] && cat "${ACCOUNT_PASSWD_CONF}" > "${account_passwd_conf_temp}" || touch "${account_passwd_conf_temp}"

    for tmp_file in "${account_ip_conf_temp}" "${account_user_conf_temp}" "${account_passwd_conf_temp}"; do
        if [ -f "${tmp_file}" ]; then
            chmod 600 "${tmp_file}"
        fi
    done
}

##################################################
# Description:  Copy tmp to conf.
# Parameters:   null
##################################################
function copy_tmp_to_conf() {
    log info "Start copying temporary files to configuration files...." --nostd

    [ -s "${account_ip_conf_temp}" ] && cat "${account_ip_conf_temp}" > "${ACCOUNT_IP_CONF}"
    [ -s "${account_user_conf_temp}" ] && cat "${account_user_conf_temp}" > "${ACCOUNT_USER_CONF}"
    [ -s "${account_passwd_conf_temp}" ] && cat "${account_passwd_conf_temp}" > "${ACCOUNT_PASSWD_CONF}"

    for conf_file in "${ACCOUNT_IP_CONF}" "${ACCOUNT_USER_CONF}" "${ACCOUNT_PASSWD_CONF}"; do
        if [ -f "${conf_file}" ]; then
            chmod 600 "${conf_file}"
        fi
    done
}

##################################################
# Description:  Delete tmp files.
# Parameters:   null
##################################################
function delete_tmp_files() {
    log info "Start deleting temporary files...." --nostd

    [ -f "${account_ip_conf_temp}" ] && rm -f "${account_ip_conf_temp}" || true
    [ -f "${account_user_conf_temp}" ] && rm -f "${account_user_conf_temp}" || true
    [ -f "${account_passwd_conf_temp}" ] && rm -f "${account_passwd_conf_temp}" || true
}

##################################################
# Description:  Check ip limit.
# Parameters:   null
##################################################
function check_ip_limit() {
    log info "Start checking ip limit...." --nostd

    local ip_num=$(cat "${account_ip_conf_temp}" 2>/dev/null |wc -l)
    if [ "${ip_num}" -ge 1 ]; then
        echo "Warning, The ip address has reached the upper limit."
        select_account_action
        exit $?
    fi
}

##################################################
# Description:  Create login ip.
# Parameters:   null
##################################################
function create_login_ip() {
    log info "Start creating login ip...." --nostd

    while true
    do
        echo -e "\nPlease enter the ip address of the login node:"
        read login_ip

        check_ip_format "${login_ip}" || { echo "Warning, Please enter a valid ip address." && continue; }
        break
    done

    while true
    do
        echo -e "\nPlease enter the ip address of the backup node:"
        read backup_ip

        check_ip_format "${backup_ip}" || { echo "Warning, Please enter a valid ip address." && continue; }
        break
    done

    local ip_list=("${login_ip}" "${backup_ip}")
    for ip in ${ip_list[@]}
    do
        if grep -qw "^${ip}" "${account_ip_conf_temp}" 2>/dev/null; then
            if grep -qw "^${ip}" "${account_user_conf_temp}" 2>/dev/null; then
                echo "Warning, The ip address ${ip} already exists."
                select_account_action
                exit $?
            fi
        fi
    done

    grep -qw "^${login_ip} " "${account_ip_conf_temp}" 2>/dev/null && sed -i "/^${login_ip} /d" "${account_ip_conf_temp}"
    grep -qw " ${backup_ip}" "${account_ip_conf_temp}" 2>/dev/null && sed -i "/ ${backup_ip}/d" "${account_ip_conf_temp}"
    echo "${login_ip} ${backup_ip}" >> "${account_ip_conf_temp}"
}

##################################################
# Description:  Create login account.
# Parameters:   null
##################################################
function create_login_account() {
    log info "Start creating login account...." --nostd

    local type="login"
    local ip="$1"

    if [ -z "${ip}" ]; then
        echo "Error, The ${type} ip: No such string."
        return 1
    fi

    while true
    do
        echo -e "\nPlease enter the login user of the login node (default=sopuser):"
        read login_user

        [ -z "${login_user}" ] && login_user="sopuser"
        check_account_user "${login_user}" || { echo "Warning, Please enter a valid user." && continue; }
        break
    done

    while true
    do
        echo "Please enter the password of the ${login_user} user"
        read -p "Password:" -rs login_passwd
        read -p $'\nRetype password:' -rs login_passwd_confirm
        echo -e "\n"

        [ "${login_passwd}" = "${login_passwd_confirm}" ] || {
        echo "Warning, The passwords entered twice do not match, Please re-enter." && continue; }

        break
    done

    grep -qw "^${ip} .*${type}" "${account_user_conf_temp}" 2>/dev/null && sed -i "/^${ip} .*${type}/d" "${account_user_conf_temp}"
    echo "${ip} ${login_user} "null" ${type}" >> "${account_user_conf_temp}" || {
    echo "Error, Failed to store login account user."; return 1; }

    echo "${INPUT_TYPE_ENCRYPT}" "${INPUT_PASSWD_LOGIN}=${login_passwd}" |"${BIN_PYTHON}" "${DATA_CRYPT_PYC}" || {
    echo "Error, Failed to store login account password."; return 1; }
}

##################################################
# Description:  Create execute account.
# Parameters:   null
##################################################
function create_execute_account() {
    log info "Start creating execute account...." --nostd

    local type="execute"
    local ip="$1"

    if [ -z "${ip}" ]; then
        echo "Error, The ${type} ip: No such string."
        return 1
    fi

    while true
    do
        echo -e "\nPlease enter the execute user of the login node (default=ossadm):"
        read execute_user

        [ -z "${execute_user}" ] && execute_user="ossadm"
        check_account_user "${execute_user}" || { echo "Warning, Please enter a valid user." && continue; }
        break
    done

    while true
    do
        echo "Please enter the password of the ${execute_user} user:"
        read -p "Password:" -rs execute_passwd
        read -p $'\nRetype password:' -rs execute_passwd_confirm
        echo -e "\n"

        [ "${execute_passwd}" = "${execute_passwd_confirm}" ] || {
        echo "Warning, The passwords entered twice do not match, Please re-enter." && continue; }

        break
    done

    grep -qw "^${ip} .*${type}" "${account_user_conf_temp}" 2>/dev/null && sed -i "/^${ip} .*${type}/d" "${account_user_conf_temp}"
    echo "${ip} ${execute_user} "null" ${type}" >> "${account_user_conf_temp}" || {
    echo "Error, Failed to store execute account user."; return 1; }

    echo "${INPUT_TYPE_ENCRYPT}" "${INPUT_PASSWD_EXECUTE}=${execute_passwd}" |"${BIN_PYTHON}" "${DATA_CRYPT_PYC}" || {
    echo "Error, Failed to store execute account password."; return 1; }
}

##################################################
# Description:  Create sftp account.
# Parameters:   null
##################################################
function create_sftp_account() {
    log info "Start creating sftp accounts...." --nostd

    local type="sftp"
    local ip="$1"

    if [ -z "${ip}" ]; then
        echo "Error, The ${type} ip: No such string."
        return 1
    fi

    while true
    do
        echo -e "\nPlease enter the sftp user of the backup node (default=ftpuser):"
        read sftp_user

        [ -z "${sftp_user}" ] && sftp_user="ftpuser"
        check_account_user "${sftp_user}" || { echo "Warning, Please enter a valid user." && continue; }
        break
    done

    while true
    do
        echo "Please enter ${sftp_user} user's backup directory (default=backup):"
        read sftp_backup_path

        [ -z "${sftp_backup_path}" ] && sftp_backup_path="backup"
        # Remove the path with ..
        sftp_backup_path=$(sed 's#\.\.##g' <<< ${sftp_backup_path})
        # Path must not start with /
        sftp_backup_path=$(sed 's#\(^/\+\)\(.*\)#\2#g' <<< "${sftp_backup_path}")
        break
    done

    while true
    do
        echo "Please enter the password of the ${sftp_user} user"
        read -p "Password:" -rs sftp_passwd
        read -p $'\nRetype password:' -rs sftp_passwd_confirm
        echo -e "\n"

        [ "${sftp_passwd}" = "${sftp_passwd_confirm}" ] || {
        echo "Warning, The passwords entered twice do not match, Please re-enter." && continue; }

        break
    done

    grep -qw "^${ip} .*${type}" "${account_user_conf_temp}" 2>/dev/null && sed -i "/^${ip} .*${type}/d" "${account_user_conf_temp}"
    echo "${ip} ${sftp_user} ${sftp_backup_path} ${type}" >> "${account_user_conf_temp}" || {
    echo "Error, Failed to store sftp account user."; return 1; }

    echo "${INPUT_TYPE_ENCRYPT}" "${INPUT_PASSWD_SFTP}=${sftp_passwd}" |"${BIN_PYTHON}" "${DATA_CRYPT_PYC}" || {
    echo "Error, Failed to store sftp account password."; return 1; }
}

##################################################
# Description:  Enter the recorded ip.
# Parameters:   null
##################################################
function select_recorded_ip() {
    log info "Start to select the recorded ip...." --nostd

    if [ ! -s "${account_ip_conf_temp}" ]; then
        echo "Warning, The ip address does not exist, please create ip first."
        select_account_action
        exit $?
    fi

    while true
    do
        echo -e "\n========================================"
        echo -e "Please select the ip address to be modified:"
        cat -n "${account_ip_conf_temp}"
        echo -e "========================================\n"
        read -p "Please enter the number:" number_login_ip

        local ip_num=$(cat "${account_ip_conf_temp}" |wc -l)
        egrep -qw '^[1-9][0-9]?$' 2>/dev/null <<< "${number_login_ip}" && [ "${number_login_ip}" -le "${ip_num}" ] || {
        echo "Warning, Please enter a valid number." && continue; }
        break
    done

    record_ip_list=($(sed -n "${number_login_ip}p" "${account_ip_conf_temp}"))
    record_login_ip="${record_ip_list[0]}"
    record_backup_ip="${record_ip_list[1]}"


    if [ -z "${record_login_ip}" -o -z "${record_backup_ip}" ]; then
        echo "Error, Selected address does not exist, please check the ip configuration file."
        return 1
    fi
}

##################################################
# Description:  Modify login ip.
# Parameters:   null
##################################################
function modify_login_ip() {
    log info "Start modifying login ip...." --nostd

    select_recorded_ip || return $?

    sed -i "/^${record_login_ip} /d" "${account_ip_conf_temp}"
    sed -i "/ ${record_backup_ip}/d" "${account_ip_conf_temp}"
    create_login_ip || return $?

    sed -i "s/^${record_login_ip} \(.* login\)/${login_ip} \1/g" "${account_user_conf_temp}" || true
    sed -i "s/^${record_login_ip} \(.* execute\)/${login_ip} \1/g" "${account_user_conf_temp}" || true
    sed -i "s/^${record_backup_ip} \(.* sftp\)/${backup_ip} \1/g" "${account_user_conf_temp}" || true
}

##################################################
# Description:  Modify user info.
# Parameters:   null
##################################################
function modify_user_info() {
    log info "Start modifying user information...." --nostd

    select_recorded_ip || return $?
    create_login_account "${record_login_ip}" || return $?
    create_execute_account "${record_login_ip}" || return $?
    create_sftp_account "${record_backup_ip}" || return $?
}

##################################################
# Description:  Create account information.
# Parameters:   null
##################################################
function create_account_information() {
    log info "Start creating account information...." --nostd

    check_ip_limit || return $?
    create_login_ip || return $?
    create_login_account "${login_ip}" || return $?
    create_execute_account "${login_ip}" || return $?
    create_sftp_account "${backup_ip}" || return $?
}

##################################################
# Description:  Modify account information.
# Parameters:   null
##################################################
function modify_account_information() {
    log info "Start modifying account information...." --nostd

    while true
    do
        echo -e "\n========================================"
        echo -e "Please select the item to be modified:"
        echo -e "1. Ip address"
        echo -e "2. User information"
        echo -e "========================================\n"
        read -p "Please enter the number:" number_modify

        grep -qw '[1-2]' <<< "${number_modify}" 2>/dev/null || { echo "Warning, Please enter a valid number." && continue; }
        break
    done

    if [ "${number_modify}" -eq 1 ]; then
        modify_login_ip || return $?
    elif [ "${number_modify}" -eq 2 ]; then
        modify_user_info || return $?
    else
        echo "Failed to select item to be modified."
        exit 1
    fi
}

##################################################
# Description:  Remove invalid accounts.
# Parameters:   null
##################################################
function remove_invalid_account() {
    log info "Start removing invalid account...." --nostd

    [ -s "${account_ip_conf_temp}" ] || return 0

    while read ip || [ -n "${ip}" ]
    do
        local user_num=$(egrep -w "^${ip/ /|}" "${account_user_conf_temp}" 2>/dev/null |wc -l)
        [ "${user_num}" -eq 3 ] || delete_recorded_account "${ip}"
    done < "${account_ip_conf_temp}"
}

##################################################
# Description:  Delete recorded account.
# Parameters:   login_ip
##################################################
function delete_recorded_account() {
    log info "Start deleting recorded account...." --nostd

    for ip in $@
    do
        grep -qw "^${ip}" "${account_ip_conf_temp}" 2>/dev/null && sed -i "/^${ip} /d" "${account_ip_conf_temp}"
        grep -qw "^${ip}" "${account_user_conf_temp}" 2>/dev/null && sed -i "/^${ip} /d" "${account_user_conf_temp}"
    done
}

##################################################
# Description:  Test node transmission.
# Parameters:   null
##################################################
function test_node_communication() {
    log info "Start checking configuration...." --nostd

    local login_ip="${login_ip}"
    local sftp_ip="${backup_ip}"
    local login_user="${login_user}"
    local login_passwd="${login_passwd}"
    local execute_user="${execute_user}"
    local execute_passwd="${execute_passwd}"
    local sftp_user="${sftp_user}"
    local sftp_passwd="${sftp_passwd}"
    local sftp_backup_path="${sftp_backup_path}"
    local allpasswd="passwd_login=${login_passwd} passwd_execute=${execute_passwd} passwd_sftp=${sftp_passwd}"

    [ -z "${login_ip}" ] && login_ip="${record_login_ip}"
    [ -z "${backup_ip}" ] && backup_ip="${record_backup_ip}"

    define_backup_dynamic_var "null" "${login_ip} ${backup_ip}"

    source "${MINI_BACKUP_ACCOUNT_INFO_PATH}"/read_account.sh "execute" "${allpasswd}" || {
    log error "Failed to read account information."; return 1; }
    source "${MINI_BACKUP_ACCOUNT_INFO_PATH}"/read_account.sh "sftp" "${allpasswd}" || {
    log error "Failed to read account information."; return 1; }

    local cmd="pwd"
    execute_expect_connection "sshsftp" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="pwd"
    execute_expect_connection "sftpcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    log info "Configuration check complete!"  --nostd
}

##################################################
# Description:  Select account action.
# Parameters:   null
##################################################
function select_account_action() {
    log info "Start selecting account action...." --nostd

    number_select=0
    number_modify=0

    while true
    do
        echo -e "\n========================================"
        echo -e "Please select an account action:"
        echo -e "1. Create account information"
        echo -e "2. Modify account information"
        echo -e "0. Exit"
        echo -e "========================================\n"
        read -p "Please enter the number:" number_select

        grep -qw '[0-2]' <<< "${number_select}" 2>/dev/null || { echo "Warning, Please enter a valid number." && continue; }
        break
    done

    copy_conf_to_tmp || return $?

    if [ "${number_select}" -eq 1 ]; then
        remove_invalid_account
        if ! create_account_information; then
            delete_recorded_account "${login_ip}" "${backup_ip}"
            return 1
        fi
    elif [ "${number_select}" -eq 2 ]; then
        modify_account_information || return $?
    elif  [ "${number_select}" -eq 0 ]; then
        copy_tmp_to_conf || return $?
        delete_tmp_files || return $?
        exit 0
    else
        echo "Failed to select account action."
        exit 1
    fi

    copy_tmp_to_conf || return $?
    delete_tmp_files || return $?

    if [ "${number_select}" -eq 1 -o "${number_modify}" -eq 2 ]; then
        echo "Checking configuration, please wait...."
        test_node_communication || {
        echo "Configuration check failed."; return 1; }
    fi

    echo "Account information entered successfully!"
}

##################################################
# Description:  main function.
# Parameters:   null
##################################################
function main_enter_account() {
    init_env "$@" || return $?

    log info "Start entering account...." --nostd

    local account_ip_conf_temp="${ACCOUNT_IP_CONF}.tmp"
    local account_user_conf_temp="${ACCOUNT_USER_CONF}.tmp"
    local account_passwd_conf_temp="${ACCOUNT_PASSWD_CONF}.tmp"

    select_account_action || {
    echo "Account information entry failed."; return 1; }
}

main_enter_account