#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: read_account.sh
# Description: Read account.
###############################################################################


##################################################
# Description:  Read the login account.
# Parameters:   $@
##################################################
function read_login_account() {
    log info "Start reading the login account...." --nostd

    [ -f "${ACCOUNT_USER_CONF}" ] || {
    log error "The account ip conf ${ACCOUNT_USER_CONF}: No such file." --nostd; return 1; }
    [ -f "${ACCOUNT_PASSWD_CONF}" ] || {
    log error "The account password conf ${ACCOUNT_PASSWD_CONF}: No such file." --nostd; return 1; }

    local user_list=($(grep -w "^${BACKUP_LOGIN_IP} .*login" "${ACCOUNT_USER_CONF}" 2>/dev/null |head -1))

    login_ip="${user_list[0]}"
    login_user="${user_list[1]}"
    login_passwd=$(egrep -ow "passwd_login=[^ ]+" <<< "$@" |awk -F '=' '{print $2}')

    [ -n "${login_ip}" ] || {
    log error "The login ip ${login_ip}: No such string." --nostd; return 1; }
    [ -n "${login_user}" ] || {
    log error "The login user ${login_user}: No such string." --nostd; return 1; }
    [ -n "${login_passwd}" ] || {
    log error "The login password of ${login_user}: No such string." --nostd; return 1; }
}

##################################################
# Description:  Read the execute account.
# Parameters:   $@
##################################################
function read_execute_account() {
    log info "Start reading the execute account...." --nostd

    read_login_account "$@" || return $?

    local user_list=($(grep -w "^${BACKUP_LOGIN_IP} .*execute" "${ACCOUNT_USER_CONF}" 2>/dev/null |head -1))

    login_ip="${user_list[0]}"
    execute_user="${user_list[1]}"
    execute_passwd=$(egrep -ow "passwd_execute=[^ ]+" <<< "$@" |awk -F '=' '{print $2}')

    [ -n "${execute_user}" ] || {
    log error "The execute user ${execute_user}: No such string." --nostd; return 1; }
    [ -n "${execute_passwd}" ] || {
    log error "The execute password of ${execute_user}: No such string." --nostd; return 1; }
}

##################################################
# Description:  Read the sftp account.
# Parameters:   $@
##################################################
function read_sftp_account() {
    log info "Start reading the sftp account...." --nostd

    [ -f "${ACCOUNT_USER_CONF}" ] || {
    log error "The account ip conf ${ACCOUNT_USER_CONF}: No such file." --nostd; return 1; }
    [ -f "${ACCOUNT_PASSWD_CONF}" ] || {
    log error "The account password conf ${ACCOUNT_PASSWD_CONF}: No such file." --nostd; return 1; }

    local user_list=($(grep -w "^${BACKUP_SFTP_IP} .*sftp" "${ACCOUNT_USER_CONF}" 2>/dev/null |head -1))

    sftp_ip="${user_list[0]}"
    sftp_user="${user_list[1]}"
    sftp_backup_path="${user_list[2]}"
    sftp_passwd=$(egrep -ow "passwd_sftp=[^ ]+" <<< "$@" |awk -F '=' '{print $2}')

    [ -n "${sftp_ip}" ] || {
    log error "The sftp ip ${sftp_ip}: No such string." --nostd; return 1; }
    [ -n "${sftp_user}" ] || {
    log error "The sftp user ${sftp_user}: No such string." --nostd; return 1; }
    [ -n "${sftp_passwd}" ] || {
    log error "The sftp password of ${sftp_user}: No such string." --nostd; return 1; }
    [ -n "${sftp_backup_path}" ] || {
    log error "The sftp backup path of ${sftp_user}: No such string." --nostd; return 1; }
}

##################################################
# Description:  main function.
# Parameters:   null
##################################################
function main_read_account() {
    source "${PUBLIC_DEFINITION_PATH}"/public_definition.sh
    source "${MINI_BACKUP_COMMON_DEFINITION_PATH}"/common_definition.sh

    log info "Start reading account...." --nostd

    local account_type="$1"

    check_account_type "${account_type}" || {
    log error "The account type ${account_type}: Type not allowed." --nostd; return 1; }

    [ "${account_type}" = "login" ] && { read_login_account "$@" || return $?; }
    [ "${account_type}" = "execute" ] && { read_execute_account "$@" || return $?; }
    [ "${account_type}" = "sftp" ] && { read_sftp_account "$@" || return $?; }

    return 0
}

main_read_account "$@"