#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: check_function.sh
# Description: Check related functions.
###############################################################################


##################################################
# Description:  Check execution user.
# Parameters:   null
##################################################
function check_execution_user() {
    local allow_user="ossadm"

    id -un |grep -qw "${allow_user}" || return $?
}

##################################################
# Description:  Check if the script is executing.
# Parameters:   null
##################################################
function check_execution_script() {
    local check_script=("${EXECUTE_START_BACKUP_SCRIPT}")

    for script in ${check_script[@]}
    do
        ps -ef |grep -v 'grep' |grep -qw "${script}" && return $?
    done

    return 1
}

##################################################
# Description:  Check ip format.
# Parameters:   ip
##################################################
function check_ip_format() {
    [[ $1 =~ ^([1-9][0-9]?|1[0-9][0-9]|2[0-4][0-9]|25[0-5])(\.([1-9]?[0-9]|1[0-9][0-9]?|2[0-4][0-9]|25[0-5])){3}$ ]] || \
    [[ $1 =~ ^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|\
([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|\
([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|\
[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|\
fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|\
1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|\
(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$ ]]
}

##################################################
# Description:  Check account user.
# Parameters:   user
##################################################
function check_account_user() {
    local user="$1"

    [ -n "${user}" ] || return $?

    if [[ "${user}" =~ "\$" ]] || [[ "${user}" =~ "\`" ]] || [[ "${user}" =~ "|" ]] || [[ "${user}" =~ " " ]] || \
        [[ "${user}" =~ ";" ]] || [[ "${user}" =~ "[" ]] || [[ "${user}" =~ "{" ]] || [[ "${user}" =~ "=" ]] || \
        [[ "${user}" =~ "&" ]] || [[ "${user}" =~ "+" ]] || [[ "${user}" =~ "]" ]] || [[ "${user}" =~ "}" ]] || \
        [[ "${user}" =~ "%" ]] || [[ "${user}" =~ "<" ]] || [[ "${user}" =~ ">" ]] || [[ "${user}" =~ "!" ]]|| \
        [[ "${user}" =~ "\n" ]] || [[ "${user}" =~ "\\" ]]; then
        return 1
    fi

    return 0
}

##################################################
# Description:  Check account password.
# Parameters:   password
##################################################
function check_account_passwd() {
    local passwd="$1"

    [ -n "${passwd}" ] || return $?

    cracklib-check <<< "${passwd}" |grep -q ': OK$' || return $?
}

##################################################
# Description:  Check backup data type.
# Parameters:   type
##################################################
function check_backup_data_type() {
    local allow_type_list=("${INPUT_TYPE_PRODUCTDATA}")
    local curr_type_list="$@"

    [ -n "${curr_type_list}" ] || return $?

    for curr_type in ${curr_type_list}
    do
        grep -qw "${curr_type}" <<< "${allow_type_list[@]}" || return $?
    done
}

##################################################
# Description:  Check expect transmission type.
# Parameters:   type
##################################################
function check_expect_transfer_type() {
    local allow_type_list=("ssh" "sftp")
    local curr_type="$1"

    [ -n "${curr_type}" ] || return $?

    for allow_type in ${allow_type_list[@]}
    do
        grep -qw "^${allow_type}" <<< "${curr_type}" && return $?
    done

    return 1
}

##################################################
# Description:  Check account type.
# Parameters:   type
##################################################
function check_account_type() {
    local allow_type_list=("login" "execute" "sftp")
    local curr_type="$1"

    [ -n "${curr_type}" ] || return $?

    for allow_type in ${allow_type_list[@]}
    do
        [ "${curr_type}" = "${allow_type}" ] && return $?
    done

    return 1
}

##################################################
# Description:  Check positive integer.
# Parameters:   integer
##################################################
function check_positive_integer() {
    local integer="$1"

    egrep -qw "^[0-9]+$" <<< "${integer}" || return $?
}

##################################################
# Description:  Check node connections.
# Parameters:   login_ip login_user login_passwd [execute_user execute_passwd]
##################################################
function check_node_connection_ssh() {
    local login_ip="$1"
    local login_user="$2"
    local login_passwd="$3"
    local execute_user="$4"
    local execute_passwd="$5"

    [ -n "${login_ip}" ] || return $?
    [ -n "${login_user}" ] || return $?

    if [ -z "${execute_user}" ]; then
        expect_connection_ssh --loginip "${login_ip}" --loginuser "${login_user}" --loginpwd "${login_passwd}" \
            --sshcmd "ls" --sshtimeout "5" &>> "${LOGFILE}" || return $?
    fi

    if [ -n "${execute_user}" ]; then
        expect_connection_ssh --loginip "${login_ip}" --loginuser "${login_user}" --loginpwd "${login_passwd}" \
            --executeuser "${execute_user}" --executepwd "${execute_passwd}" \
            --sshcmd "ls" --sshtimeout "5" &>> "${LOGFILE}" || return $?
    fi
}

##################################################
# Description:  Check sftp connections.
# Parameters:   login_ip sftp_user sftp_passwd
##################################################
function check_node_connection_sftp() {
    local sftp_ip="$1"
    local sftp_user="$2"
    local sftp_passwd="$3"

    [ -n "${sftp_ip}" ] || return $?
    [ -n "${sftp_user}" ] || return $?

    expect_connection_sftp --sftpip "${sftp_ip}" --sftpuser "${sftp_user}" --sftppwd "${sftp_passwd}" \
        --sftpcmd "ls" --sftptimeout "5" &>> "${LOGFILE}" || return $?
}