#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: expect_function.sh
# Description: Expect related functions.
###############################################################################


##################################################
# Description:  Check expect incoming option parameters.
# Parameters:   $@
##################################################
function check_expect_option_params() {
    local args=$(getopt -q -o "" -l sshopts:,sftpopts:,sshtimeout:,sftptimeout:,loginip:,sftpip:,loginuser:,loginpwd:,executeuser:,executepwd:,sftpuser:,sftppwd:,sshcmd:,sftpcmd:,  -- "$@")
    [ $? -eq 0 ] || {
    log error "The getopt option format exception."; return 1; }

    eval set -- "${args}"

    while true
    do
        case "$1" in
            --sshopts|--sftpopts)
                check_expect_transfer_type "$2" || {
                log error "The expect transfer type $2: Type not allowed."; return 1; }
                [ "$1" = "--sshopts" ] && sshopts="$2"
                [ "$1" = "--sftpopts" ] && sftpopts="$2"
                shift 2
                ;;
            --sshtimeout|--sftptimeout)
                if [ -n "$2" ]; then
                    check_positive_integer "$2" || {
                    log error "The expect timeout $2: Format error."; return 1; }
                fi
                [ "$1" = "--sshtimeout" ] && sshtimeout="$2"
                [ "$1" = "--sftptimeout" ] && sftptimeout="$2"
                shift 2
                ;;
            --loginip|--sftpip)
                check_ip_format "$2" || {
                log error "The expect ip $2: Format error."; return 1; }
                [ "$1" = "--loginip" ] && loginip="$2"
                [ "$1" = "--sftpip" ] && sftpip="$2"
                shift 2
                ;;
            --loginuser|--executeuser|--sftpuser)
                check_account_user "$2" || {
                log error "The expect user $2: User not allowed."; return 1; }
                [ "$1" = "--loginuser" ] && loginuser="$2"
                [ "$1" = "--executeuser" ] && executeuser="$2"
                [ "$1" = "--sftpuser" ] && sftpuser="$2"
                shift 2
                ;;
            --loginpwd|--executepwd|--sftppwd)
                [ "$1" = "--loginpwd" ] && loginpwd="$2"
                [ "$1" = "--executepwd" ] && executepwd="$2"
                [ "$1" = "--sftppwd" ] && sftppwd="$2"
                shift 2
                ;;
            --sshcmd|--sftpcmd)
                [ "$1" = "--sshcmd" ] && sshcmd="$2"
                [ "$1" = "--sftpcmd" ] && sftpcmd="$2"
                shift 2
                ;;
            --)
                shift
                break
                ;;
            *)
                log error "The getopt option identification exception."
                exit 1
                ;;
        esac
    done
}

##################################################
# Description:  expect-ssh connection.
# Parameters:   --loginip ip --loginuser user --loginpwd password \
#               --executeuser user --executepwd password --sshcmd command
#               --sftpuser user --sftppwd password --sftpcmd command
#Instructions:  expect_connection_ssh --loginip "10.10.10.10" --loginuser "sopuser" --loginpwd "mypasswd" \
#                   --executeuser "ossadm" --executepwd "mypasswd" --sshcmd "echo ok"
#               expect_connection_ssh --loginip "10.10.10.10" --loginuser "sopuser" --loginpwd "mypasswd" \
#                   --executeuser "ossadm" --executepwd "mypasswd" \
#                   --sftpip "10.10.10.10" --sftpuser "ftpuser" --sftppwd "mypasswd" --sftpcmd "ls"
##################################################
function expect_connection_ssh() {
    local sshopts=''
    local sftpopts=''
    local loginip=''
    local sftpip=''
    local loginuser=''
    local loginpwd=''
    local executeuser=''
    local executepwd=''
    local sftpuser=''
    local sftppwd=''
    local sshcmd=''
    local sftpcmd=''
    local sshtimeout=''
    local sftptimeout=''
    local fixtimeout=10

    check_expect_option_params "$@" || { log error "expect-ssh parameter error."; return 1; }

    [ -z "${sshopts}" ] && sshopts="ssh"
    [ -z "${sftpopts}" ] && sftpopts="sftp"
    [ -z "${sshtimeout}" ] && sshtimeout="${fixtimeout}"
    [ -z "${sftptimeout}" ] && sftptimeout="${fixtimeout}"
    [ -n "${loginpwd}" ] && loginpwd=$(echo "${loginpwd}" |sed 's/\\/\\\\/g' |sed 's/}/\\}/g' |sed 's/{/\\{/g' |sed 's/\[/\\[/g' |sed 's/\$/\\\$/g' |sed 's/`/\\`/g' |sed 's/\"/\\\"/g')
    [ -n "${executepwd}" ] && executepwd=$(echo "${executepwd}" |sed 's/\\/\\\\/g' |sed 's/}/\\}/g' |sed 's/{/\\{/g' |sed 's/\[/\\[/g' |sed 's/\$/\\\$/g' |sed 's/`/\\`/g' |sed 's/\"/\\\"/g')
    [ -n "${sftppwd}" ] && sftppwd=$(echo "${sftppwd}" |sed 's/\\/\\\\/g' |sed 's/}/\\}/g' |sed 's/{/\\{/g' |sed 's/\[/\\[/g' |sed 's/\$/\\\$/g' |sed 's/`/\\`/g' |sed 's/\"/\\\"/g')

    expect << END
        #exp_internal 1
        set timeout ${fixtimeout}

        spawn ${sshopts} ${loginuser}@${loginip}

        expect {
            "*assword:" { send -- "${loginpwd}\r" }
            "yes/no" { send "yes\r"; exp_continue }
            "*Permission denied*" { exit 11 }
            timeout { exit 201 }
        }

        if { "${executeuser}" != "" } {
            expect {
                -re ".*${loginuser}@.*\]" { send "su - ${executeuser}\r" }
                timeout { exit 202 }
            }

            expect {
                "*assword:" { send -- "${executepwd}\r" }
                "yes/no" { send "yes\r"; exp_continue }
                "*Permission denied*" { exit 12 }
                timeout { exit 203 }
            }
        }

        if { "${sshcmd}" != "" } {
            expect {
                -re ".*@.*\]" { send -- "${sshcmd}\r" }
                timeout { exit 204 }
            }

            set timeout ${sshtimeout}
        }

        if { "${sftpuser}" != "" } {
            expect {
                -re ".*@.*\]" { send "${sftpopts} ${sftpuser}@${sftpip}\r" }
                timeout { exit 205 }
            }

            expect {
                "*assword:" { send -- "${sftppwd}\r" }
                "yes/no" { send "yes\r"; exp_continue }
                "*Permission denied*" { exit 13 }
                timeout { exit 206 }
            }
        }

       if { "${sftpcmd}" != "" } {
            expect {
                "\nsftp>*" { send -- "${sftpcmd}\r" }
                timeout { exit 207 }
            }

            set timeout ${sftptimeout}

            expect {
                "\nInvalid command." { exit 101 }
                "\n*not found." { exit  102 }
                "\nsftp>*" { send "exit\r" }
                timeout { exit 208 }
            }

            set timeout ${fixtimeout}
       }

       if { "${executeuser}" != "" } {
            expect {
                -re ".*${executeuser}@.*\]" { send "exit\r" }
                timeout { exit 209 }
            }
       }

        expect {
            -re ".*${loginuser}@.*\]" { send "exit\r" }
            timeout { exit 210 }
        }

        expect {
            timeout { exit 255 }
            eof { send_user "====End Of Expect SSH====\n" }
        }

        catch wait result
        exit [lindex \$result 3]
END

    return $?
}

##################################################
# Description:  expect-sftp connection.
# Parameters:   --sftpip ip --sftpuser user --sftppwd password --sftpcmd command
#Instructions:  expect_connection_sftp --sftpip "10.10.10.10" --sftpuser "sftpuser" --sftppwd "mypasswd" \
#                   --sftpcmd "ls"
##################################################
function expect_connection_sftp() {
    local sftpopts=''
    local sftpip=''
    local sftpuser=''
    local sftppwd=''
    local sftpcmd=''
    local sftptimeout=''
    local fixtimeout=10

    check_expect_option_params "$@" || { log error "expect-sftp parameter error."; return 1; }

    [ -z "${sftpopts}" ] && sftpopts="sftp"
    [ -z "${sftptimeout}" ] && sftptimeout="${fixtimeout}"
    [ -n "${sftppwd}" ] && sftppwd=$(echo "${sftppwd}" |sed 's/\\/\\\\/g' |sed 's/}/\\}/g' |sed 's/{/\\{/g' |sed 's/\[/\\[/g' |sed 's/\$/\\\$/g' |sed 's/`/\\`/g' |sed 's/\"/\\\"/g')

    expect << END
        #exp_internal 1
        set timeout ${fixtimeout}

        spawn ${sftpopts} ${sftpuser}@${sftpip}

        expect {
            "*assword:" { send -- "${sftppwd}\r" }
            "yes/no" { send "yes\r"; exp_continue }
            "*Permission denied*" { exit 11 }
            timeout { exit 201 }
        }

        expect {
            "\nsftp>*" { send -- "${sftpcmd}\r" }
            timeout { exit 202 }
        }

        set timeout ${sftptimeout}

        expect {
            "\nInvalid command." { exit 101 }
            "\n*not found." { exit 102 }
            "\nsftp>*" { send "exit\r" }
            timeout { exit 203 }
        }

        expect {
            timeout { exit 255 }
            eof { send_user "====End Of Expect SFTP====\n" }
        }

        catch wait result
        exit [lindex \$result 3]
END

    return $?
}