#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: verify_backup_data.sh
# Description: Verify backup data.
###############################################################################


##################################################
# Description:  Check parameters.
# Parameters:   $@
##################################################
function check_parameters() {
    echo "Start checking parameters...."

    local backupdata_type_name="$1"
    local timestamp_name="$2"

    if [ -z "${backupdata_type_name}" ]; then
        echo "Error, The parameter \$1: Parameter error."
        return 1
    fi

    if ! egrep -qw "^[0-9]+$" <<< "${timestamp_name}"; then
        echo "Error, The parameter \$2: Parameter error."
        return 1
    fi
}

##################################################
# Description:  Search the data to be verified.
# Parameters:   $@
##################################################
function search_verify_data() {
    echo "Start searching data to be verified...."

    if [ ! -d "${timestamp_path}" ]; then
        echo "Error, The timestamp path ${timestamp_path}: No such directory."
        return 1
    fi

    find ${timestamp_path} -type f |grep -v '.*\.sign$' > "${timestamp_verify_conf}"

    if [ ! -s "${timestamp_verify_conf}" ]; then
        echo "Error, The timestamp verify conf ${timestamp_verify_conf}: File is empty."
        return 1
    fi
}

##################################################
# Description:  Execute validation script.
# Parameters:   $@
##################################################
function execute_validation_script() {
    echo "Start executing validation script...."

    if [ ! -f "${validation_script}" ]; then
        echo "Error, The validation script ${validation_script}: No such file."
        return 1
    fi

    while read file || [ -n "${file}" ]
    do
        bash "${validation_script}" backup verify_sign -backup-pkg "${file}"
        if [ $? -ne 0 ]; then
            echo "Error, The file ${file}: Verification failed."
            return 1
        fi
    done < "${timestamp_verify_conf}"
}

##################################################
# Description:  main function.
# Parameters:   $@
##################################################
function main_verify_backup_data() {
    echo "Start verifying backup data...."

    local backupdata_temp_conf_path="/opt/backup/minibackup_temp/conf"
    local backupdata_temp_raw_path="/opt/backup/minibackup_temp/data/raw"
    local validation_script="/opt/oss/manager/apps/UniEPAgent/bin/uniep_tool"

    local timestamp_dir_name="$1"
    local backupdata_type_name="${timestamp_dir_name%/*}"
    local timestamp_name="${timestamp_dir_name##*/}"

    local timestamp_verify_conf="${backupdata_temp_conf_path}/${timestamp_name}_verify.conf"
    local timestamp_path="${backupdata_temp_raw_path}/${backupdata_type_name##*/}/${timestamp_name}"

    check_parameters "${backupdata_type_name}" "${timestamp_name}" || return $?
    search_verify_data || return $?
    execute_validation_script || return $?
}

main_verify_backup_data "$@"