#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: run_backup_service.sh
# Description: Backup service running.
###############################################################################


##################################################
# Description:  Initialize environment.
# Parameters:   null
##################################################
function init_env() {
    local workspace="$(dirname $(readlink -f $0))"

    source "${workspace}"/../public_definition/public_definition.sh
    source "${workspace}"/common_definition/common_definition.sh

    check_execution_user || {
    echo "The current execution user is wrong, please switch to the correct user."; return 1; }

    export LOGFILE="${BACKUP_LOG}"
    echo -e "\n========================================\n" >> "${LOGFILE}"
    chmod 600 "${LOGFILE}"

    log info "Start initializing environment...."
}

##################################################
# Description:  only delete minirestore data.
# Parameters:   null
##################################################
function is_only_delete_minirestore_data() {
    log info "Start checking whether only minirestore data is deleted...."

    local input_params="$@"

    if ! egrep -qow "${INPUT_TYPE_DELETEDATA}" <<< "${input_params}"; then
        log info "This is not the target instruction, the program will continue to execute."
        return 1
    fi

    log info "This is the target instruction, the instruction will execute and exit the program."

    if bash "${MINI_BACKUP_DELETE_UNNECESSARY_DATA_PATH}"/delete_minirestore_data.sh; then
        log info "Successfully deleted unnecessary data of minirestore."
    else
        log error "Failed to delete unnecessary data of minirestore."
    fi

    return 0
}

##################################################
# Description:  Check whether to retry backing up data.
# Parameters:   null
##################################################
function is_not_retry_backup_data() {
    log info "Start checking whether to retry backing up data...."

    local input_params="$@"

    if ! egrep -qow "${INPUT_TYPE_TETRYDATA}" <<< "${input_params}"; then
        log info "This is not the target instruction, the program will continue to execute."
        return 1
    fi

    if bash "${MINI_BACKUP_CHECK_BACKUP_DATA_PATH}"/check_backup_data.sh; then
        log info "Backup data is normal, no need to retry."
        return 0
    else
        log error "Backup data is abnormal, need to retry."
        return 1
    fi
}

##################################################
# Description:  Record backup data type.
# Parameters:   $@
##################################################
function record_backup_data_type() {
    log info "Start recording backup data types...."

    check_backup_data_type "$@" || {
    log error "The backup data type $@: Type not allowed."; return 1; }

    for type in $@
    do
        echo "${type}" > "${PRODUCT_TYPE_CONF}"
    done

    [ -s "${PRODUCT_TYPE_CONF}" ] || {
    log error "The backup data type conf ${PRODUCT_TYPE_CONF}: File is empty."; return 1; }

    chmod 600 "${PRODUCT_TYPE_CONF}" || {
    log error "Change permissions ${PRODUCT_TYPE_CONF}: Change failed."; return 1; }
}

##################################################
# Description:  main function.
# Parameters:   $@
##################################################
function main_run_backup_service() {
    init_env || return $?

    log info "Start running backup service...."

    check_execution_script && {
    log warning "The program is running, will exit this execution."; exit 0; }

    is_only_delete_minirestore_data "$@" && exit 0
    is_not_retry_backup_data "$@" && exit 0

    record_backup_data_type "$@" || return $?

    echo "${INPUT_TYPE_DECRYPT}" |"${BIN_PYTHON}" "${DATA_CRYPT_PYC}" || {
    log error "Execute "${BIN_PYTHON}" ${DATA_CRYPT_PYC}: Execute failed."; return 1; }

    log info "Running backup service successfully...."
}

main_run_backup_service "$@" 2>> /opt/backup/minirestore_service/log/backup.log