#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: start_backup.sh
# Description: Backup service running.
###############################################################################


##################################################
# Description:  Initialize environment.
# Parameters:   null
##################################################
function init_env() {
    local workspace="$(dirname $(readlink -f $0))"

    source "${workspace}"/../../public_definition/public_definition.sh
    source "${workspace}"/../common_definition/common_definition.sh

    check_execution_user || {
    echo "The current execution user is wrong, please switch to the correct user."; return 1; }

    export LOGFILE="${BACKUP_LOG}"
}

##################################################
# Description:  Record backup product name.
# Parameters:   null
##################################################
function record_backup_product_name() {
    log info "Start recording backup product name...."

    local login_ip=""
    local sftp_ip=""
    local login_user=""
    local login_passwd=""
    local execute_user=""
    local execute_passwd=""
    local sftp_user=""
    local sftp_passwd=""
    local sftp_backup_path=""

    source "${MINI_BACKUP_ACCOUNT_INFO_PATH}"/read_account.sh "execute" "${allpasswd}" || {
    log error "Failed to read account information."; return 1; }
    source "${MINI_BACKUP_ACCOUNT_INFO_PATH}"/read_account.sh "sftp" "${allpasswd}" || {
    log error "Failed to read account information."; return 1; }

    local sftp_temp_conf_path="${sftp_backup_path}/${SFTP_BACKUPDATA_TEMP_CONF_PATH}"
    local temp_backup_product_name_conf="${BACKUPDATA_TEMP_CONF_PATH}/${PRODUCT_NAME_CONF##*/}"

    log info "Start executing command...."

    local cmd="bash /opt/oss/manager/tools/resmgr/queryproduct.sh -pn all -output ${BACKUPDATA_TEMP_CONF_PATH}"
    execute_expect_connection "sshcmd" "${cmd}" "${MAX_SSH_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="ls ${BACKUPDATA_TEMP_CONF_PATH}/product_*.json |sed 's/\\\(.*product_\\\)\\\(.*\\\)\\\(.json\\\)/\\\2/g' > ${temp_backup_product_name_conf}"
    execute_expect_connection "sshcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="put ${temp_backup_product_name_conf} ${sftp_temp_conf_path}"
    execute_expect_connection "sshsftp" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="get ${sftp_temp_conf_path}/${PRODUCT_NAME_CONF##*/} ${BACKUPDATA_TEMP_CONF_PATH}"
    execute_expect_connection "sftpcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    [ -f "${temp_backup_product_name_conf}" ] || {
    log warning "The backup product name conf ${temp_backup_product_name_conf}: No such file."; return 0; }

    mv -f "${temp_backup_product_name_conf}" "${PRODUCT_NAME_CONF}"

    if [ ! -s "${PRODUCT_NAME_CONF}" ]; then
        log error "The backup product name conf ${PRODUCT_NAME_CONF}: File is empty."
        log error "Please check whether product name exists on the backup server."
        return 1
    fi

    if [ -f "${PRODUCT_NAME_CONF}" ]; then
        chmod 600 "${PRODUCT_NAME_CONF}" || {
        log error "Change permissions ${PRODUCT_NAME_CONF}: Change failed."; return 1; }
    fi
}

##################################################
# Description:  Check backup dynamic variables.
# Parameters:   null
##################################################
function check_backup_dynamic_var() {
    log info "Start checking backup dynamic variables...."

    local config_param_list=("${MAX_BACKUP_PKG_NUM}" "${MAX_SSH_TIMEOUT}" "${MAX_SFTP_TIMEOUT}")

    for param in "${config_param_list[@]}"
    do
        check_positive_integer "${param}" || {
        log error "The configuration parameter ${param}: Parameter is invalid."; return 1; }
    done
}

##################################################
# Description:  Run the specified backup.
# Parameters:   $@
##################################################
function backup_specified_data() {
    log info "Start backing up data of the specified type...."

    local backup_data_type="$1"
    local backup_node_ip="$2"

    log info "The current backup data type is ${backup_data_type}."
    log info "The current backup node ip is ${backup_node_ip}."

    # Define global variables based on data types
    define_backup_dynamic_var "${backup_data_type}" "${backup_node_ip}" || return $?
    check_backup_dynamic_var || return $?

    # Pre-operations, which are used to prepare for downloading necessary files
    bash "${MINI_BACKUP_DELETE_UNNECESSARY_DATA_PATH}"/delete_minirestore_data.sh
    echo "${allpasswd}" |bash "${MINI_BACKUP_DELETE_UNNECESSARY_DATA_PATH}"/delete_node_data.sh || return $?
    echo "${allpasswd}" |bash "${MINI_BACKUP_PREPARE_BEFORE_BACKUP_PATH}"/prepare_backup_directory.sh || return $?
    # Download necessary files
    record_backup_product_name || return $?

    while read backup_product_name || [ -n "${backup_product_name}" ]
    do
        log info "The current backup product name is ${backup_product_name}."

        # Define global variables based on product name
        define_backup_product_name_var "${backup_data_type}" "${backup_product_name}" || return $?
        create_backup_dynamic_files || return $?

        echo "${allpasswd}" |bash "${MINI_BACKUP_PREPARE_BEFORE_BACKUP_PATH}"/prepare_backup_directory.sh || return $?
        echo "${allpasswd}" |bash "${MINI_BACKUP_PREPARE_BEFORE_BACKUP_PATH}"/prepare_backup_data.sh || return $?

        bash "${MINI_BACKUP_BACKUP_TABLE_INFO_PATH}"/has_backup_table.sh || return $?
        echo "${allpasswd}" |bash "${MINI_BACKUP_BACKUP_TABLE_INFO_PATH}"/need_backup_table.sh || return $?
        bash "${MINI_BACKUP_BACKUP_TABLE_INFO_PATH}"/not_backup_table.sh || return $?
        bash "${MINI_BACKUP_BACKUP_TABLE_INFO_PATH}"/redundant_backup_table.sh || return $?

        echo "${allpasswd}" |bash "${MINI_BACKUP_INCREMENTAL_BACKUP_DATA_PATH}"/incremental_backup_data.sh || return $?

        bash "${MINI_BACKUP_DELETE_UNNECESSARY_DATA_PATH}"/delete_minirestore_data.sh
        echo "${allpasswd}" |bash "${MINI_BACKUP_DELETE_UNNECESSARY_DATA_PATH}"/delete_node_data.sh || return $?

        log info "The ${backup_product_name} product data backup successfully!"
    done < "${PRODUCT_NAME_CONF}"

    log info "The ${backup_data_type} type data backup successfully!"
}

##################################################
# Description:  main function.
# Parameters:   null
##################################################
function main_start_backup() {
    init_env || return $?

    log info "Start backing up data...."

    read -rs -t 3 passwd1
    read -rs -t 3 passwd2
    read -rs -t 3 passwd3
    local allpasswd="${passwd1} ${passwd2} ${passwd3}"

    for file in "${PRODUCT_TYPE_CONF}" "${ACCOUNT_IP_CONF}"; do
        [ -s "${file}" ] || {
        log error "The necessary conf: No such file."; return 1; }
    done

    while read node_ip || [ -n "${node_ip}" ]
    do
        while read backup_type || [ -n "${backup_type}" ]
        do
            backup_specified_data "${backup_type}" "${node_ip}" || return $?
        done < "${PRODUCT_TYPE_CONF}"
    done < "${ACCOUNT_IP_CONF}"

    log info "Successfully backed up data!"
}

main_start_backup