#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: incremental_backup_data.sh
# Description: Incremental backup data.
###############################################################################


##################################################
# Description:  Check whether incremental backup is required.
# Parameters:   null
##################################################
function is_need_incremental_backup() {
    log info "Start checking whether incremental backup is required...."

    # Do not execute without new data
    if [ ! -s "${NOT_BACKUP_TABLE}" ]; then
        log info "No new ${BACKUP_DATA_TYPE} data found, no incremental backup required."
        return 1
    fi

    return 0
}

##################################################
# Description:  Upload verification script.
# Parameters:   null
##################################################
function upload_verification_script() {
    log info "Start uploading verification script...."

    local sftp_temp_conf_path="${sftp_backup_path}/${SFTP_BACKUPDATA_TEMP_CONF_PATH}"
    local validation_script="${MINI_BACKUP_INCREMENTAL_BACKUP_DATA_PATH}/verify_backup_data.sh"

    local cmd="put ${validation_script} ${sftp_temp_conf_path}"
    execute_expect_connection "sftpcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="get ${sftp_temp_conf_path}/${validation_script##*/} ${BACKUPDATA_TEMP_CONF_PATH}"
    execute_expect_connection "sshsftp" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }
}

##################################################
# Description:  Backup incremental data.
# Parameters:   null
##################################################
function backup_incremental_data() {
    log info "Start backing up incremental data...."

    [ -f "${DISCARD_BACKUP_TABLE}" ] && rm -rf "${DISCARD_BACKUP_TABLE}"

    while read data || [ -n "${data}" ]
    do
        local timestamp_dir_name=$(awk -F ' ' '{print $1}' <<< "${data}")
        [ -n "${timestamp_dir_name}" ] || {
        log error "The timestamp dir name of ${data}: No such string."; return 1; }

        verify_incremental_data "${timestamp_dir_name}" || return $?
        package_incremental_data "${timestamp_dir_name}" || return $?
    done < "${NOT_BACKUP_TABLE}"

    if [ -f "${DISCARD_BACKUP_TABLE}" ]; then
        chmod 600 "${DISCARD_BACKUP_TABLE}" || {
        log error "Change permissions ${DISCARD_BACKUP_TABLE}: Change failed."; return 1; }
    fi
}

##################################################
# Description:  Verify incremental data.
# Parameters:   timestamp_dir_name
##################################################
function verify_incremental_data() {
    log info "Start verifying incremental data...."

    local validation_script="${MINI_BACKUP_INCREMENTAL_BACKUP_DATA_PATH}/verify_backup_data.sh"
    local timestamp_dir_name="$1"

    # Do not package invalid data, skip it
    local cmd="bash ${BACKUPDATA_TEMP_CONF_PATH}/${validation_script##*/} ${timestamp_dir_name}"
    local timeout="$[MAX_SSH_TIMEOUT*2]"
    if ! execute_expect_connection "sshcmd" "${cmd}" "${timeout}"; then
        log warning "Check ${timestamp_dir_name} backup data validity failed, skip it."
        echo "${timestamp_dir_name} ${file}" >> "${DISCARD_BACKUP_TABLE}"
        return 0
    fi
}

##################################################
# Description:  Package incremental data.
# Parameters:   timestamp_dir_name
##################################################
function package_incremental_data() {
    log info "Start packaging incremental data...."

    local timestamp_dir_name="$1"
    local timestamp_name="${timestamp_dir_name##*/}"
    local timestamp_pkg_name="${timestamp_name}.tar"
    local temp_data_raw_type_path="${BACKUPDATA_TEMP_RAW_PATH}/${BACKUPDATA_TYPE_DIR_NAME##*/}"
    local timestamp_path="${temp_data_raw_type_path}/${timestamp_name}"

    # Do not package invalid data, skip it
    if grep -qw "${timestamp_dir_name}" "${DISCARD_BACKUP_TABLE}" 2>/dev/null; then
        log warning "The ${timestamp_dir_name} backup data is invalid, skip it."
        return 0
    fi

    local cmd="ls ${timestamp_path} && test -d ${timestamp_path}"
    execute_expect_connection "sshcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    # Do not package incomplete data, skip it
    local cmd="cd ${timestamp_path} && ls ${BACKUPDATA_TYPE_FINISH_TAG_SIGN}"
    if ! execute_expect_connection "sshcmd" "${cmd}"; then
        log warning "Check ${timestamp_dir_name} backup data integrity failed, skip it."
        echo "${timestamp_dir_name} ${BACKUPDATA_TYPE_FINISH_TAG_SIGN}" >> "${DISCARD_BACKUP_TABLE}"
        return 0
    fi

    local cmd="cd ${temp_data_raw_type_path} && tar -cf ${BACKUPDATA_TEMP_NEW_PATH}/${timestamp_pkg_name} ${timestamp_name}"
    execute_expect_connection "sshcmd" "${cmd}" "${MAX_SSH_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }
}

##################################################
# Description:  Backup planning file.
# Parameters:   null
##################################################
function backup_planning_files() {
    log info "Start backing up planning files...."

    local cmd="bash /opt/oss/manager/tools/resmgr/queryproduct.sh -pn all -output ${BACKUPDATA_TEMP_NEW_PATH}"
    execute_expect_connection "sshcmd" "${cmd}" "${MAX_SSH_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    # The excel file may not exist for some products, skip it
    local cmd="cp -f /opt/pub/software/repository/*.xlsx ${BACKUPDATA_TEMP_NEW_PATH}"
    execute_expect_connection "sshcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 0; }
}

##################################################
# Description:  Package compressed data.
# Parameters:   null
##################################################
function package_compress_data() {
    log info "Start packing compressed data...."

    local cmd="ls ${BACKUPDATA_TEMP_NEW_PATH} |egrep '.*\.tar$|.*\.json$|.*\.xlsx$'"
    execute_expect_connection "sshcmd" "${cmd}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="cd ${BACKUPDATA_TEMP_NEW_PATH} && tar -cf ${backup_pkg_name} ./*"
    execute_expect_connection "sshcmd" "${cmd}" "${MAX_SSH_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="cd ${BACKUPDATA_TEMP_NEW_PATH} && sha256sum ${backup_pkg_name} > ${backup_pkg_name}.sum"
    execute_expect_connection "sshcmd" "${cmd}" "${MAX_SSH_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }
}

##################################################
# Description:  Download compress packages.
# Parameters:   null
##################################################
function download_compress_packages() {
    log info "Start downloading compressed packages...."

    local sftp_temp_data_path="${sftp_backup_path}/${SFTP_BACKUPDATA_TEMP_DATA_PATH}"

    local cmd="put ${BACKUPDATA_TEMP_NEW_PATH}/${backup_pkg_name}* ${sftp_temp_data_path}"
    execute_expect_connection "sshsftp" "${cmd}" "${MAX_SFTP_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }

    local cmd="get ${sftp_temp_data_path}/${backup_pkg_name}* ${BACKUPDATA_TEMP_DATA_PATH}"
    execute_expect_connection "sftpcmd" "${cmd}" "${MAX_SFTP_TIMEOUT}" || {
    log error "Execute command ${cmd}, return $?: Execute failed."; return 1; }
}

##################################################
# Description:  Verify backup data.
# Parameters:   null
##################################################
function verify_backup_data() {
    log info "Start verifying backup data...."

    cd "${BACKUPDATA_TEMP_DATA_PATH}" || return $?

    sha256sum -c "${backup_pkg_name}.sum" &>> "${LOGFILE}" || {
    log error "Verify file ${backup_pkg_name}: Verification failed."; return 1; }

    tar -xf "${backup_pkg_name}" || {
    log error "Decompress file ${backup_pkg_name}: Decompression failed."; return 1; }

    ls "${BACKUPDATA_TEMP_DATA_PATH}/${backup_pkg_name}"* |xargs rm -rf

    egrep '.*\.json$|.*\.xlsx$' <<< "$(ls ${BACKUPDATA_TEMP_DATA_PATH})" |sed \
        "s#\(.*\)#${BACKUPDATA_TYPE_DIR_NAME}/\1#g" >> "${NOT_BACKUP_TABLE}"
}

##################################################
# Description:  Copy backup data.
# Parameters:   null
##################################################
function copy_backup_data() {
    log info "Start copying backup data...."

    > "${FINISH_BACKUP_TABLE}"
    chmod 600 "${FINISH_BACKUP_TABLE}" || {
    log error "Change permissions ${FINISH_BACKUP_TABLE}: Change failed."; return 1; }

    while read data || [ -n "${data}" ]
    do
        cp -f "${BACKUPDATA_TEMP_DATA_PATH}/${data}" "${BACKUPDATA_TYPE_PATH}" || {
        log error "Copy file ${backup_pkg_name}: Copy failed."; return 1; }

        chmod 400 "${BACKUPDATA_TYPE_PATH}/${data}" || {
        log error "Change permissions ${BACKUPDATA_TYPE_PATH}/${data}: Change failed."; return 1; }

        local src_size=$(stat -c%s "${BACKUPDATA_TEMP_DATA_PATH}/${data}")
        local dst_size=$(stat -c%s "${BACKUPDATA_TYPE_PATH}/${data}")
        [ "${src_size}" -eq "${dst_size}" ] &&  echo "${data}" >> "${FINISH_BACKUP_TABLE}"
    done <<< "$(ls ${BACKUPDATA_TEMP_DATA_PATH})"
}

##################################################
# Description:  main function.
# Parameters:   $@
##################################################
function main_incremental_backup_data() {
    source "${PUBLIC_DEFINITION_PATH}"/public_definition.sh
    source "${MINI_BACKUP_COMMON_DEFINITION_PATH}"/common_definition.sh

    log info "Start backing up incremental data...."

    read -rs -t 3 passwd1
    read -rs -t 3 passwd2
    read -rs -t 3 passwd3
    local allpasswd="${passwd1} ${passwd2} ${passwd3}"

    local login_ip=""
    local sftp_ip=""
    local login_user=""
    local login_passwd=""
    local execute_user=""
    local execute_passwd=""
    local sftp_user=""
    local sftp_passwd=""
    local sftp_backup_path=""

    source "${MINI_BACKUP_ACCOUNT_INFO_PATH}"/read_account.sh "execute" "${allpasswd}" || {
    log error "Failed to read account information."; return 1; }
    source "${MINI_BACKUP_ACCOUNT_INFO_PATH}"/read_account.sh "sftp" "${allpasswd}" || {
    log error "Failed to read account information."; return 1; }

    local backup_pkg_name="${BACKUP_DATA_TYPE}.tar"

    is_need_incremental_backup || exit 0
    upload_verification_script || return $?
    backup_incremental_data || return $?
    backup_planning_files || return $?
    package_compress_data || return $?
    download_compress_packages || return $?
    verify_backup_data || return $?
    copy_backup_data || return $?
}

main_incremental_backup_data