#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: scheduled_backup_task.sh
# Description: Read account.
###############################################################################


##################################################
# Description:  Check backup data type.
# Parameters:   type
##################################################
function check_backup_data_type() {
    local allow_type_list=("productdata" "deletedata" "retrydata")

    local curr_type_list="$@"

    [ -n "${curr_type_list}" ] || return $?

    for curr_type in ${curr_type_list}
    do
        grep -qw "${curr_type}" <<< "${allow_type_list[@]}" || return $?
    done
}

##################################################
# Description:  Configure scheduled backup tasks.
# Parameters:   $@
##################################################
function config_scheduled_backup_task() {
    echo "Start configuring scheduled backup tasks...."

    local user="ossadm"
    local crontab_path="/etc/crontab"

    [ -f "${run_backup_script}" ] || {
    echo "The backup script ${run_backup_script}: No such file."; return 1; }

    local scheduled_backup_time=$(egrep -w "^SCHEDULED_BACKUP_TIME=[0-9:]+" ${config_params_conf} |head -1 |awk -F '=' '{print $2}')
    egrep -qw "^[0-2]?[0-9]:[0-9]?[0-9]" <<< "${scheduled_backup_time}" || scheduled_backup_time="00:00"

    local hours=${scheduled_backup_time%:*}
    local minutes=${scheduled_backup_time##*:}

    [ "${hours}" -gt 24 ] && {
    echo "Error, The hours ${hours}: Parameter is invalid."; return 1; }
    [ "${minutes}" -gt 59 ] && {
    echo "Error, The minutes ${minutes}: Parameter is invalid."; return 1; }

    local crontab_cmd="${minutes} ${hours} * * * ${user} bash ${run_backup_script} $@ > /dev/null 2>&1"
    grep -qw "${run_backup_script##*/}" "${crontab_path}" && sed -i "\|${run_backup_script##*/}|d" "${crontab_path}"
    echo "${crontab_cmd}" >> "${crontab_path}"
}

##################################################
# Description:  Configure system logrotate.
# Parameters:   null
##################################################
function config_system_logrotate() {
    echo "Start configuring system logrotate...."

    local minirestote_lograte_file="/etc/logrotate.d/minirestore"

    local max_log_num=$(egrep -w "^MAX_LOG_NUM=[0-9]+" ${config_params_conf} |head -1 |awk -F '=' '{print $2}')
    egrep -qw "^[0-9]+" <<< "${max_log_num}" || max_log_num="10"

    cat > "${minirestote_lograte_file}" << EOF
/opt/backup/minirestore_service/log/*.log {
    daily
    dateext
    rotate ${max_log_num}
    compress
    notifempty
    missingok
    copytruncate
    sharedscripts
    postrotate
        chmod 0400 //opt/backup/minirestore_service/log/*.log-*
    endscript
}
EOF
}

##################################################
# Description:  Create minirestore service.
# Parameters:   null
##################################################
function create_minirestore_service() {
    echo "Start creating minirestore service...."

    local minirestore_clean_service_file="/usr/lib/systemd/system/minirestore-clean.service"
    local minirestore_retry_service_file="/usr/lib/systemd/system/minirestore-retry.service"
    local minirestore_retry_timer_file="/usr/lib/systemd/system/minirestore-retry.timer"

    cat > "${minirestore_clean_service_file}" << EOF
[Unit]
Description=minirestore-clean service
Wants=network.target sshd-keygen.service

[Service]
Type=simple
ExecStart=/bin/su - ossadm -c "bash ${run_backup_script} deletedata"
Restart=on-failure
RestartSec=1800s

[Install]
WantedBy=multi-user.target
Alias=minires-clean.service
EOF

    cat > "${minirestore_retry_service_file}" << EOF
[Unit]
Description=minirestore-retry service
Wants=network.target sshd-keygen.service

[Service]
Type=simple
ExecStart=/bin/su - ossadm -c "bash ${run_backup_script} retrydata"

[Install]
WantedBy=multi-user.target
Alias=minires-retry.service
EOF

    cat > "${minirestore_retry_timer_file}" << EOF
[Unit]
Description=minirestore-retry timer

[Timer]
Unit=minirestore-retry.service
OnBootSec=10min
OnUnitActiveSec=2h
Persistent=true

[Install]
WantedBy=multi-user.target
Alias=minires-retry.timer
EOF

    chmod 644 "${minirestore_clean_service_file}" "${minirestore_retry_service_file}" "${minirestore_retry_timer_file}"

    systemctl daemon-reload
    systemctl enable minirestore-clean.service || return $?
    systemctl start minirestore-clean.service || return $?
    systemctl enable minirestore-retry.timer || return $?
    systemctl start minirestore-retry.timer || return $?
}

##################################################
# Description:  main function.
# Parameters:   $@
##################################################
function main_scheduled_backup_task() {
    echo "Start deploying scheduled backup tasks...."

    local config_params_conf="/opt/backup/minirestore_service/conf/custom/custom.conf"
    local run_backup_script="/opt/backup/minirestore_service/bin/mini_backup/run_backup_service.sh"

    check_backup_data_type "$@" || {
    echo "Error, The backup data type $@: Type not allowed."; return 1; }

    config_scheduled_backup_task "$@" || {
    echo "Error, Configuring scheduled backup tasks: Configuration failed."; return 1; }

    config_system_logrotate || {
    echo "Error, Configuring system logrotate: Configuration failed."; return 1; }

    create_minirestore_service || {
    echo "Error, Create minirestore service: Create failed."; return 1; }

    echo "Scheduled backup task deployed successfully!"
}

main_scheduled_backup_task "$@"