#!/bin/bash

###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# File name: scheduled_backup_task.sh
# Description: Read account.
###############################################################################


##################################################
# Description:  Configure ntp service port.
# Parameters:   null
##################################################
function config_ntp_service_port() {
    echo "Start configuring the ntp service port...."

    local euler_rc_local="/etc/rc.d/rc.local"
    local suse_rc_local="/etc/rc.d/after.local"
    local iptables_comment="ntp service rule"
    local ip_list=$(ip addr |egrep 'inet[6]?' |awk -F' ' '{print $2}' |cut -d '/' -f 1)

    local os_rc_local="${euler_rc_local}"
    [ -f "${suse_rc_local}" ] && os_rc_local="${suse_rc_local}"

    iptables-save |grep -v -w "${iptables_comment}" |iptables-restore
    ip6tables-save |grep -v -w "${iptables_comment}" |ip6tables-restore
    [ -f "${os_rc_local}" ] && sed -i "/${iptables_comment}/d" "${os_rc_local}"

    for ip in ${ip_list}
    do
        local iptables_cmd="iptables"
        local iptables_options="-w -A INPUT -p udp -d "${ip}" --dport 123 -j ACCEPT -m comment --comment '${iptables_comment}'"
        grep -q ":" <<< "${ip}" && iptables_cmd="ip6tables"

        eval "${iptables_cmd}" "${iptables_options}" || return $?
        echo "${iptables_cmd} ${iptables_options}" >> "${os_rc_local}"
    done

    if grep -q "\." <<< "${ip_list}"; then
        local iptables_cmd="iptables"
        local iptables_options="-w -A INPUT -p udp --dport 123 -j DROP -m comment --comment '${iptables_comment}'"
        eval "${iptables_cmd}" "${iptables_options}" || return $?
        echo "${iptables_cmd} ${iptables_options}" >> "${os_rc_local}"
    fi

    if grep -q ":" <<< "${ip_list}"; then
        local iptables_cmd="ip6tables"
        local iptables_options="-w -A INPUT -p udp --dport 123 -j DROP -m comment --comment '${iptables_comment}'"
        eval "${iptables_cmd}" "${iptables_options}" || return $?
        echo "${iptables_cmd} ${iptables_options}" >> "${os_rc_local}"
    fi
}

##################################################
# Description:  Configure sftp service.
# Parameters:   null
##################################################
function config_sftp_service() {
    echo "Start configuring the sftp service...."

    local sshd_config="/etc/ssh/sshd_config"

    sed -i 's/^\s*Subsystem\s*sftp/#Subsystem sftp/g' "${sshd_config}" || return $?
}

##################################################
# Description:  Create a user.
# Parameters:   user
##################################################
function create_user() {
    echo "Start creating user...."

    local user="$1"
    local uid="$2"
    local group="$3"
    local groups="$4"

    [ -n "${user}" -a -n "${uid}" -a -n "${group}" ] || { echo "Error, The user parameter is incorrect."; return 1; }

    check_user=$(id "${user}" 2>/dev/null |grep -w "${user}")
    if [ -n "${check_user}" ]; then
        echo "Warning, The ${user} user already exists and will be created again."
        userdel -rf "${user}"
    fi

    local default_shell="/bin/bash"
    local default_home="/home"

    useradd -u "${uid}" -g "${group}" -G "${groups}" -m -d "${default_home}/${user}" -s "${default_shell}" "${user}" || return $?
}

##################################################
# Description:  Create a user's passwd.
# Parameters:   passwd
##################################################
function create_passwd() {
    echo "Start creating passwd...."

    local user="$1"

    [ -n "${user}" ] || { echo "Error, The user parameter is incorrect."; return 1; }

    check_user=$(id "${user}" 2>/dev/null |grep -w "${user}")
    [ -n "${check_user}" ] || { echo "Error, The ${user} user does not exist."; return 1; }

    read -rs -t 3 passwd
    passwd=$(echo "${passwd}" |sed 's/\\/\\\\/g' |sed 's/}/\\}/g' |sed 's/{/\\{/g' |sed 's/\[/\\[/g' |sed 's/\$/\\\$/g' |sed 's/`/\\`/g' |sed 's/\"/\\\"/g')

    expect << END
        set timeout 10
        spawn passwd ${user}

        expect {
            "New password:" { send -- "${passwd}\r"; exp_continue}
            "Retype new password:" { send -- "${passwd}\r"}
            "BAD PASSWORD" { exit 11 }
            timeout { exit 201 }
        }

        expect {
            "not match" { exit 13 }
            "already used" { exit 14 }
            -re ".*root@.*\]" { send "exit\r" }
            timeout { exit 202 }
        }

        catch wait result
        exit [lindex \$result 3]
END
}

##################################################
# Description:  Configure password validity.
# Parameters:   null
##################################################
function config_passwd_validity() {
    echo "Start configuring password validity...."

    local user=$(egrep -o "user=[[:alnum:]]+" <<< "$@" |sed "s/user=//g")
    local min_day=$(egrep -o "min_day=[0-9]+" <<< "$@" |sed "s/min_day=//g")
    local max_day=$(egrep -o "max_day=[0-9]+" <<< "$@" |sed "s/max_day=//g")
    local warn_day=$(egrep -o "warn_day=[0-9]+" <<< "$@" |sed "s/warn_day=//g")
    local grace_day=$(egrep -o "grace_day=[0-9]+" <<< "$@" |sed "s/grace_day=//g")

    local option_min_day=""
    local option_max_day=""
    local option_warn_day=""
    local option_grace_day=""

    [ -n "${user}" ] ||  { echo "Error, The user does not exist."; return 1; }
    [ -n "${min_day}" ] && option_min_day="-m ${min_day}"
    [ -n "${max_day}" ] && option_max_day="-M ${max_day}"
    [ -n "${warn_day}" ] && option_warn_day="-W ${warn_day}"
    [ -n "${grace_day}" ] && option_grace_day="-I ${grace_day}"

    chage ${option_min_day} ${option_max_day} ${option_warn_day} ${option_grace_day} ${user} || return $?
}

##################################################
# Description:  Configure allowed users.
# Parameters:   null
##################################################
function config_allowed_users() {
    echo "Start configuring allowed users...."

    local sshd_config="/etc/ssh/sshd_config"

    echo "AllowUsers sopuser" >> "${sshd_config}" || return $?
}

##################################################
# Description:  Restart system services.
# Parameters:   null
##################################################
function restart_system_services() {
    echo "Start Restarting system services...."

    systemctl restart sshd || return $?
}

##################################################
# Description:  main function.
# Parameters:   $@
##################################################
function main_scheduled_backup_task() {
    echo "Start executing system security configuration items...."

    config_ntp_service_port || {
    echo "Error, Configuring ntp service port: Configuration failed."; return 1; }

    config_sftp_service || {
    echo "Error, Configuring sftp service: Configuration failed."; return 1; }

    create_user "sopuser" "3008" "sopgroup" "wheel" || {
    echo "Error, Creating user: Creation failed."; return 1; }

    create_passwd "sopuser" || {
    echo "Error, Creating passwd: Configuration failed."; return 1; }

    config_passwd_validity "user=sopuser" "min_day=7" "max_day=180" "warn_day=90" "grace_day=35" || {
    echo "Error, Configuring password validity: Configuration failed."; return 1; }

    config_allowed_users || {
    echo "Error, Configuring allowed user: Configuration failed."; return 1; }

    restart_system_services || {
    echo "Error, Restarting system services: Restart failed."; return 1; }

    echo "The system security configuration items executed successfully!"
}

main_scheduled_backup_task