#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2018 Huawei Technologies Co. Ltd. All rights reserved.
"""
    huawei websocket frame class
"""

import struct
import random
import six
from networking_huawei.drivers.ac.encode_convert import convert_to_bytes

CONTROL_FRAME_MAX_LENGTH = 125


class Frame(object):
    """ Simple object to hold frame data in websocket """

    # opcode list:
    OPCODE_CONTINUATION = 0x0
    OPCODE_TEXT = 0x1
    OPCODE_BINARY = 0x2
    OPCODE_CLOSE = 0x8
    OPCODE_PING = 0x9
    OPCODE_PONG = 0xA

    # defined status codes when sending a Close frame.
    STATUS_NORMAL = 1000

    # initialization
    def __init__(self, fin=0, rsv1=0, rsv2=0, rsv3=0,
                 opcode=OPCODE_TEXT, mask=0, length=0, data=""):
        self.fin = fin
        self.rsv1 = rsv1
        self.rsv2 = rsv2
        self.rsv3 = rsv3
        self.opcode = opcode
        self.mask = mask
        self.payload_length = length
        self.payload = data
        self.masking_key = None

    def get_ping_frame(self, data=""):
        """get a ping frame
        """
        self.fin = 1
        self.mask = 1
        self.masking_key = self.generate_random_masking_key()
        self.opcode = self.OPCODE_PING
        self.payload = data
        self.payload_length = len(data)
        if self.payload_length > CONTROL_FRAME_MAX_LENGTH:
            raise Exception("PING frame too large.")
        return self

    def get_pong_frame(self, data=""):
        """get a pong frame"""
        self.fin = 1
        self.mask = 1
        self.masking_key = self.generate_random_masking_key()
        self.opcode = self.OPCODE_PONG
        self.payload = data
        self.payload_length = len(data)
        if self.payload_length > CONTROL_FRAME_MAX_LENGTH:
            raise Exception("PONG frame too large.")
        return self

    def get_close_frame(self, status_code=STATUS_NORMAL, reason=""):
        """get a close frame"""
        self.opcode = self.OPCODE_CLOSE
        self.payload += struct.pack("!H", status_code)
        if isinstance(reason, six.text_type):
            self.payload += reason.encode('utf-8')
        else:
            self.payload += reason
        self.payload_length = len(self.payload)
        if self.payload_length > CONTROL_FRAME_MAX_LENGTH:
            raise Exception("CLOSE frame too large.")
        self.fin = 1
        self.mask = 1
        self.masking_key = self.generate_random_masking_key()
        return self

    def get_message_frame(self, message, first=True, end=True, mask=True):
        """get a text message frame"""
        if mask:
            self.mask = 1
            self.masking_key = self.generate_random_masking_key()
        else:
            self.mask = 0

        if first:
            self.opcode = self.OPCODE_TEXT
        else:
            self.opcode = self.OPCODE_CONTINUATION
        self.payload = message
        self.payload_length = len(message)
        if end:
            self.fin = 1
        else:
            self.fin = 0
        return self

    def get_close_status_and_reason(self):
        """get the status and reason tuple of a close frame"""
        if self.opcode == self.OPCODE_CLOSE:
            status = 1000
            reason = u''
            if self.payload_length >= 2:
                status = struct.unpack_from('!H', self.payload[:2])[0]
                reason = self.payload[2:]
                if reason:
                    reason = reason.decode('utf8', errors='strict')
            return status, reason
        else:
            return None

    def to_data_frame(self):
        """generate data send to controller

        :return: data send controller
        """
        # the first byte
        first_byte = (self.fin << 7) + (self.rsv1 << 6) + (self.rsv2 << 5) \
                     + (self.rsv3 << 4) + self.opcode
        frame_data = convert_to_bytes(first_byte)

        if self.payload_length < 126:
            # the second byte
            frame_data += struct.pack("B", (self.mask << 7) +
                                      self.payload_length)
        elif self.payload_length < 65536:
            # the second byte and the extended payload length
            frame_data += struct.pack("!BH", (self.mask << 7) + 126,
                                      self.payload_length)
        else:
            # the second byte and the extended payload length
            frame_data += struct.pack("!BQ", (self.mask << 7) + 127,
                                      self.payload_length)
        # add masking key
        if self.mask:
            for mask_data in self.masking_key:
                frame_data += convert_to_bytes(ord(mask_data))
            i = 0
            for data in self.payload:
                frame_data += \
                    convert_to_bytes(ord(data) ^ ord(self.masking_key[i % 4]))
                i += 1
        else:
            frame_data += self.payload
        return frame_data

    def generate_random_masking_key(self):
        """
        generate a random masking key
        :return: random masking key
        """
        masking_key = ""
        while len(masking_key) < 4:
            masking_key += chr(random.randint(0, 255))
        return masking_key

    # for debug
    def __str__(self):
        return "FIN=" + str(self.fin) + ", RSV1=" + str(self.rsv1) + \
               ", RSV2=" + str(self.rsv2) + ", RSV3=" + str(self.rsv3) + \
               ", OPCODE=" + str(self.opcode) + ", MASK=" + str(self.mask) + \
               ", Payload Length=" + str(self.payload_length) + \
               ", Masking Key=" + str(self.masking_key) + \
               ", Payload=" + str(self.payload)
