#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
""" WebSocket Client Class"""

import os
import threading

from concurrent.futures import ThreadPoolExecutor
from distutils.sysconfig import get_python_lib, get_python_version
from oslo_config import cfg
import eventlet
from oslo_serialization import jsonutils
from networking_huawei._i18n import _LI, _LE
from networking_huawei.common.exceptions import NoConnectionFromIpException
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.ac_agent.rpc.websocket. \
    websocket_terminal import WebSocketTerminal
from networking_huawei.drivers.ac.ac_agent.rpc.websocket.\
    websocket import WebSocket
from networking_huawei.drivers.ac.ac_agent.rpc.websocket.frame import Frame
from networking_huawei.drivers.ac.ac_agent.rpc.websocket. \
    websocket_future import Future
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.encode_convert import convert_to_str

eventlet.monkey_patch()
LOG = logging.getLogger(__name__)
UTF8 = "UTF-8"
PING = "ping"
PONG = "pong"


class WebSocketClient(object):
    """ WebSocket Client Class"""
    def __init__(self, remote_address, local_address, aware,
                 ssl_crt_file_path):
        self.remote_address = remote_address
        self.local_address = local_address
        self.ssl_crt_file_path = ssl_crt_file_path
        self.inputs = []
        self.outputs = []
        self.aware = aware
        self.input_executor = ThreadPoolExecutor(
            max_workers=constants.RPC_HANDLE_MAX_WORKERS)
        self.lock = threading.Lock()

        self.websock = WebSocket(remote_address, local_address,
                                 ssl_crt_file_path)
        self.terminal = WebSocketTerminal(None, self.remote_address,
                                          self.input_executor)
        self.terminal.start_req_timeout()
        self.shutdown = True
        self.future = Future()
        self.send_close_flag = False
        self.buf = ""

    def start(self):
        """receive message"""
        curr_thread = threading.currentThread()
        curr_thread.setName("WebsocketClient")
        if self.inputs:
            LOG.info(_LI("[AC]Rpc dispatcher get sock connect"))
        else:
            LOG.error(_LE("[AC]Rpc dispatcher do not get sock connect"))

        while self.inputs:
            try:
                frame = self.websock.recv()
                if frame:
                    self._handle_msg(frame)
            except Exception as ex:
                LOG.error(_LE("[AC]websock recv error: %s"), str(ex))

    def _handle_msg(self, frame):
        """handle message"""
        opcode = frame.opcode
        if opcode == Frame.OPCODE_CLOSE:
            LOG.info(_LI("[AC]recv a close msg"))
            if not self.send_close_flag:
                self.websock.send_close()
            status, reason = frame.get_close_status_and_reason()
            LOG.info("[AC]status:%d, close reason:%s", status, reason)
            self.close_connection()
            self.send_close_flag = False
        elif opcode == Frame.OPCODE_PING:
            LOG.warn(_LI("[AC]recv a ping!"))
            self.websock.send_pong()
        elif opcode == Frame.OPCODE_PONG:
            LOG.info("[HEARTBEAT]recv a pong!")
            self.future.notify(PONG, PONG, None)
        else:
            self.buf += convert_to_str(frame.payload)
            LOG.info("[AC]recv frame:%s", frame)
            if not self.buf:
                return
            if frame.fin == 1:
                buf_data = self.buf
                self.buf = ""
                self.terminal._handle_request(buf_data)

    def has_connection(self):
        """Check connection"""
        return self.websock.connection is not None

    def get_client(self):
        """get client"""
        if self.has_connection():
            return self.websock.connection
        else:
            raise NoConnectionFromIpException()

    def create_connection(self):
        """create connection"""
        with self.lock:
            LOG.info(_LI("[AC]create websocket connection"))
            if self.websock.connect() == -1:
                return -1
            self.inputs.append(self.websock.connection)
            self.terminal.websock = self.websock
            self.aware.added(self.terminal)
            self.shutdown = False
            self.send_close_flag = False
            return 0

    def close_with_signal(self):
        """close with signal"""
        LOG.info(_LI("[AC]websocket client close with signal"))
        if self.websock.connection:
            if self.websock.connection in self.inputs:
                self.close_connection()
                self.send_close_flag = True

    def close_connection(self):
        """init client after connection close"""
        with self.lock:
            LOG.info(_LI("[AC]websocket client close connection"))
            if self.websock.connection:
                if self.websock.connection in self.inputs:
                    self.inputs.remove(self.websock.connection)
                    self.websock.close()
                    self.terminal.websock = None
                    self.aware.removed(self.terminal)
                    self.shutdown = True

    def register(self, handler):
        """register"""
        self.terminal._register(handler)

    @classmethod
    def _get_certs(cls):
        """get certs"""
        dir_certs = os.path.dirname(os.path.realpath(__file__))
        key_file_path = os.path.join(dir_certs, "json_rpc_privkey.pem")
        cert_file_path = os.path.join(dir_certs, "json_rpc_cert.pem")
        ca_cert_path = os.path.join(dir_certs, "json_rpc_cacert.pem")
        if os.path.exists(key_file_path) and \
                os.path.exists(cert_file_path) and \
                os.path.exists(ca_cert_path):
            return key_file_path, cert_file_path, ca_cert_path

        LOG.error(_LE("[AC] %s, %s, %s are not found"),
                  key_file_path, cert_file_path, ca_cert_path)
        return None, None, None

    def send_plugin_version(self, local_ip, cloud_name):
        """send plugin version"""
        try:
            lib_path = get_python_lib()
            version_path = lib_path + '/networking_huawei/version.txt'
            python_version = get_python_version()
            if not os.path.exists(version_path):
                version_path = os.path.realpath(
                    '/usr/local/lib/python' + python_version + '/dist-packages'
                    '/networking_huawei/version.txt')
            with open(version_path) as ver_path:
                try:
                    version_info = ver_path.read().split('\n')
                    version_info = '{}'.format(version_info[0])
                except Exception:
                    version_info = ''
            json_request = {"id": "sync_plugin_version",
                            "method": "sync_plugin_version",
                            "plugin_version": version_info,
                            "agent_ip": local_ip,
                            "ops_name": cloud_name}
            LOG.info(_LI("[AC]Send plugin version info to AC: %s"),
                     json_request)
            req = jsonutils.dumps(json_request)
            self.websock.send(req)
        except Exception as ex:
            LOG.error(_LE(str(ex)))

    def send_network_black_white_list(self, local_ip, cloud_name):
        """send network black white list"""
        try:
            json_request = {
                "id": "sync_network_black_white_list",
                "method": "sync_network_black_white_list",
                "white_list":
                    cfg.CONF.huawei_ac_config.network_white_list,
                "black_list":
                    cfg.CONF.huawei_ac_config.network_black_list,
                "matching_list":
                    cfg.CONF.huawei_ac_config.network_list_matching,
                "suffix_prefix":
                    cfg.CONF.huawei_ac_config.network_list_matching,
                "agent_ip": local_ip,
                "ops_name": cloud_name}
            LOG.info(_LI("[AC]Send network black list and "
                         "white list to AC: %s"), json_request)
            req = jsonutils.dumps(json_request)
            self.websock.send(req)
        except Exception as ex:
            LOG.error(_LE(str(ex)))
