#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Websocket rpc agent for status reporting."""

import sched
import time
import six
from six.moves import range
from oslo_config import cfg

from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.common.constants import \
    THREAD_CANCEL_MAX_TIMES
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.common.exceptions import NoRpcClientFromIpException, \
    NoTerminalFromIpException
from networking_huawei.drivers.ac.ac_agent.ac_agent_api import ACAgentAPI
from networking_huawei.drivers.ac.ac_agent.decorator import Singleton
from networking_huawei.drivers.ac.ac_agent.rpc.rpc_client import \
    RpcClient
from networking_huawei.drivers.ac.client.service import ACReSTService
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

LOG = logging.getLogger(__name__)


class RpcAgent(six.with_metaclass(Singleton, object)):
    """Websocket rpc agent for status reporting."""

    def __init__(self):
        LOG.info(_LI("[AC]Init rpc agent"))
        rpc_server_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip. \
            replace(' ', '').split(',')
        rpc_server_list = [x.lower() for x in rpc_server_list]
        self.remote_host_dict = {}
        for rpc_server in rpc_server_list:
            self.remote_host_dict[rpc_server] = rpc_server
        self.ac_db_interface = ACdbInterface()
        session = self.ac_db_interface.get_session()
        active_ac = self.ac_db_interface.get_active_ac_ip(session)
        if active_ac is None:
            remote_host = rpc_server_list[0]
        elif active_ac == constants.DEFAULT_AC_IP:
            remote_host = rpc_server_list[0]
        else:
            remote_host = self.remote_host_dict.get(active_ac.lower())

        self.ops_version = ncu.get_ops_version()
        self.host_ip = ACCommonUtil.get_local_host_ip()
        self.remote_address = (remote_host, constants.websocket_server_port)
        self.cloud_name = cfg.CONF.huawei_ac_config.cloud_name

        self.client = RpcClient()
        # add rpc client
        self.client.add_client(self.remote_address, (self.host_ip, 0))

        # register API
        self.client.register_all(ACAgentAPI())

        # start rpc client
        self.start_rpc_client()

        rest_service = ACReSTService()
        rest_service.client.callback_restart_rpc = self.restart_client

    def restart_client(self, new_ip):
        """Restart websocket rpc client."""
        remote_address = (self.remote_host_dict.get(new_ip),
                          constants.websocket_server_port)
        if not self.client:
            return
        original_remote_addr = self.client.remote_addr

        if original_remote_addr == remote_address:
            LOG.info(_LI("[AC] RPC server IP is not changed."))
            return

        LOG.info(_LI(
            "[AC] AC cluster has been switchover. Register RPC client to new "
            "AC southbound IP (%s)."), self.remote_host_dict.get(new_ip))

        if self.client.websocket_clients.get(original_remote_addr):
            self.client.websocket_clients[
                original_remote_addr].close_with_signal()
        self.client.websocket_clients = {}
        if self.client.futures.get(original_remote_addr):
            is_running = self.client.futures.get(original_remote_addr).running()
            cancel_count = 0
            while is_running and cancel_count < THREAD_CANCEL_MAX_TIMES:
                self.client.futures.get(original_remote_addr).cancel()
                is_running = self.client.futures.get(original_remote_addr).running()
                time.sleep(0.1)
                cancel_count += 1
            LOG.debug("[AC] Cancel RPC client thread.")
        self.client.executor._threads.clear()

        self.client.add_client(remote_address, (self.host_ip, 0))
        self.client.register_all(ACAgentAPI())

    def start_rpc_client(self):
        """Start websocket rpc client."""
        try:
            schedule = sched.scheduler(time.time, time.sleep)

            # start rpc client
            schedule.enter(constants.DELAY_TIME_TO_START_CLIENT, 0,
                           self.client.start_client,
                           (self.remote_address, self.host_ip,
                            self.cloud_name))
            schedule.run()
        except (NoRpcClientFromIpException, NoTerminalFromIpException) as ex:
            LOG.error(_LE("[AC]No rpc client or no terminal exception: %s"),
                      str(ex))
        except Exception as ex:
            LOG.error(_LE("[AC]Start json rpc client exception: %s"), str(ex))
