# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能：处理文件相关：权限、用户
| 版本：2022-03-28 10:48 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import os
import stat

try:
    from stat import filemode as stat_filemode
except ImportError:
    _filemode_table = (((stat.S_IFLNK, "l"), (stat.S_IFSOCK, "s"), (stat.S_IFREG, "-"), (stat.S_IFBLK, "b"),
                        (stat.S_IFDIR, "d"), (stat.S_IFCHR, "c"), (stat.S_IFIFO, "p")),
                       ((stat.S_IRUSR, "r"),), ((stat.S_IWUSR, "w"),),
                       ((stat.S_IXUSR | stat.S_ISUID, "s"), (stat.S_ISUID, "S"), (stat.S_IXUSR, "x")),
                       ((stat.S_IRGRP, "r"),), ((stat.S_IWGRP, "w"),),
                       ((stat.S_IXGRP | stat.S_ISGID, "s"), (stat.S_ISGID, "S"), (stat.S_IXGRP, "x")),
                       ((stat.S_IROTH, "r"),), ((stat.S_IWOTH, "w"),),
                       ((stat.S_IXOTH | stat.S_ISVTX, "t"), (stat.S_ISVTX, "T"), (stat.S_IXOTH, "x")))


    def stat_filemode(mode):
        """
        Convert a file's mode to a string of the form '-rwxrwxrwx'.
        因为stat.filemode()是python 3.4版本后新增方法，搬运官方代码。为了避免门禁问题，修改
        """
        perm = ''
        for table in _filemode_table:
            for bit, char in table:
                if mode & bit == bit:
                    perm += char
                    break
            else:
                perm += '-'
        return perm


def transfer_to_linux_filemode(mode):
    """将文件权限表示从数字转换成人类易读的文本格式

    :param mode: int,数字表示的文件权限
    :return: str，例如'-rwxrwxrwx'
    """
    return stat_filemode(mode)


class FileMeta:
    """描述文件信息的类"""

    def __init__(self, filepath):
        file_meta = os.stat(filepath)
        # 文件大小
        self.size = file_meta.st_size
        # 文件权限
        self.permission = transfer_to_linux_filemode(file_meta.st_mode)

        # 文件所有者
        try:
            from pwd import getpwuid
            self.user = getpwuid(file_meta.st_uid).pw_name
        except ImportError:
            self.user = file_meta.st_uid

        # 文件所属用户组
        try:
            from grp import getgrgid
            self.user_group = getgrgid(file_meta.st_gid).gr_name
        except ImportError:
            self.user_group = file_meta.st_gid
        self.name = os.path.basename(filepath)

    def __repr__(self):
        return '<FileMeta>%s %s %s %sB %s' % (self.permission, self.user, self.user_group, self.size, self.name)
