#!/usr/bin/env python
# -*- coding: UTF-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""
    ACPluginAlarm class
"""
import os
import socket
import time
from subprocess import Popen

from oslo_config import cfg

from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil

LOG = ncu.ac_log.getLogger(__name__)

ALARM_TYPE_GENERATE = 0
ALARM_TYPE_RECOVERY = 1

ALARM_LEVEL_CRITICAL = 1
ALARM_LEVEL_MAJOR = 2
ALARM_LEVEL_MINOR = 3

MOC_NAME = 'CloudService' if ncu.after_or_equal_fsp_6_5_1() else 'network'
MOC_NAME_HOST = 'Host'
MOC_NAME_NETWORK = 'network'
AC_CLOUD_SERVICE = "OpenStack"
AC_MICRO_SERVICE = "AC_NEUTRON_PLUGIN"
AC_SERVICE = "neutron" if ncu.after_or_equal_fsp_6_5_1() else "ACDCN"

CERT_EXPIRED = 1
CERT_ABOUT_TO_EXPIRED = 2


def get_fs_host_id():
    """读取文件里保存的UUID"""
    uuid_file = os.path.realpath('/etc/uuid')
    if not os.path.exists(uuid_file):
        return None

    with open(os.path.realpath(uuid_file), 'r') as reader:
        result = reader.read().rstrip('\n')
        return result


class ACPluginAlarm(object):
    """alarm class"""

    @staticmethod
    def send_alarm(alarm_info):
        """send alarm"""
        if not ncu.IS_FSP:
            return
        try:
            cmd = ['sendAlarm', str(alarm_info['id']), str(alarm_info['alarm_type']), str(alarm_info['level']),
                   str(alarm_info['cause']), str(time.time()), str(alarm_info['moc_name']),
                   str(alarm_info['resource_id']), str(alarm_info['location']), str(alarm_info['addition'])]
            Popen(args=cmd)
            LOG.info("[AC]send alarm: %s", cmd)
        except Exception as ex:
            LOG.error('[AC] send alarm failed: %s', str(ex))

    @staticmethod
    def get_neutron_sync_fail_alarm(reason):
        """ get neutron sync fail alarm """
        resource_id = "OpenStack" if ncu.after_or_equal_fsp_6_5_1() else \
            "event=compare_data_between_AC_and_neutron"
        alarm_info = {
            'alarm_type': ALARM_TYPE_GENERATE, 'level': ALARM_LEVEL_MAJOR,
            'resource_id': resource_id,
            'location': ACPluginAlarm.get_neutron_sync_alarm_location(),
            'addition': ACPluginAlarm.get_neutron_sync_alarm_additional_info(
                reason), 'cause': '0', 'moc_name': MOC_NAME, 'id': '1240001'}

        return alarm_info

    @staticmethod
    def clear_comma_in_alarm_detail(reason):
        """ replace . by ; """
        return str(reason).replace(";", ".")

    @staticmethod
    def get_neutron_sync_alarm_location():
        """ get neutron sync alarm location """
        # Service;MicroService
        first_info = AC_CLOUD_SERVICE if ncu.after_or_equal_fsp_6_5_1() \
            else AC_SERVICE
        return first_info + ";" + AC_MICRO_SERVICE

    @staticmethod
    def get_neutron_sync_alarm_additional_info(reason):
        """ get neutron sync alarm additional info """
        # Service;MicroService;host_id;hostname;
        # local_address;peer_address;detail_info;
        host_id = get_fs_host_id()
        host_name = socket.gethostname()
        if not host_id:
            host_id = host_name
        local_address = ACCommonUtil.get_local_host_ip()
        peer_address = cfg.CONF.huawei_ac_agent_config.rpc_server_ip
        detail = ACPluginAlarm.clear_comma_in_alarm_detail(reason)
        if ncu.after_or_equal_fsp_6_5_1():
            additional_info = "%s;%s;%s;%s;%s;%s" % (
                AC_CLOUD_SERVICE, host_id, host_name, local_address,
                peer_address, detail)
        else:
            additional_info = "%s;%s;%s;%s;%s;%s;%s" % (
                AC_SERVICE, AC_MICRO_SERVICE, host_id, host_name,
                local_address, peer_address, detail)
        return additional_info

    @staticmethod
    def get_neutron_sync_recovery_alarm():
        """ get neutron_sync recovery alarm """
        resource_id = "OpenStack" if ncu.after_or_equal_fsp_6_5_1() else \
            "event=compare_data_between_AC_and_neutron"
        alarm_info = {
            'alarm_type': ALARM_TYPE_RECOVERY, 'level': ALARM_LEVEL_MAJOR,
            'resource_id': resource_id,
            'location': ACPluginAlarm.get_neutron_sync_alarm_location(),
            'addition': ACPluginAlarm.get_neutron_sync_alarm_additional_info(
                "between AC and neutron success"), 'cause': '0',
            'moc_name': MOC_NAME, 'id': '1240001'}
        return alarm_info

    @staticmethod
    def get_host_id():
        """get host id"""
        # host_id
        host_id = get_fs_host_id()
        if not host_id:
            return socket.gethostname()
        return host_id

    @staticmethod
    def get_connection_alarm_additional_info(reason):
        """ get connection alarm additional info """
        # Service;MicroService;hostname;local_address;peer_address;detail_info;
        host_name = socket.gethostname()
        local_address = ACCommonUtil.get_local_host_ip()
        peer_address = cfg.CONF.huawei_ac_agent_config.rpc_server_ip
        detail = ACPluginAlarm.clear_comma_in_alarm_detail(reason)
        if ncu.after_or_equal_fsp_6_5_1():
            additional_info = "%s;%s;%s;%s;%s" % (
                AC_CLOUD_SERVICE, host_name, local_address, peer_address,
                detail)
        else:
            additional_info = "%s;%s;%s;%s;%s;%s" % (
                AC_SERVICE, AC_MICRO_SERVICE, host_name,
                local_address, peer_address, detail)
        return additional_info

    @staticmethod
    def get_http_connection_fail_alarm(reason):
        """ http connection fail alarm """
        alarm_info = {'alarm_type': ALARM_TYPE_GENERATE,
                      'level': ALARM_LEVEL_CRITICAL}
        if ncu.after_or_equal_fsp_6_5_1():
            alarm_info['resource_id'] = ACPluginAlarm.get_host_id()
        else:
            alarm_info['resource_id'] = "event=" \
                                        "http_connection_between_AC_and_neutron"
        alarm_info['location'] = ACPluginAlarm.get_host_id()
        alarm_info['addition'] = \
            ACPluginAlarm.get_connection_alarm_additional_info(reason)
        alarm_info['cause'] = '0'
        alarm_info['moc_name'] = MOC_NAME_HOST \
            if ncu.after_or_equal_fsp_6_5_1() else MOC_NAME_NETWORK
        alarm_info['id'] = '1240002'
        return alarm_info

    @staticmethod
    def get_http_connection_recovery_alarm():
        """connection recovery alarm"""
        alarm_info = ACPluginAlarm.get_http_connection_fail_alarm(
            "between AC and neutron success")
        alarm_info['alarm_type'] = ALARM_TYPE_RECOVERY
        return alarm_info

    @staticmethod
    def get_websocket_connection_fail_alarm(reason):
        """get websocket connection fail alarm"""
        alarm_info = {'alarm_type': ALARM_TYPE_GENERATE,
                      'level': ALARM_LEVEL_CRITICAL}
        if ncu.after_or_equal_fsp_6_5_1():
            alarm_info['resource_id'] = ACPluginAlarm.get_host_id()
        else:
            alarm_info['resource_id'] = "event=websocket_connection_between_" \
                                        "AC_and_neutron"
        alarm_info['location'] = ACPluginAlarm.get_host_id()
        alarm_info['addition'] = \
            ACPluginAlarm.get_connection_alarm_additional_info(reason)
        alarm_info['cause'] = '0'
        alarm_info['moc_name'] = MOC_NAME_HOST \
            if ncu.after_or_equal_fsp_6_5_1() else MOC_NAME_NETWORK
        alarm_info['id'] = '1240003'
        return alarm_info

    @staticmethod
    def get_websocket_connection_recovery_alarm():
        """ websocket connection recovery alarm """
        alarm_info = ACPluginAlarm.get_websocket_connection_fail_alarm(
            "between AC and neutron success")
        alarm_info['alarm_type'] = ALARM_TYPE_RECOVERY
        return alarm_info

    @staticmethod
    def get_cert_alarm_additional_info(detail):
        """cert alarm additional info"""
        # Service;MicroService;hostname;local_address;peer_address;detail_info;
        host_name = socket.gethostname()
        local_address = ACCommonUtil.get_local_host_ip()
        peer_address = cfg.CONF.huawei_ac_agent_config.rpc_server_ip
        detail = ACPluginAlarm.clear_comma_in_alarm_detail(detail)
        additional_info = "%s;%s;%s;%s;%s" % (
            AC_CLOUD_SERVICE, host_name, local_address, peer_address, detail)
        return additional_info

    @staticmethod
    def get_cert_expired_alarm(cert_alarm_type, reason):
        """ cert alarm """
        alarm_info = get_cert_alarm_info(cert_alarm_type, reason)
        alarm_info['alarm_type'] = ALARM_TYPE_GENERATE
        return alarm_info

    @staticmethod
    def get_cert_recovery_alarm(cert_alarm_type, reason):
        """cert recovery alarm """
        alarm_info = get_cert_alarm_info(cert_alarm_type, reason)
        alarm_info['alarm_type'] = ALARM_TYPE_RECOVERY
        return alarm_info

    @classmethod
    def send_neutron_sync_alarm(cls, reason):
        """send neutron sync alarm"""
        alarm_info = cls.get_neutron_sync_fail_alarm(reason)
        cls.send_alarm(alarm_info)

    @classmethod
    def send_neutron_sync_recovery_alarm(cls):
        """send neutron sync recovery alarm"""
        alarm_info = cls.get_neutron_sync_recovery_alarm()
        cls.send_alarm(alarm_info)

    @staticmethod
    def get_port_alarm_info(alarm_type, port_id, reason):
        """ port error alarm info """
        alarm_info = {
            'alarm_type': alarm_type,
            'level': ALARM_LEVEL_MAJOR,
            'location': port_id,
            'addition': "%s;%s" % (port_id, reason),
            'id': '1240006'
        }
        return alarm_info


def get_cert_alarm_info(cert_alarm_type, reason):
    """cert alarm info """
    level = ALARM_LEVEL_CRITICAL if cert_alarm_type == CERT_EXPIRED else \
        ALARM_LEVEL_MAJOR
    moc_name = MOC_NAME_HOST if ncu.SUPPORT_VERIFY_CERT else MOC_NAME_NETWORK
    alarm_info = {
        'level': level,
        'resource_id': ACPluginAlarm.get_host_id(),
        'location': "%s;%s;%s" % (
            ACPluginAlarm.get_host_id(), reason['cert_type'],
            reason['plugin_type']),
        'addition': ACPluginAlarm.get_cert_alarm_additional_info(
            reason['detail']),
        'cause': '0', 'moc_name': moc_name,
        'id': '1240005' if cert_alarm_type == CERT_EXPIRED else '1240004'}
    return alarm_info
