#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.

# Function: deal different openstack version reference in this file

"""
neutron compatible util
"""

import json
import os
import socket
import string
from datetime import datetime
from distutils.sysconfig import get_python_lib

import OpenSSL
import netaddr
import requests
import six
from neutron import manager
from neutron.db import models_v2
from neutron.plugins.ml2 import models
from oslo_config import cfg

from networking_huawei.common.exceptions import CertExpiredException
from networking_huawei.drivers.ac.common import constants as ac_constants

try:
    from networking_huawei.drivers.ac.common import config
except ImportError:
    pass

try:
    from FSSecurity import crypt
except ImportError:
    pass

try:
    # logging for openstack
    from oslo_log import log as logging
except ImportError:
    # logging for fusionsphere
    from neutron.openstack.common import log as logging

try:
    from neutron.db.l3_db import Router, RouterPort, FloatingIP
except ImportError:
    from neutron.db.models.l3 import Router, RouterPort, FloatingIP
try:
    from neutron.db.l3_attrs_db import RouterExtraAttributes
except ImportError:
    from neutron.db.models.l3_attrs import RouterExtraAttributes
try:
    from neutron.common import utils
except ImportError:
    utils = None

try:
    from neutron.plugins.common.constants import \
        TYPE_FLAT, TYPE_GRE, TYPE_LOCAL, TYPE_VXLAN, TYPE_VLAN
except ImportError:
    from neutron_lib.constants import \
        TYPE_FLAT, TYPE_GRE, TYPE_LOCAL, TYPE_VXLAN, TYPE_VLAN

try:
    from neutron.extensions.portbindings import VNIC_TYPE, VIF_DETAILS, \
        VIF_TYPE_BINDING_FAILED, VIF_TYPE_UNBOUND, OVS_HYBRID_PLUG, \
        CAP_PORT_FILTER, VNIC_DIRECT, PROFILE
except ImportError:
    from neutron_lib.api.definitions.portbindings import VNIC_TYPE, \
        VIF_DETAILS, VIF_TYPE_BINDING_FAILED, VIF_TYPE_UNBOUND, \
        OVS_HYBRID_PLUG, CAP_PORT_FILTER, VNIC_DIRECT, PROFILE

try:
    from neutron.common.constants import DEVICE_OWNER_ROUTER_INTF, \
        DEVICE_OWNER_DVR_INTERFACE, DEVICE_OWNER_ROUTER_HA_INTF, \
        FLOATINGIP_STATUS_ACTIVE, DEVICE_OWNER_FLOATINGIP, \
        PORT_STATUS_NOTAPPLICABLE, FLOATINGIP_STATUS_DOWN
except ImportError:
    from neutron_lib.constants import DEVICE_OWNER_ROUTER_INTF, \
        DEVICE_OWNER_DVR_INTERFACE, DEVICE_OWNER_ROUTER_HA_INTF, \
        FLOATINGIP_STATUS_ACTIVE, DEVICE_OWNER_FLOATINGIP, \
        PORT_STATUS_NOTAPPLICABLE, FLOATINGIP_STATUS_DOWN

try:
    from neutron.plugins.ml2 import driver_api as api
except ImportError:
    from neutron_lib.plugins.ml2 import api

try:
    from neutron import context as neutron_context
except ImportError:
    from neutron_lib import context as neutron_context

try:
    # try to import neutron callbacks
    from neutron.callbacks import events
    from neutron.callbacks import registry
    from neutron.callbacks import resources
except ImportError:
    # there is no neutron callbacks in fusionsphere
    from neutron_lib.callbacks import events
    from neutron_lib.callbacks import registry
    from neutron_lib.callbacks import resources

try:
    # K/L/M/FSP6.0/FSP6.3.0/FSP6.3.1
    from neutron.plugins.ml2.models import NetworkSegment
except ImportError:
    try:
        # N
        from neutron.db.segments_db import NetworkSegment
    except ImportError:
        # O/P/Q/R/FSP6.5
        from neutron.db.models.segment import NetworkSegment

try:
    from neutron.db import common_db_mixin as base_db
except ImportError:
    from networking_huawei.drivers.ac.common import common_db_mixin as base_db

IS_FSP = cfg.CONF.huawei_ac_config.OPS_version.startswith('FusionSphere')

HOST_VERSION = utils.get_host_version() \
    if hasattr(utils, 'get_host_version') else None

FS_CERT_FILE = None
FS_KEY_FILE = None
FS_PRIVATE_KEY_PASSWORD = None

ac_log = logging
DEVICE_OWNER_ROUTER_GW = "network:router_gateway"
DEVICE_OWNER_DHCP = "network:dhcp"
DEVICE_OWNER_COMPUTE_PREFIX = "compute:"
ac_ml2_api = api
LOG = logging.getLogger(__name__)

NEUTRON_READ_DICT = {
    ac_constants.NW_HW_NETWORKS: models_v2.Network,
    ac_constants.NW_HW_SUBNETS: models_v2.Subnet,
    ac_constants.NW_HW_PORTS: models_v2.Port,
    ac_constants.NW_HW_ROUTERS: Router,
    ac_constants.NW_HW_ROUTER_EXTRA_ATTRIBUTES: RouterExtraAttributes,
    ac_constants.NW_HW_ROUTER_IF: RouterPort,
    ac_constants.NW_HW_FIP: FloatingIP
}
CPS_INFO = {}
CPS_CERT_FILE = os.path.realpath('/etc/FSSecurity/server-cert/neutron_ca.crt')
CPS_GET_EXT_PARAM_URL = \
    "/cps/v1/template/params/neutron/neutron-server?commit_state=commited"
CPS_ADD_EXT_PARAM_URL = '/cps/v1/template/params'
CPS_COMMIT_URL = '/cps/v1/commit'

CODE_MAIN_NAME = 'networking_huawei'
PACKAGE_HOME_PATH = get_python_lib()
HTTPS_CA_CERT = '%s/%s/%s' % (PACKAGE_HOME_PATH, CODE_MAIN_NAME,
                              'drivers/ac/client/ssl_cacert.pem')
HTTPS_CRL = '%s/%s/%s' % (PACKAGE_HOME_PATH, CODE_MAIN_NAME,
                          'drivers/ac/client/https.crl')
WEBSOCKET_CA_CERT = '%s/%s/%s' % \
                    (PACKAGE_HOME_PATH, CODE_MAIN_NAME,
                     'drivers/ac/ac_agent/rpc/websocket/trust.cer')
WEBSOCKET_CLIENT_CERT = '%s/%s/%s' % \
                        (PACKAGE_HOME_PATH, CODE_MAIN_NAME,
                         'drivers/ac/ac_agent/rpc/websocket/client.cer')
WEBSOCKET_CRL = '%s/%s/%s' % \
                (PACKAGE_HOME_PATH, CODE_MAIN_NAME,
                 'drivers/ac/ac_agent/rpc/websocket/websocket.crl',)


def get_service_plugin():
    """ get service plugin """
    if get_ops_version() in ac_constants.OPS_VERSION_O_PQRTW_6_21:
        from neutron_lib.plugins import directory
        service_plugins = directory.get_plugins()
    else:
        service_plugins = manager.NeutronManager.get_service_plugins()
    return service_plugins


def get_core_plugin():
    """ get core plugin """
    if get_ops_version() in ac_constants.OPS_VERSION_O_PQRTW_6_21:
        from neutron_lib.plugins import directory
        core_plugin = directory.get_plugin()
    else:
        core_plugin = manager.NeutronManager.get_plugin()
    return core_plugin


def get_ops_version():
    """ get ops version """
    ops_version = cfg.CONF.huawei_ac_config.OPS_version
    if ops_version == ac_constants.FSP_6_5_private:
        return ac_constants.FSP_6_3_1
    if ops_version == ac_constants.FSP_6_5_NFVI:
        return ac_constants.FSP_6_5
    if IS_FSP and HOST_VERSION:
        host_version = '.'.join(HOST_VERSION.split('.')[:-1])
        if host_version.startswith('21.') or host_version.startswith('22.'):
            ops_version = ac_constants.FSP_21_0
        elif host_version >= ac_constants.DECOUPLING_FSP_VERSION:
            ops_version = 'FusionSphere' + ac_constants.DECOUPLING_FSP_VERSION
    return ops_version


SUPPORT_VERIFY_CERT = get_ops_version() in [ac_constants.FSP_8_0_0,
                                            ac_constants.FSP_8_0_3,
                                            ac_constants.FSP_21_0]


def get_cps_header_and_url(base_url):
    """ get cps header and url """
    headers = {"Content-type": "application/json",
               "X-Auth-User": CPS_INFO.get('cps_user'),
               "X-Auth-Password": crypt.decrypt(CPS_INFO.get('cps_pwd'))}
    url = CPS_INFO.get('cps_server_ip', '') + base_url
    return headers, url


def cps_commit():
    """ cps commit """
    headers, url = get_cps_header_and_url(CPS_COMMIT_URL)
    body = {'timeout': 60}
    requests.post(url, headers=headers, data=json.dumps(body),
                  verify=CPS_CERT_FILE)


def get_neutron_ext_param(param_name):
    """ get neutron ext param """
    headers, url = get_cps_header_and_url(CPS_GET_EXT_PARAM_URL)
    res = requests.get(url, headers=headers, verify=CPS_CERT_FILE)
    params = res.json()['params']
    return params.get(param_name)


def add_neutron_ext_param(param):
    """ add neutron ext param """
    headers, url = get_cps_header_and_url(CPS_ADD_EXT_PARAM_URL)
    body = {
        'service_name': 'neutron',
        'template_name': 'neutron-server',
        'params': param
    }
    requests.post(url, data=json.dumps(body), headers=headers,
                  verify=CPS_CERT_FILE)
    cps_commit()


def is_cert_expired(pem):
    """ is cert expired """
    current_time = datetime.utcnow()
    return current_time > get_cert_expired_time(pem)


def get_cert_expired_time(cer):
    """ get cert expired time """
    cert = OpenSSL.crypto.load_certificate(OpenSSL.crypto.FILETYPE_PEM, cer)
    timestamp = cert.get_notAfter()
    return datetime.strptime(timestamp.decode()[0:-1], '%Y%m%d%H%M%S')


def verify_san(server_cert):
    """ verify san """
    flag = False
    host = cfg.CONF.huawei_ac_agent_config.rpc_server_ip
    san_info = _get_subject_alt_name(server_cert)
    if san_info:
        for host_ip in host.strip().split(','):
            for peer_ip in san_info:
                flag = flag | match_ip(host_ip, peer_ip)
        return flag
    return True


def _get_subject_alt_name(server_cert):
    """ get san info from cert """
    ip_list = []
    cert = OpenSSL.crypto.load_certificate(OpenSSL.crypto.FILETYPE_ASN1,
                                           server_cert)
    for count in six.moves.range(cert.get_extension_count()):
        ext_subj = cert.get_extension(count)
        add_san_ip_to_ip_list(ext_subj, ip_list)
    return ip_list


def add_san_ip_to_ip_list(ext_subj, ip_list):
    """ add san ip to ip list """
    if ext_subj.get_short_name() == 'subjectAltName':
        san = ext_subj.__str__().split(', ')
        for ele in san:
            if "ip address" in ele.lower():
                server_ip = \
                    ele.lower().replace('\n', '').split('ip address:')[1]
                ip_list.append(server_ip)
    return ip_list


def is_revoked(crl_file, cert_content):
    """ whether cert is on crl """
    if not os.path.exists(crl_file):
        LOG.info("no such crl file: %s" % crl_file)
        return False
    with open(crl_file, 'r') as crl_f:
        crl_content = crl_f.read()
    if not crl_content or not cert_content:
        LOG.info("crl or cert is empty")
        return False
    crl = OpenSSL.crypto.load_crl(OpenSSL.crypto.FILETYPE_PEM, crl_content)
    cert = OpenSSL.crypto.load_certificate(OpenSSL.crypto.FILETYPE_PEM,
                                           cert_content)
    for revoked_obj in crl.get_revoked():
        if cert.get_serial_number() == int(revoked_obj.get_serial(), 16):
            return True
    return False


def match_ip(ip1, ip2):
    """ match ip """
    if netaddr.valid_ipv4(ip1) and netaddr.valid_ipv4(ip2):
        return socket.inet_pton(socket.AF_INET, ip1) == \
            socket.inet_pton(socket.AF_INET, ip2)
    elif netaddr.valid_ipv6(ip1) and netaddr.valid_ipv6(ip2):
        return socket.inet_pton(socket.AF_INET6, ip1) == \
            socket.inet_pton(socket.AF_INET6, ip2)
    return False


def after_fsp_6_3_0():
    """ whether fsp version is after 6.3.0"""
    return get_ops_version() in [ac_constants.FSP_6_3_1,
                                 ac_constants.FSP_6_5,
                                 ac_constants.FSP_6_5_1,
                                 ac_constants.FSP_8_0_0,
                                 ac_constants.FSP_8_0_3,
                                 ac_constants.FSP_21_0]


def after_or_equal_fsp_6_5_1():
    """ whether fsp version is after of equal 6.5.1"""
    return get_ops_version() in [ac_constants.FSP_6_5_1,
                                 ac_constants.FSP_8_0_0,
                                 ac_constants.FSP_8_0_3,
                                 ac_constants.FSP_21_0]


def verify_plugin_cert(certs, connection=None):
    """ verify plugin cert"""
    for cert_type, cert_path in six.iteritems(certs):
        if cert_type != 'key_file':
            with open(cert_path, "rb") as cert:
                cer = cert.read()
            if is_cert_expired(cer):
                LOG.error("plugin %s %s is expired" % (connection, cert_type))
                raise CertExpiredException()


def need_ac_status(driver):
    """ Check the resource need present status or not

    :param driver: the extension driver of the resource
    :return: True if need present status
    """
    if IS_FSP and driver in cfg.CONF.ml2.extension_drivers:
        return True
    return False


def is_gw_port(port):
    """check gw port"""
    device_owner = port.get('device_owner')
    if device_owner == DEVICE_OWNER_ROUTER_GW:
        return True
    return False


# check the port if trunk parent port or not
def is_trunk_parent_port(port):
    """check the port if trunk parent port or not"""
    if not IS_FSP:
        return False
    if "trunk_details" in port and port['trunk_details'] is not None:
        return True
    return False


def is_valid_result_filename(filepath):
    """ Function to validate the compare result filename.

    :param filepath: filename to be validated.
    :return: True/False, msg[reason for failure]
    """
    safechars = string.ascii_letters + string.digits + "-_."
    if not os.path.dirname(filepath):
        return False, 'Invalid COMPARE_RESULT_FILE_NAME.'
    filename = os.path.basename(filepath)
    if filename.startswith('-') or filename.startswith('.'):
        return False, 'COMPARE_RESULT_FILE_NAME is not allowed to start ' \
                      'with \'-\' or \'.\' characters.'
    str_len = len(filename)
    if str_len > 255:
        return False, 'Length of COMPARE_RESULT_FILE_NAME is greater ' \
                      'than 255.'
    ext = os.path.splitext(filename)[1][1:]
    if ext != 'csv':
        return False, 'COMPARE_RESULT_FILE_NAME is not in .csv format.'
    for character in filename:
        if character not in safechars:
            return False, 'Invalid characters in COMPARE_RESULT_FILE_NAME.'
    return True, None


def get_port_binding(session, port_id):
    """从PortBinding表中获取port对应的binding信息，改动原因为从Fsp8.0到Fsp 22.1，PortBinding表主键从
    单一的port_id主键变更到了port_id+host主键，并且新增了status字段

    :param session: context session
    :param port_id: port_id
    :return: port_binding
    """
    port_bindings = session.query(models.PortBinding).filter_by(
        port_id=port_id).all()
    try:
        from neutron_lib import constants as neutron_constants
        for port_binding in port_bindings:
            if port_binding.get('status') == neutron_constants.ACTIVE:
                return port_binding
    except ImportError:
        pass
    return port_bindings[0] if port_bindings else None
