#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""osprofiler warp"""
import functools

from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

try:
    from osprofiler import profiler as ac_profiler
except ImportError:
    pass
from oslo_utils import timeutils

LOG = ncu.ac_log.getLogger(__name__)


def _get_utc_time():
    return str(timeutils.utcnow()).split(".")[0]


def _format_content(content):
    return "<AC>{} {}<AC>".format(_get_utc_time(), content)


def _format_exception_content(content):
    return "<AC-ERROR>{} {}<AC-ERROR>".format(_get_utc_time(), content)


def support_chain(func):
    """function chain decorator"""

    @functools.wraps(func)
    def _wrap(*args, **kwargs):
        if ncu.get_ops_version() in [ac_constants.FSP_6_3_0] or \
                ncu.after_fsp_6_3_0():
            return func(*args, **kwargs)
        return None

    return _wrap


def suppress_exception(func):
    """suppress exception decorator"""

    @functools.wraps(func)
    def _wrap(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as ex:
            LOG.error("[AC] call %s failed: %s", func.__name__, ex)

    return _wrap


@support_chain
@suppress_exception
def record_call_chain(content):
    """record_call_chain"""
    ac_profiler.trace_ext_info(_format_content(content))


@support_chain
@suppress_exception
def record_chain_exception(exp):
    """record_chain_exception"""
    exp = exp.split("error:")[1].strip() if "error:" in exp else exp
    ac_profiler.trace_ext_info(_format_exception_content(exp))


@support_chain
@suppress_exception
def record_chain_end():
    """record_chain_end"""
    ac_profiler.trace_ext_info(_format_content("AC plugin end"))


@support_chain
@suppress_exception
def record_chain_start(content):
    """record_chain_start"""
    ac_profiler.trace_ext_info(_format_content("AC plugin start. " + content))


@support_chain
@suppress_exception
def record_chain_end_with_reason(reason):
    """record_chain_end_with_reason"""
    ac_profiler.trace_ext_info(_format_content(reason + " AC plugin end"))


@support_chain
@suppress_exception
def record_chain_exception_end(reason):
    """record_chain_exception_end"""
    reason = reason.split("error:")[1].strip() if "error:" in reason else reason
    ac_profiler.trace_ext_info(_format_exception_content(
        reason + " AC plugin end"))
