#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Bgp-route db"""

import copy
import base64
import six
import netaddr

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound
from sqlalchemy.orm import exc
from oslo_log import log as logging
from oslo_utils import uuidutils
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import security_util
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.db.bgp_route import schema
from networking_huawei.drivers.ac.db.decrypt_factor.decrypt_factor_db import \
    DecryptFactorDbMixin
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.encode_convert import convert_to_bytes

LOG = logging.getLogger(__name__)


class BgpRouteNotFound(NotFound):
    """BgpRouteNotFound class"""
    message = "Bgp route %(bgp_route_id)s could not be found."


class BgpRouteDbMixin(ncu.base_db.CommonDbMixin):
    """Bgp route DB
    """

    def __init__(self):
        self.dfactor_db = DecryptFactorDbMixin()

    def _get_bgp_route_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except exc.NoResultFound:
            raise BgpRouteNotFound(bgp_route_id=v_id)

    @classmethod
    def create_bgp_route_dict(cls, bgp_route_db):
        """call by BgpRouteDbMixin.make_bgp_route_dict,
        BgpNeighborDbMixin._make_bgpneighbor_dict

        :param bgp_route_db: bgp_route
        :return: dict
        """
        res = {'id': bgp_route_db['id'],
               'tenant_id': bgp_route_db['tenant_id'],
               'name': bgp_route_db['name'],
               'description': bgp_route_db['description'],
               'router_id': bgp_route_db['router_id'],
               'router_interface_id': bgp_route_db['router_interface_id'],
               'peer_ip_address': bgp_route_db['peer_ip_address'],
               'peer_as_number': bgp_route_db['peer_as_number']}
        if 'router_interface_id' in bgp_route_db:
            res['router_interface_id'] = bgp_route_db['router_interface_id']
        if 'ip_version' in bgp_route_db:
            res['ip_version'] = bgp_route_db['ip_version']
        if 'bfd' in bgp_route_db:
            res['bfd'] = bgp_route_db['bfd']
        if 'source_ips' in bgp_route_db:
            res['source_ips'] = bgp_route_db['source_ips']
        if 'peer_type' in bgp_route_db:
            res['peer_type'] = bgp_route_db['peer_type']
        if 'session_attribute' in bgp_route_db:
            res['session_attribute'] = bgp_route_db['session_attribute']
        return res

    def _make_bgp_route_dict(self, bgp_route_db, fields=None):
        """make bgp route"""
        return self._fields(self.create_bgp_route_dict(bgp_route_db), fields)

    def create_db_bgp_route(self, context, bgp_route):
        """the function to create bgp route"""
        bgp_route_log = copy.deepcopy(bgp_route)['bgp_route']
        bgp_route_log = validate.validate_log_record(
            bgp_route_log, ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI('[AC]Create bgp route in NeutronDB for %s.'),
                 bgp_route_log)
        bgp_route_info = self._init_bgp_route_info(bgp_route, context)

        with context.session.begin(subtransactions=True):
            bgp_route_db = schema.ACBgpRouteSchema(
                id=bgp_route_info.get('id'),
                tenant_id=bgp_route_info['tenant_id'],
                name=bgp_route_info['name'],
                description=bgp_route_info['description'],
                router_id=bgp_route_info['router_id'],
                ip_version=bgp_route_info['ip_version'],
                router_interface_id=bgp_route_info['router_interface_id'],
                peer_ip_address=bgp_route_info['peer_ip_address'],
                peer_as_number=bgp_route_info['peer_as_number'],
                bfd=bgp_route_info['bfd'],
                session_attribute=bgp_route_info['session_attribute'],
                source_ips=bgp_route_info['source_ips'],
                peer_type=bgp_route_info['peer_type'])
            context.session.add(bgp_route_db)
            context.session.flush()

        return self._make_bgp_route_dict(
            self._decrypt_auth(context, bgp_route_db))

    def _init_bgp_route_info(self, bgp_route, context):
        result = bgp_route['bgp_route']
        if not result.get('id'):
            result['id'] = uuidutils.generate_uuid()
        if not result.get('description'):
            result['description'] = None
        if not result.get('router_interface_id'):
            result['router_interface_id'] = ""
        peer_ip = result.get('peer_ip_address', None)
        self._init_bgp_route_ip_version(peer_ip, result)
        if not result.get('bfd'):
            result['bfd'] = None
        if not result.get('session_attribute'):
            result['session_attribute'] = None
        else:
            # if auth type is password, need to encrypt the password text
            # and save the decrypt factor in db huawei_ac_decrypt_factor
            if result['session_attribute'].get('auth'):
                auth = result['session_attribute']['auth']
                if ('auth_type' in auth) and auth['auth_type'] == 'password':
                    pw_text = auth['password_text']
                    data_encrypt = self.dfactor_db.create_db_decrypt_factor(
                        context, result.get('id'), pw_text)
                    auth['password_text'] = data_encrypt

        if not result.get('source_ips'):
            result['source_ips'] = []

        if not result.get('peer_type'):
            result['peer_type'] = None
        return result

    @classmethod
    def _init_bgp_route_ip_version(cls, peer_ip, result):
        """call by _init_bgp_route_info:处理bgp"""
        if result.get('ip_version') == ac_constants.VERSION_IPV4:
            if netaddr.valid_ipv6(peer_ip):
                LOG.error(_LE('[AC]Peer ip address:%s,is not ipv4.'), peer_ip)
                raise ValueError
        elif result.get('ip_version') == ac_constants.VERSION_IPV6:
            if netaddr.valid_ipv4(peer_ip):
                LOG.error(_LE('[AC]Peer ip address : %s,is not ipv6.'), peer_ip)
                raise ValueError
        else:
            if netaddr.valid_ipv6(peer_ip):
                result['ip_version'] = '6'
            if netaddr.valid_ipv4(peer_ip):
                result['ip_version'] = '4'

    def _set_session_attribute(self, context, bgp_route_id, bgp_route,
                               bgp_route_db):
        session_attribute = bgp_route['session_attribute']
        if session_attribute.get('auth'):
            auth = session_attribute['auth']
            if auth.get('auth_type') == 'password':
                encrypt_text = self.dfactor_db.update_db_decrypt_factor(
                    context, bgp_route_id, auth['password_text'])
                auth['password_text'] = encrypt_text

        if bgp_route_db.get('session_attribute'):
            session_attribute_db = bgp_route_db['session_attribute']
            if 'keepalive_time' in session_attribute_db and 'keepalive_time' \
                    not in session_attribute:
                session_attribute['keepalive_time'] = \
                    session_attribute_db['keepalive_time']
            if 'hold_time' in session_attribute_db and 'hold_time' \
                    not in session_attribute:
                session_attribute['hold_time'] = \
                    session_attribute_db['hold_time']
            if 'suppress' in session_attribute_db and 'suppress' \
                    not in session_attribute:
                session_attribute['suppress'] = \
                    session_attribute_db['suppress']
            if 'auth' in session_attribute_db and 'auth' \
                    not in session_attribute:
                session_attribute['auth'] = \
                    session_attribute_db['auth']

    def update_db_bgp_route(self, context, bgp_route_id, bgp_route):
        """the function to update bgp route"""
        LOG.info(_LI('[AC] Update bgp route in Neutron DB '
                     'by id: %s.'), bgp_route_id)
        bgp_route = bgp_route['bgp_route']
        with context.session.begin(subtransactions=True):
            bgp_route_db = context.session.query(schema.ACBgpRouteSchema). \
                filter_by(id=bgp_route_id).first()
            if bgp_route.get('session_attribute'):
                self._set_session_attribute(
                    context, bgp_route_id, bgp_route, bgp_route_db)
            if bgp_route_db:
                bgp_route_db.update(bgp_route)
        return self._make_bgp_route_dict(
            self._decrypt_auth(context, bgp_route_db))

    def delete_db_bgp_route(self, context, bgp_route_id):
        """the function to delete bgp route"""
        LOG.info(_LI('[AC] Delete bgp route in Neutron '
                     'DB by id: %s.'), bgp_route_id)
        with context.session.begin(subtransactions=True):
            bgp_route_db = self._get_bgp_route_resource(
                context, schema.ACBgpRouteSchema, bgp_route_id)
            context.session.delete(bgp_route_db)
            self.dfactor_db.delete_decrypt_factor(
                context, bgp_route_id)

    def delete_db_bgp_routes_by_router(self, context, router_id):
        """the function to delete bgp routes by router"""
        LOG.info(_LI('[AC] Delete bgp route in Neutron '
                     'DB by router id: %s.'), router_id)
        query_dbs = context.session. \
            query(schema.ACBgpRouteSchema). \
            filter_by(router_id=router_id).all()
        if query_dbs:
            bgp_route_ids = [q.id for q in query_dbs]
            for bgp_route_id in bgp_route_ids:
                self.delete_db_bgp_route(context, bgp_route_id)
                self.dfactor_db.delete_decrypt_factor(context, bgp_route_id)

    def get_db_bgp_routes(self, context, filters=None, fields=None,
                          need_decrypt=True):
        """the function to get bgp routes"""
        LOG.info(_LI("[AC] Get bgp routes from Neutron DB."))
        bgp_routes = self._get_collection(context, schema.ACBgpRouteSchema,
                                          self._make_bgp_route_dict,
                                          filters=filters, fields=fields)
        if not need_decrypt:
            return bgp_routes
        return [self._decrypt_auth(context, bgp_r) for bgp_r in bgp_routes]

    def get_db_bgp_route(self, context, bgp_route_id, fields=None,
                         need_decrypt=True):
        """the function to get bgp route"""
        LOG.info(_LI('[AC] Get bgp route from Neutron DB by id: %s.'),
                 bgp_route_id)
        bgp_route_db = context.session.query(schema.ACBgpRouteSchema).filter_by(
            id=bgp_route_id).first()
        if bgp_route_db:
            decrypt_bgp_route = self._decrypt_auth(context, bgp_route_db) \
                if need_decrypt else bgp_route_db
            return self._make_bgp_route_dict(decrypt_bgp_route, fields=fields)
        raise BgpRouteNotFound(bgp_route_id=bgp_route_id)

    def _decrypt_auth(self, context, bgp_route_db):
        modify_bgp_route_db = bgp_route_db
        if not bgp_route_db:
            return modify_bgp_route_db
        if bgp_route_db.get('session_attribute'):
            session_attr = bgp_route_db['session_attribute']
            auth = session_attr.get('auth')
            if auth and auth.get('auth_type') == 'password':
                pw_text = auth['password_text']
                decrypt_dict = self.dfactor_db.get_db_decrypt_factor(
                    context, bgp_route_db['id'])
                pw_key = convert_to_bytes(decrypt_dict['pw_key'])
                pw_iv = convert_to_bytes(decrypt_dict['pw_iv'])
                key_iv = convert_to_bytes(decrypt_dict['key_iv'])
                data_decrypt = six.text_type(security_util.decrypt_data(
                    pw_text, pw_key=base64.b64decode(pw_key),
                    pw_iv=base64.b64decode(pw_iv),
                    key_iv=base64.b64decode(key_iv)
                ), errors='ignore')
                modify_bgp_route_db['session_attribute'][
                    'auth']['password_text'] = data_decrypt
        return modify_bgp_route_db
