#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""compare result db"""

from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log as logging
from networking_huawei.drivers.ac.db.compare_result import schema

try:
    from neutron.db import common_db_mixin
except ImportError:
    from networking_huawei.drivers.ac.common import common_db_mixin

LOG = logging.getLogger(__name__)


class CompareDbMixin(common_db_mixin.CommonDbMixin):
    """ Compare Db Mixin """

    def _make_inconsistent_db_dict(self, inconsistent_db, fields=None):
        if inconsistent_db:
            res = {
                'id': inconsistent_db['id'],
                'tenant_name': inconsistent_db['tenant_name'],
                'res_name': inconsistent_db['res_name'],
                'res_type': inconsistent_db['res_type'],
                'create_time': inconsistent_db['create_time'],
                'update_time': inconsistent_db['update_time'],
                'status': inconsistent_db['status']
            }
            return self._fields(res, fields)
        return None

    def create_db_inconsistent(self, context, inconsistent):
        """创建inconsistent

        :param context: context
        :param inconsistent: inconsistent
        :return: dict
        """
        LOG.info("[AC] Create data in neutron sync compare db, data: %s", inconsistent)
        with context.session.begin(subtransactions=True):
            uuid = inconsistent['Id']
            inconsistent_db = schema.ACCompareResultSchema(
                id=uuid,
                tenant_name=inconsistent.get('TenantName', ''),
                res_name=inconsistent.get('Name', ''),
                res_type=inconsistent.get('Resource', ''),
                create_time=inconsistent.get('CreateTime', ''),
                update_time=inconsistent.get('UpdateTime', ''),
                status=inconsistent.get('Status', '')
            )
            context.session.add(inconsistent_db)
            context.session.flush()
        return self._make_inconsistent_db_dict(inconsistent_db)

    @staticmethod
    def create_db_inconsistents(context, inconsistents):
        """创建inconsistents

        :param context: context
        :param inconsistents: list
        """
        LOG.info("[AC] Create data in neutron sync compare db, len: %s", len(inconsistents))
        data_list = list()
        for inconsistent in inconsistents:
            uuid = inconsistent['Id']
            inconsistent_db = schema.ACCompareResultSchema(
                id=uuid,
                tenant_name=inconsistent.get('TenantName', ''),
                res_name=inconsistent.get('Name', ''),
                res_type=inconsistent.get('Resource', ''),
                create_time=inconsistent.get('CreateTime', ''),
                update_time=inconsistent.get('UpdateTime', ''),
                status=inconsistent.get('Status', '')
            )
            data_list.append(inconsistent_db)
        with context.session.begin(subtransactions=True):
            context.session.bulk_save_objects(data_list)
            context.session.flush()

    @classmethod
    def delete_db_compare_results(cls, context):
        """清空compare_db表

        :param context: context
        """
        compare_results_query = context.session.query(schema.ACCompareResultSchema)
        LOG.info("[AC] Delete neutron sync compare db, len: %s", len(compare_results_query.all()))
        compare_results_query.delete()

    def get_db_compare_results_query(self, context, **kwargs):
        """条件查询

        :param context: context
        :param kwargs: params
        :return: a list of dicts
        """
        LOG.info(_LI("[AC] Get compare results from Neutron DB."))
        filters = kwargs.get('filters')
        fields = kwargs.get('fields')
        sorts = kwargs.get('sorts')
        limit = kwargs.get('limit')
        marker_obj = kwargs.get('marker_obj')
        page_reverse = kwargs.get('page_reverse')
        return self._get_collection(context, schema.ACCompareResultSchema,
                                    self._make_inconsistent_db_dict,
                                    filters=filters, fields=fields, sorts=sorts,
                                    limit=limit, marker_obj=marker_obj,
                                    page_reverse=page_reverse)
