# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能：AC配置数据库模型
| 版本：2022-04-08 10:57 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import sqlalchemy as sa

from oslo_utils import uuidutils

try:
    from neutron.db import model_base
except ImportError:
    from neutron_lib.db import model_base


class ACBasicConfig(model_base.BASEV2):
    """AC配置信息数据库模型"""
    __tablename__ = 'huawei_ac_basic_config'

    id = sa.Column(sa.String(36), primary_key=True, default=uuidutils.generate_uuid)
    # 配置文件名
    config_file = sa.Column(sa.String(255), nullable=False)
    # 文件所有者
    owner = sa.Column(sa.String(128), nullable=False)
    # 文件所属用户组
    user_group = sa.Column(sa.String(128), nullable=False)
    # 文件权限
    right = sa.Column(sa.String(10), nullable=False)
    # 文件内容
    config_value = sa.Column(sa.PickleType(protocol=2), default={})
    # 创建时间
    create_time = sa.Column(sa.DateTime, default=sa.func.now())

    def __repr__(self):
        return str(self.to_dict())

    @classmethod
    def prefix_list(cls):
        """config_value 里面包含的一些特殊的列表类型字段，真正的原文是去掉该前缀的键对应的字符串。

        :return: str
        """
        return '__list_'

    @classmethod
    def get_list_key(cls, key):
        """获取原字符串对应特殊列表对应的键名

        :param key: str,原始键名
        :return: str
        """
        return '%s%s' % (cls.prefix_list(), key)

    @classmethod
    def from_dict(cls, ac_config):
        """从字典中生成模型

        :param ac_config: dict
        :return: ACBasicConfig
        """
        return ACBasicConfig(
            config_file=ac_config.get('config_file'),
            owner=ac_config.get('owner'),
            user_group=ac_config.get('user_group'),
            right=ac_config.get('right'),
            config_value=ac_config.get('config_value', {}), )

    def to_dict(self):
        """模型转化成字典

        :return: dict
        """
        return {'id': self.id, 'config_file': self.config_file, 'owner': self.owner, 'user_group': self.user_group,
                'right': self.right, 'create_time': self.create_time, 'config_value': self.config_value}

    def show(self):
        """前端显示，过滤掉特殊字段

        :return: dict
        """
        result = {'id': self.id, 'config_file': self.config_file, 'owner': self.owner, 'user_group': self.user_group,
                  'right': self.right, 'create_time': str(self.create_time)}
        try:
            config_value = {}
            for key, value in self.config_value.items():
                # 显示的时候过滤掉
                if key.startswith(self.prefix_list()):
                    continue
                config_value[key] = value
        except AttributeError:
            config_value = self.config_value
        result['config_value'] = config_value
        return result
