#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""exroute"""
import re
import traceback

import netaddr
import sqlalchemy as sa
from oslo_serialization import jsonutils

try:
    from neutron.db import model_base
except ImportError:
    from neutron_lib.db import model_base
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log as logging
from networking_huawei.drivers.ac.extensions.exroute import l3_exroutes
from networking_huawei.drivers.ac.extensions.exroute.l3_exroutes import InvalidParam

LOG = logging.getLogger(__name__)


def simple_deserializer(data, deserializer=jsonutils.loads, default_value=None):
    """简易字符串解析工具

    :param data: str or none,json 字符串
    :param deserializer: function,具体反序列化函数
    :param default_value: None or int,当data为None返回该值，默认值为None
    :return: int or None
    """
    if data is None:
        return default_value
    if deserializer == jsonutils.loads:
        if not isinstance(data, (str, bytes, bytearray)):
            return data
    return deserializer(data)


def bool_deserializer(data, default_value=None, name=None):
    """bool类型反序列化

    :param data: str or None,字符串
    :param default_value:None or int,当data为None返回该值，默认值为None
    :param name: str,参数名
    :return: bool or None
    """
    if data is None:
        return default_value
    if isinstance(data, bool):
        return data
    if isinstance(data, str):
        data_lower = data.lower()
        if data_lower == 'true':
            return True
        elif data_lower == 'false':
            return False
    raise InvalidParam(name=name, value=data)


def validate_ex_route_destination(ex_route):
    """检验exRoute的destination字段

    :param ex_route: dict,ex-route信息
    :return:str,标准化后的destination
    """
    destination = ex_route.get('destination')
    if not destination:
        raise l3_exroutes.ParamNotSpecified(param='destination', exroute=ex_route)

    try:
        std_destination = netaddr.IPNetwork(destination)
        destination = '%s/%s' % (std_destination.network, std_destination.prefixlen)
    except netaddr.AddrFormatError as e:
        LOG.error('ex_route.destination(%s) is valid:%s', destination, traceback.format_exc())
        raise l3_exroutes.ParamInvalid(param='destination', exroute=ex_route)

    if netaddr.valid_ipv4(destination) or netaddr.valid_ipv6(destination):
        raise l3_exroutes.ParamInvalid(param='destination', exroute=ex_route)

    return destination


def validate_ex_route_nexthop(ex_route):
    """检验exRoute的 nexthop 字段

    :param ex_route: dict,ex-route信息
    :return:str,标准化后的nexthop
    """
    next_hop = ex_route.get('nexthop')
    if not next_hop:
        raise l3_exroutes.ParamNotSpecified(param='nexthop', exroute=ex_route)

    if not (netaddr.valid_ipv4(next_hop) or netaddr.valid_ipv6(next_hop)):
        raise l3_exroutes.ParamInvalid(param='nexthop', exroute=ex_route)
    return str(netaddr.IPAddress(next_hop))


def validate_ip_version(ip_version, destination_version, error_msg):
    """校验IP协议版本：4 或者 6

    :param ip_version: str or int,ip版本
    :param destination_version: int,destination的IP版本，值为4或6.
    :param error_msg: 错误信息
    :return: int
    """
    try:
        result = simple_deserializer(ip_version, deserializer=int)
    except TypeError as e:
        LOG.error("Can't convert %s to int.", ip_version)
        raise l3_exroutes.ParamInvalid(param='ip_version', exroute=error_msg)

    if result != destination_version:
        raise l3_exroutes.ParamInvalid(param='ip_version', exroute=error_msg)
    return result


def validate_source_ips(source_ips):
    """校验source_ips

    :param source_ips: str or list or none,ip列表
    :return: list
    """
    try:
        result = simple_deserializer(source_ips, default_value=[])
    except TypeError as e:
        LOG.error("Can't convert %s to list:%s", source_ips, traceback.format_exc())
        raise InvalidParam(name='source_ips', value=source_ips, error_trace=e)

    if result and not isinstance(result, list):
        raise InvalidParam(name='source_ips', value=source_ips)

    for elem in result:
        if not (netaddr.valid_ipv4(elem) or netaddr.valid_ipv6(elem)):
            raise InvalidParam(name='source_ips', value=source_ips)
    return result


# bfd 名称正则表达式
PATTERN_BFD_NAME = re.compile(r"^[0-9a-zA-Z\-_./+<>,~!@#$%^*();:'|\[\]{}]{1,64}$")


class RouterExRoute(model_base.BASEV2):
    """RouterExRoute"""
    __tablename__ = 'huawei_ac_exroute'
    router_id = sa.Column(sa.String(36), primary_key=True, nullable=False)
    destination = sa.Column(sa.String(64), primary_key=True, nullable=False)
    nexthop = sa.Column(sa.String(64), primary_key=True, nullable=False)
    ip_version = sa.Column(sa.Integer, primary_key=True, default=4, autoincrement=False)
    priority = sa.Column(sa.Integer, primary_key=True, nullable=True, autoincrement=False)
    type = sa.Column(sa.Integer, primary_key=True, nullable=False, autoincrement=False)
    ports = sa.Column(sa.PickleType(protocol=2), nullable=True)
    extra_opt = sa.Column(sa.PickleType(protocol=2), nullable=True)
    peer6_fw_enable = sa.Column(sa.Boolean, nullable=True)
    source_ips = sa.Column(sa.PickleType(protocol=2), nullable=True)
    bfd = sa.Column(sa.Boolean, default=True, nullable=False)
    bfd_name = sa.Column(sa.String(64), nullable=True)
    preference = sa.Column(sa.Integer, nullable=True)
    interface_name = sa.Column(sa.String(255), nullable=True)

    def show(self):
        """模型转成前端要求返回格式

        :return: dict
        """
        result = {'destination': self.destination, 'nexthop': self.nexthop, 'type': self.type,
                  'ip_version': self.ip_version, 'bfd': self.bfd, 'bfd_name': self.bfd_name,
                  'preference': self.preference, 'interface_name': self.interface_name}
        if self.source_ips:
            result['source_ips'] = self.source_ips
        return result

    def to_dict(self):
        ext_opt = simple_deserializer(self.extra_opt)
        return {'destination': self.destination, 'nexthop': self.nexthop, 'type': self.type,
                'ip_version': self.ip_version, 'bfd': self.bfd, 'bfd_name': self.bfd_name,
                'preference': self.preference, 'interface_name': self.interface_name, 'extra_opt': ext_opt,
                'source_ips': self.source_ips, 'ports': simple_deserializer(self.ports),
                'router_id': self.router_id, 'priority': self.priority, 'peer6_fw_enable': self.peer6_fw_enable}

    @classmethod
    def from_dict(cls, router_id, data):
        """将字典转换成模型

        :param router_id: str,uuid格式，所属路由的ID
        :param data: dict,扩展路由字典格式
        :return: RouterExRoute
        >>> import re
        >>> re.match(PATTERN_BFD_NAME,"09azAZ_.-/+<>,~!@#$%^*();:'|[]{}") is None
        False
        >>> re.match(PATTERN_BFD_NAME, '"') is None
        True
        >>> re.match(PATTERN_BFD_NAME, '') is None
        True
        >>> re.match(PATTERN_BFD_NAME, 'a'*65) is None
        True
        >>> re.match(PATTERN_BFD_NAME, '\\') is None
        True
        """
        if not isinstance(data, dict):
            return None
        destination = validate_ex_route_destination(data)
        ip_version = data.get('ip_version')
        destination_version = netaddr.IPNetwork(destination).version
        if ip_version is None:
            ip_version = destination_version
        else:
            ip_version = validate_ip_version(ip_version, destination_version, data)
        bfd_name = data.get('bfd_name')
        if bfd_name and not re.match(PATTERN_BFD_NAME, bfd_name):
            raise InvalidParam(name='bfd_name', value=bfd_name)
        result = RouterExRoute(router_id=router_id,
                               destination=destination,
                               nexthop=validate_ex_route_nexthop(data),
                               ip_version=ip_version,
                               priority=Exroutes_db_mixin._validate_priority(data),
                               type=Exroutes_db_mixin._validate_type(data),
                               ports=data.get('ports', []),
                               extra_opt=simple_deserializer(Exroutes_db_mixin._validate_extra_opt(data)),
                               peer6_fw_enable=Exroutes_db_mixin._validate_peer6_fw_enable(data),
                               source_ips=validate_source_ips(data.get('source_ips')),
                               bfd=bool_deserializer(data.get('bfd', True), name='bfd'),
                               bfd_name=bfd_name,
                               preference=simple_deserializer(data.get('preference'), deserializer=int),
                               interface_name=data.get('interface_name'))

        return result


class Exroutes_db_mixin(l3_exroutes.RouterExroutesBase):
    """Exroutes_db_mixin"""

    @classmethod
    def _validate_ip_version(cls, exroute):
        if not exroute.get('ip_version'):
            exroute['ip_version'] = netaddr.IPNetwork(
                exroute['destination']).version

        if not isinstance(exroute['ip_version'], (str, int)):
            raise l3_exroutes.ParamInvalid(param='ip_version', exroute=exroute)

        if int(exroute['ip_version']) not in [4, 6]:
            raise l3_exroutes.ParamInvalid(param='ip_version', exroute=exroute)
        return int(exroute['ip_version'])

    @classmethod
    def _validate_priority(cls, exroute):
        if not exroute.get('priority'):
            return None

        if not isinstance(exroute['priority'], int):
            raise l3_exroutes.ParamInvalid(param='priority', exroute=exroute)

        if exroute['priority'] not in range(256):
            raise l3_exroutes.ParamOutOfRange(
                param='priority', exroute=exroute, range='[0, 255]')

        return exroute['priority']

    @classmethod
    def _validate_type(cls, exroute):
        if not exroute.get('type'):
            raise l3_exroutes.ParamNotSpecified(param='type', exroute=exroute)

        if not isinstance(exroute['type'], int):
            raise l3_exroutes.ParamInvalid(param='type', exroute=exroute)

        if exroute['type'] not in [400, 401, 402]:
            raise l3_exroutes.ParamOutOfRange(
                param='type', exroute=exroute, range='400, 401, 402')

        return exroute['type']

    @classmethod
    def _validate_ports(cls, exroute):
        if exroute['type'] == 401 and not exroute.get('ports'):
            raise l3_exroutes.ParamNotSpecified(param='ports', exroute=exroute)

        if not exroute.get('ports'):
            return None

        if not isinstance(exroute['ports'], list):
            raise l3_exroutes.ParamInvalid(param='ports', exroute=exroute)

        for port in exroute['ports']:
            if not port:
                raise l3_exroutes.ParamInvalid(param='ports', exroute=exroute)

        return exroute['ports']

    @classmethod
    def _validate_extra_opt(cls, exroute):
        if not exroute.get('extra_opt'):
            return None

        if exroute['extra_opt'].get('remote_ip'):
            if not (netaddr.valid_ipv4(exroute['extra_opt']['remote_ip']) or
                    netaddr.valid_ipv6(exroute['extra_opt']['remote_ip'])):
                raise l3_exroutes.ParamInvalid(
                    param='remote_ip', exroute=exroute)

        if exroute['extra_opt'].get('health_detect'):
            if exroute['extra_opt']['health_detect'] not in ['true', 'false']:
                raise l3_exroutes.ParamInvalid(
                    param='health_detect', exroute=exroute)

        return exroute['extra_opt']

    @classmethod
    def _validate_peer6_fw_enable(cls, exroute):
        if 'peer6_fw_enable' not in exroute:
            exroute['peer6_fw_enable'] = False

        if not isinstance(exroute['peer6_fw_enable'], bool):
            raise l3_exroutes.ParamInvalid(
                param='peer6_fw_enable', exroute=exroute)

        return exroute['peer6_fw_enable']

    def add_exroutes(self, context, router_id, exroutes_info):
        """add_exroutes"""
        with context.session.begin(subtransactions=True):
            exroutes = exroutes_info.get('exroutes')
            for exroute in exroutes:
                router_routes = RouterExRoute(
                    router_id=router_id,
                    destination=validate_ex_route_destination(exroute),
                    nexthop=validate_ex_route_nexthop(exroute),
                    ip_version=self._validate_ip_version(exroute),
                    priority=self._validate_priority(exroute),
                    type=self._validate_type(exroute),
                    ports=self._validate_ports(exroute),
                    extra_opt=self._validate_extra_opt(exroute),
                    peer6_fw_enable=self._validate_peer6_fw_enable(exroute),
                )
                context.session.add(router_routes)
        return self._make_exroutes_dict(router_id, exroutes)

    def remove_exroutes(self, context, router_id, exroutes_info):
        """remove_exroutes"""
        all_deleted = []
        with context.session.begin(subtransactions=True):
            exroutes = exroutes_info.get('exroutes')
            for exroute in exroutes:
                validate_ex_route_destination(exroute)
                validate_ex_route_nexthop(exroute)
                self._validate_ip_version(exroute)
                self._validate_priority(exroute)
                self._validate_type(exroute)
                self._validate_ports(exroute)
                self._validate_extra_opt(exroute)

                deleted = self._delete_exroute_by_filter(
                    context, router_id, exroute)
                all_deleted.extend(deleted.get('exroutes', []))
        return self._make_exroutes_dict(router_id, all_deleted)

    def _delete_exroute_by_filter(self, context, router_id, exroute):
        query = context.session.query(RouterExRoute).filter(
            RouterExRoute.router_id == router_id)
        if 'destination' in exroute.keys():
            query = query.filter(
                RouterExRoute.destination == exroute.get('destination'))
        if 'nexthop' in exroute.keys():
            query = query.filter(
                RouterExRoute.nexthop == exroute.get('nexthop'))
        if 'ip_version' in exroute.keys():
            query = query.filter(
                RouterExRoute.ip_version == exroute.get('ip_version'))
        if 'priority' in exroute.keys():
            query = query.filter(
                RouterExRoute.priority == exroute.get('priority'))
        if 'type' in exroute.keys():
            query = query.filter(RouterExRoute.type == exroute.get('type'))
        if 'peer6_fw_enable' in exroute.keys():
            query = query.filter(
                RouterExRoute.peer6_fw_enable == exroute.get('peer6_fw_enable'))
        deleted = self._make_exroutes_dict(router_id, query.all())
        query.delete()
        return deleted

    def list_router_exroutes(self, context, router_id, type_value=None,
                             nexthop_value=None):
        """list_router_exroutes"""
        query = context.session.query(RouterExRoute).filter(
            RouterExRoute.router_id == router_id)
        if type_value is not None:
            query = query.filter(RouterExRoute.type == type_value)
        if nexthop_value is not None:
            query = query.filter(RouterExRoute.nexthop == nexthop_value)
        return self._make_exroutes_dict(router_id, query.all())

    def _make_exroutes_dict(self, router_id, exroutes_info):
        exroutes = []
        for exroute in exroutes_info:
            exroute_dict = self.get_exroute_dict(exroute)
            exroutes.append(exroute_dict)
        return {'id': router_id, 'exroutes': exroutes}

    @classmethod
    def get_exroute_dict(cls, exroute):
        """get_exroute_dict"""
        extra_opt = exroute.get('extra_opt') or {}
        if not isinstance(extra_opt, dict):
            extra_opt = jsonutils.loads(extra_opt)
        exroute_dict = {
            'destination': exroute.get('destination'),
            'nexthop': exroute.get('nexthop'),
            'ip_version': exroute.get('ip_version', 4),
            'type': exroute.get('type'),
            'extra_opt': extra_opt
        }
        if exroute.get('description'):
            exroute_dict['description'] = exroute['description']
        if exroute.get('priority'):
            exroute_dict['priority'] = exroute['priority']
        if exroute.get('ports'):
            exroute_dict['ports'] = exroute['ports']
        if 'peer6_fw_enable' in exroute:
            exroute_dict['peer6_fw_enable'] = exroute['peer6_fw_enable']
        return exroute_dict
