#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""exroute mec db"""

import netaddr
from neutron.db import models_v2
from neutron.db.l3_db import L3_NAT_dbonly_mixin
from oslo_serialization import jsonutils

from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log as logging
from networking_huawei.drivers.ac.db.exroute.exroute import RouterExRoute
from networking_huawei.drivers.ac.db.exroute.exroute import validate_ex_route_destination, \
    validate_ex_route_nexthop
from networking_huawei.drivers.ac.db.exroute.exroute_custom_db import warp_ex_route_valid_exception
from networking_huawei.drivers.ac.extensions.exroute import l3_exroutes

try:
    from neutron.common.exceptions import BadRequest
except ImportError:
    from neutron_lib.exceptions import BadRequest

LOG = logging.getLogger(__name__)


def query_router_mec_exroute(context, exroute, router_id):
    """query router exroute"""
    result = context.session.query(RouterExRoute).filter(RouterExRoute.router_id == router_id)
    if 'ip_version' in exroute.keys():
        result = result.filter(RouterExRoute.ip_version == exroute.get('ip_version'))
    if 'destination' in exroute.keys():
        result = result.filter(RouterExRoute.destination == exroute.get('destination'))
    if 'nexthop' in exroute.keys():
        result = result.filter(RouterExRoute.nexthop == exroute.get('nexthop'))
    return result


class ExroutesDbMixinMec(ncu.base_db.CommonDbMixin, L3_NAT_dbonly_mixin):
    """Exroutes db mixin custom"""

    @classmethod
    def _validate_ip_version_custom(cls, exroute):
        """validate ip version custom"""
        if not exroute.get('ip_version'):
            emsg = l3_exroutes.ParamNotSpecified(param='ip_version',
                                                 exroute=exroute)
            raise BadRequest(resource='ip_version', msg=emsg)

        if not isinstance(exroute['ip_version'], (str, int)):
            emsg = l3_exroutes.ParamInvalid(param='ip_version', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)

        if int(exroute['ip_version']) not in [4, 6]:
            emsg = l3_exroutes.ParamInvalid(param='ip_version', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)

        return int(exroute['ip_version'])

    @classmethod
    def _validate_type_custom(cls, exroute):
        """validate type custom"""
        if not exroute.get('type'):
            emsg = l3_exroutes.ParamNotSpecified(param='type', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)

        if not isinstance(exroute['type'], int):
            emsg = l3_exroutes.ParamInvalid(param='type', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)

        if exroute['type'] not in [401, 402]:
            emsg = l3_exroutes.ParamOutOfRange(
                param='type', exroute=exroute, range='401, 402')
            raise BadRequest(resource='exroutes', msg=emsg)

        return exroute['type']

    def _validate_ports_custom(self, context, exroute, router_id):
        """validate ports custom"""
        if exroute['type'] == 402:
            return []
        ports = []
        router = self._get_router(context, router_id)
        device_owner = self._get_device_owner(context, router)
        if ncu.get_ops_version() in {ac_cnst.OPS_Q, ac_cnst.OPS_R,
                                     ac_cnst.FSP_6_5, ac_cnst.FSP_21_0}:
            router_attached_ports = router.attached_ports
        else:
            router_attached_ports = router.attached_ports.all()
        for port_r in (rp.port for rp in router_attached_ports):
            if port_r['device_owner'] != device_owner:
                continue
            for ip_addr in port_r['fixed_ips']:
                subnet_id = ip_addr['subnet_id']
                m2_ip_allocation = models_v2.IPAllocation
                ip_allocation = context.session.query(m2_ip_allocation).filter(
                    m2_ip_allocation.subnet_id == subnet_id,
                    m2_ip_allocation.ip_address == exroute['nexthop']).first()
                if ip_allocation and ip_allocation.port_id:
                    ports.append(ip_allocation.port_id)
                    return ports
        emsg = l3_exroutes.ParamNotSpecified(param='port', exroute=exroute)
        raise BadRequest(resource='exroutes', msg=emsg)

    @classmethod
    def _validate_bfd(cls, exroute):
        """validate bfd"""
        if 'bfd' not in exroute:
            return True
        else:
            if not isinstance(exroute['bfd'], bool):
                emsg = l3_exroutes.ParamInvalid(
                    param='bfd', exroute=exroute)
                raise BadRequest(resource='exroutes', msg=emsg)
            return exroute['bfd']

    @classmethod
    def _validate_source_ips(cls, exroute, health_detect):
        """validate source ips"""
        # 取消401场景必须有 source_ips 的校验
        source_ips = exroute.get('source_ips')
        if health_detect and not source_ips and exroute.get('type') != 401:
            emsg = l3_exroutes.ParamNotSpecified(param='source_ips', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)

        result = {"health_detect": health_detect}
        if not source_ips:
            return result

        if not health_detect:
            emsg = l3_exroutes.ParamInvalid(param='source_ips', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)
        if not isinstance(source_ips, list):
            emsg = l3_exroutes.ParamInvalid(param='source_ips', exroute=exroute)
            raise BadRequest(resource='exroutes', msg=emsg)
        source_ip_list = []
        for source_ip in source_ips:
            if not (netaddr.valid_ipv4(source_ip) or netaddr.valid_ipv6(source_ip)):
                emsg = l3_exroutes.ParamInvalid(param='source_ips', exroute=exroute)
                raise BadRequest(resource='exroutes', msg=emsg)
            source_ip_list.append(str(netaddr.IPAddress(source_ip)))
        result["source_ips"] = source_ip_list

        return result

    def _validate_extra_opt_custom(self, exroute):
        """validate extra opt custom"""
        if exroute['type'] == 402:
            if 'bfd' in exroute:
                emsg = l3_exroutes.ParamInvalid(param='bfd', exroute=exroute)
                raise BadRequest(resource='exroutes', msg=emsg)
            if 'source_ips' in exroute:
                emsg = l3_exroutes.ParamInvalid(param='source_ips',
                                                exroute=exroute)
                raise BadRequest(resource='exroutes', msg=emsg)
            return None

        health_detect = self._validate_bfd(exroute)
        extra_opt = self._validate_source_ips(exroute, health_detect)
        return jsonutils.dumps(extra_opt, {})

    @classmethod
    def get_exroute_with_type(cls, exroute, query):
        """get exroute with type"""
        index_401 = 0
        index_402 = 0
        for exroute_d in query.all():
            if exroute_d['type'] == 400:
                emsg = "Can not support type:400"
                raise BadRequest(resource='exroutes', msg=emsg)
            if exroute_d['type'] == 401:
                index_401 = index_401 + 1
            if exroute_d['type'] == 402:
                index_402 = index_402 + 1
        if index_401 >= 1 and index_402 >= 1:
            emsg = "Can not delete type:401 and type: " \
                   "402 at the same request"
            raise BadRequest(resource='exroutes', msg=emsg)
        if index_401 >= 1:
            exroute['type'] = 401
        if index_402 >= 1:
            exroute['type'] = 402
        return exroute

    def pre_delete_exroutes(self, context, router_id, exroutes_info):
        """pre delete exroutes"""
        exroutes = exroutes_info.get('exroutes')
        exroutes_del = []
        for exroute in exroutes:
            destination = warp_ex_route_valid_exception(validate_ex_route_destination, exroute)
            exroute['destination'] = destination
            if exroute.get('type'):
                self._validate_type_custom(exroute)

            if exroute.get('ip_version'):
                self._validate_ip_version_custom(exroute)

            if exroute.get('nexthop'):
                exroute['nexthop'] = warp_ex_route_valid_exception(validate_ex_route_nexthop, exroute)

            if not exroute.get('type'):
                query = query_router_mec_exroute(context, exroute, router_id)
                if query.first():
                    self.get_exroute_with_type(exroute, query)
                    exroutes_del.append(exroute)
            else:
                exroutes_del.append(exroute)

        return {'id': router_id, 'exroutes_del': exroutes_del}

    def delete_mec_exroutes(self, context, router_id, exroutes_info):
        """delete exroutes"""
        all_deleted = []
        with context.session.begin(subtransactions=True):
            exroutes = exroutes_info.get('exroutes')
            for exroute in exroutes:
                destination = warp_ex_route_valid_exception(validate_ex_route_destination, exroute)
                exroute['destination'] = destination
                if exroute.get('nexthop'):
                    exroute['nexthop'] = warp_ex_route_valid_exception(validate_ex_route_nexthop, exroute)

                deleted = self._delete_exroute_by_filter_custom(
                    context, router_id, exroute)
                all_deleted.extend(deleted.get('exroutes', []))
        return self._make_delete_exroutes_dict_custom(router_id, all_deleted)

    def _delete_exroute_by_filter_custom(self, context, router_id, exroute):
        """delete exroute by filter custom"""
        query = query_router_mec_exroute(context, exroute, router_id)
        if 'type' in exroute.keys():
            query = query.filter(
                RouterExRoute.type == exroute.get('type')
            )
        deleted = self._make_delete_exroutes_dict_custom(router_id, query.all())
        query.delete()
        return deleted

    def list_mec_exroutes(self, context, router_id, type_value=None,
                          nexthop_value=None):
        """list exroutes"""
        query = context.session.query(RouterExRoute).filter(
            RouterExRoute.router_id == router_id)
        if type_value is not None:
            query = query.filter(RouterExRoute.type == type_value)
        if nexthop_value is not None:
            query = query.filter(RouterExRoute.nexthop == nexthop_value)
        return self._make_exroutes_dict_custom(context, router_id, query.all())

    def _make_exroutes_dict_custom(self, context, router_id, exroutes_info):
        """make exroutes dict custom"""
        exroutes = []
        exroutes_list = []
        for exroute in exroutes_info:
            exroutes_list.append(self.get_exroute_dict_custom(
                exroute, context, router_id, False))
            exroutes.append(self.get_exroute_dict_custom(
                exroute, None, None, True))
        return {'id': router_id, 'exroutes': exroutes,
                'exroutes_list': exroutes_list}

    def _make_delete_exroutes_dict_custom(self, router_id, exroutes_info):
        """make delete exroutes dict custom"""
        exroutes = []
        for exroute in exroutes_info:
            exroutes.append(self.get_exroute_dict_custom(
                exroute, None, None, True))
        return {'id': router_id, 'exroutes': exroutes}

    def get_exroute_dict_custom(self, exroute, context, router_id, show_flg):
        """get exroute dict custom common"""
        result = {
            'destination': exroute.get('destination'),
            'description': exroute.get('description'),
            'nexthop': exroute.get('nexthop'),
            'type': exroute.get('type'),
            'ip_version': exroute.get('ip_version', 4),
            'preference': exroute.get('preference', ''),
            'bfd_name': exroute.get('bfd_name', ''),
            'interface_name': exroute.get('interface_name', ''),
        }
        if exroute.get('type') == 401:
            if exroute.get('extra_opt'):
                extra_opt = exroute.get('extra_opt')
            else:
                extra_opt = self._validate_extra_opt_custom(exroute)
            if not isinstance(extra_opt, dict):
                extra_opt = jsonutils.loads(extra_opt)
            if show_flg:
                if 'health_detect' in extra_opt:
                    result['bfd'] = extra_opt['health_detect']
                if extra_opt.get('source_ips'):
                    result['source_ips'] = extra_opt['source_ips']
            else:
                if exroute.get('ports'):
                    ports = exroute.get('ports')
                else:
                    ports = self._validate_ports_custom(context, exroute, router_id)
                result['extra_opt'] = extra_opt
                result['ports'] = ports
        return result
