#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2020 Huawei Technologies Co. Ltd. All rights reserved.
"""flow mirror db"""

import copy

from oslo_log import log as logging
from oslo_utils import uuidutils
from sqlalchemy.orm import exc

try:
    from neutron_lib import exceptions as nexception
except ImportError:
    from neutron.common import exceptions as nexception

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.decrypt_factor.decrypt_factor_db import DecryptFactorDbMixin
from networking_huawei.drivers.ac.db.flow_mirror import schema
from networking_huawei.drivers.ac.extensions.flowmirror import flow_mirror as flowmirror

LOG = logging.getLogger(__name__)


class ExistSameNameTapFlowException(nexception.InvalidInput):
    """ same name tap flow has already existed """
    message = "tap flow with name %(tap_flow_name)s has already existed"


class FlowMirrorDbMixin(ncu.base_db.CommonDbMixin):
    """flow mirror DB
    """

    def __init__(self):
        self.dfactor_db = DecryptFactorDbMixin()
        self.ac_db_interface = ACdbInterface()

    def _make_tap_service_dict(self, tap_servcie_db, fields=None):
        """make tap service dict"""
        res = {'id': tap_servcie_db['id'],
               'tenant_id': tap_servcie_db['tenant_id'],
               'name': tap_servcie_db['name'],
               'description': tap_servcie_db['description']}
        if 'remote_v4ip' in tap_servcie_db:
            res['remote_v4ip'] = tap_servcie_db['remote_v4ip']
        if 'remote_v6ip' in tap_servcie_db:
            res['remote_v6ip'] = tap_servcie_db['remote_v6ip']
        if 'port_id' in tap_servcie_db:
            res['port_id'] = tap_servcie_db['port_id']
        tap_service_dict = self._fields(res, fields)
        return tap_service_dict

    def _make_tap_flow_dict(self, tap_flow_db, fields=None):
        """make tap flow dict"""
        res = {'id': tap_flow_db['id'],
               'tenant_id': tap_flow_db['tenant_id'],
               'name': tap_flow_db['name'],
               'direction': tap_flow_db['direction'],
               'source_port': tap_flow_db['source_port'],
               'description': tap_flow_db['description'],
               'tap_service_id': tap_flow_db['tap_service_id'],
               'sdn_tapservice_name': tap_flow_db['sdn_tapservice_name']}
        if 'destination_v6ip_prefix' in tap_flow_db:
            res['destination_v6ip_prefix'] = \
                tap_flow_db['destination_v6ip_prefix']
        if 'destination_v4ip_prefix' in tap_flow_db:
            res['destination_v4ip_prefix'] = \
                tap_flow_db['destination_v4ip_prefix']
        tap_flow_dict = self._fields(res, fields)
        return tap_flow_dict

    def create_db_tap_service(self, context, tapservice, fields=None):
        """the function to create tap service"""
        tap_service_log = copy.deepcopy(tapservice)['tap_service']
        tap_service_log = validate.validate_log_record(
            tap_service_log, ac_constants.NW_HW_FLOW_MIRROR)
        LOG.info(_LI('[AC] Create tap service in Neutron '
                     'DB for %s.'), tap_service_log)
        tap_service_info = tapservice['tap_service']
        tap_service_name = tap_service_info.get('name')
        tap_service_description = tap_service_info.get('description')
        tap_service_tenant_id = tap_service_info.get('tenant_id')
        tap_service_port_id = tap_service_info.get('port_id')

        flowmirror.validate_tap_service_info(tap_service_info)

        with context.session.begin(subtransactions=True):
            tap_service_db = schema.ACTapServiceSchema(
                id=uuidutils.generate_uuid(),
                tenant_id=tap_service_tenant_id,
                name=tap_service_name,
                description=tap_service_description,
                port_id=tap_service_port_id,
                remote_v4ip=tap_service_info['remote_v4ip'],
                remote_v6ip=tap_service_info['remote_v6ip'])
            context.session.add(tap_service_db)
        return self._make_tap_service_dict(tap_service_db, fields)

    def create_db_tap_flow(self, context, tapflow, fields=None):
        """the function to create tap flow"""
        tap_flow_log = copy.deepcopy(tapflow)['tap_flow']
        tap_flow_log = validate.validate_log_record(
            tap_flow_log, ac_constants.NW_HW_FLOW_MIRROR)
        LOG.info(_LI('[AC] Create tap flow in Neutron '
                     'DB for %s.'), tap_flow_log)
        tap_flow_info = tapflow['tap_flow']
        tap_flow_name = tap_flow_info.get('name')
        tap_flow_description = tap_flow_info.get('description')
        tap_flow_tenant_id = tap_flow_info.get('tenant_id')
        tap_flow_tap_service_id = tap_flow_info.get('tap_service_id')
        tap_flow_direction = tap_flow_info.get('direction')
        tap_flow_source_port = tap_flow_info.get('source_port')
        tap_flow_sdn_tapservice_name = tap_flow_info.get('sdn_tapservice_name')

        tap_flow_destination_v4ip_prefix, tap_flow_destination_v6ip_prefix = \
            flowmirror.validate_tap_flow_info(tap_flow_source_port,
                                              tap_flow_tap_service_id,
                                              tap_flow_sdn_tapservice_name,
                                              tap_flow_info)

        with context.session.begin(subtransactions=True):
            if context.session.query(schema.ACTapFlowSchema).filter(schema.ACTapFlowSchema.name == tap_flow_name).all():
                LOG.error(_LE('[AC] Same name tap flow already exists'))
                raise ExistSameNameTapFlowException(tap_flow_name=tap_flow_name)
            tap_flow_db = schema.ACTapFlowSchema(
                id=uuidutils.generate_uuid(),
                tenant_id=tap_flow_tenant_id,
                name=tap_flow_name,
                description=tap_flow_description,
                direction=tap_flow_direction,
                tap_service_id=tap_flow_tap_service_id,
                destination_v4ip_prefix=tap_flow_destination_v4ip_prefix,
                source_port=tap_flow_source_port,
                destination_v6ip_prefix=tap_flow_destination_v6ip_prefix,
                sdn_tapservice_name=tap_flow_sdn_tapservice_name)
            context.session.add(tap_flow_db)

        return self._make_tap_flow_dict(tap_flow_db, fields)

    def get_tap_service_by_id(self, context, tap_service_id):
        """get tap service resource"""
        LOG.info(_LI('[TAP]Query tap-service by id:%s.'), tap_service_id)
        try:
            # 性能稍差:result = self._get_by_id(context, schema.ACTapServiceSchema, tap_service_id)
            result = context.session.query(schema.ACTapServiceSchema).filter_by(id=tap_service_id).one()
        except exc.NoResultFound:
            raise flowmirror.TapServiceNotFound(tap_service_id=tap_service_id)
        LOG.info(_LI('[TAP]Query tap-service success:id=%s,result%s.'), tap_service_id, result)
        return result

    def get_tap_flow_by_id(self, context, tap_flow_id):
        """get tap flow resource"""
        LOG.info(_LI('[TAP]Query tap-flow by id:%s.'), tap_flow_id)
        try:
            result = context.session.query(schema.ACTapFlowSchema).filter_by(id=tap_flow_id).one()
        except exc.NoResultFound:
            raise flowmirror.TapFlowNotFound(tap_flow_id=tap_flow_id)
        LOG.info(_LI('[TAP]Query tap-flow success:id=%s,result%s.'), tap_flow_id, result)
        return result

    def delete_db_tap_service(self, context, tap_service_id):
        """the function to delete tap service"""
        LOG.info(_LI('[AC] Delete tapservice in Neutron DB by tap service id: %s.'), tap_service_id)
        tap_service_db = self.get_tap_service_by_id(context, tap_service_id)
        with context.session.begin(subtransactions=True):
            context.session.delete(tap_service_db)

    def delete_db_tap_flow(self, context, tap_flow_id):
        """the function to delete tap flow"""
        LOG.info(_LI('[AC] Delete tapflow in Neutron DB by tap flow id: %s.'), tap_flow_id)
        tap_flow_db = self.get_tap_flow_by_id(context, tap_flow_id)
        with context.session.begin(subtransactions=True):
            context.session.delete(tap_flow_db)

    def get_tap_flows(self, context, filters=None, fields=None):
        """the function to get tap flows"""
        LOG.info(_LI("[AC] Get tap flowws from Neutron DB."))
        tap_flows = self._get_collection(context, schema.ACTapFlowSchema,
                                         self._make_tap_flow_dict,
                                         filters=filters, fields=fields)
        return tap_flows

    def get_tap_services(self, context, filters=None, fields=None):
        """the function to get tap services"""
        LOG.info(_LI("[AC] Get tap services from Neutron DB."))
        tap_services = self._get_collection(context, schema.ACTapServiceSchema,
                                            self._make_tap_service_dict,
                                            filters=filters, fields=fields)
        return tap_services

    def get_tap_flow(self, context, tap_flow_id, fields=None):
        """the function to get tap flow"""
        LOG.info(_LI('[AC] Get tap flow from Neutron '
                     'DB by id: %s.'), tap_flow_id)
        tap_flow_db = context.session. \
            query(schema.ACTapFlowSchema). \
            filter_by(id=tap_flow_id).first()
        if tap_flow_db:
            return self._make_tap_flow_dict(tap_flow_db, fields)
        else:
            raise flowmirror.TapFlowNotFound(tap_flow_id=tap_flow_id)

    def get_tap_service(self, context, tap_service_id, fields=None):
        """the function to get tap service"""
        LOG.info(_LI('[AC] Get tap service from Neutron '
                     'DB by id: %s.'), tap_service_id)
        tap_service_db = context.session. \
            query(schema.ACTapServiceSchema). \
            filter_by(id=tap_service_id).first()
        if tap_service_db:
            return self._make_tap_service_dict(tap_service_db, fields)
        else:
            raise flowmirror.TapServiceNotFound(tap_service_id=tap_service_id)

    def update_db_tap_service(self, context, tap_service_id, tap_service):
        """the function to update tap service"""
        LOG.info(_LI('[dryRun]update tap service in Neutron DB by tap service id: %s.'), tap_service_id)
        tap_service_db = self.get_tap_service_by_id(context, tap_service_id)
        with context.session.begin(subtransactions=True):
            tap_service_db.update(tap_service)

    def update_db_tap_flow(self, context, tap_flow_id, tap_flow):
        """the function to update tap service"""
        LOG.info(_LI('[dryRun]update tap flow in Neutron DB by tap flow id: %s.'), tap_flow_id)
        tap_flow_db = self.get_tap_flow_by_id(context, tap_flow_id)
        with context.session.begin(subtransactions=True):
            tap_flow_db.update(tap_flow)
