#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Sync result"""

try:
    from neutron.db import common_db_mixin
except ImportError:
    from networking_huawei.drivers.ac.common import common_db_mixin
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.db.sync_result import schema

LOG = logging.getLogger(__name__)


class SyncDbMixin(common_db_mixin.CommonDbMixin):
    """Sync result Db Mixin"""

    def _make_record_db_dict(self, sync_record_db, fields=None):
        """Make record db dict"""
        if sync_record_db:
            res = {'id': sync_record_db['id'],
                   'tenant_name': sync_record_db['tenant_name'],
                   'res_name': sync_record_db['res_name'],
                   'res_type': sync_record_db['res_type'],
                   'create_time': sync_record_db['create_time'],
                   'update_time': sync_record_db['update_time'],
                   'status': sync_record_db['status'],
                   'oper': sync_record_db['oper'],
                   'sync_type': sync_record_db['sync_type'],
                   'exec_result': sync_record_db['exec_result'],
                   'error_message': sync_record_db['error_message']}
            return self._fields(res, fields)
        return None

    def _make_sync_summary_dict(self, sync_summary, fields=None):
        """Make sync summary dict"""
        if sync_summary:
            res = {'sync_id': sync_summary['sync_id'],
                   'sync_start_time': sync_summary['sync_start_time'],
                   'sync_end_time': sync_summary['sync_end_time']}
            return self._fields(res, fields)
        return None

    def create_db_record_list(self, context, record_dict):
        """create db record list"""
        LOG.info(_LI('[AC] Begin to Create record in Neutron '
                     'DB for %s.'), record_dict)
        with context.session.begin(subtransactions=True):
            record_db = schema.ACSyncResultSchema(
                id=record_dict['Id'],
                tenant_name=record_dict.get('TenantName', ''),
                res_name=record_dict.get('Name', ''),
                res_type=record_dict.get('Type', ''),
                create_time=record_dict.get('CreateTime', ''),
                update_time=record_dict.get('UpdateTime', ''),
                status=record_dict.get('Result', ''),
                oper=record_dict.get('Operation', ''),
                sync_type=record_dict['sync_type'],
                exec_result=record_dict.get('exec_result', ''),
                error_message=record_dict.get('error_message', ''))
            context.session.add(record_db)
            context.session.flush()
        return self._make_record_db_dict(record_db)

    def get_sync_results_summary(self, context):
        """get sync results summary"""
        sync_date_all = self.get_db_sync_results_query(context)
        count = {
            'total_create_resources': 0, 'total_update_resources': 0,
            'total_delete_resources': 0, 'sync_success': 0,
            'sync_fail': 0, 'sync_time_out': 0, 'sync_skip_sync': 0
        }

        self._set_count(count, sync_date_all)

        total_sync_resources = \
            'total:{}, success:{}, fail:{}, time-out:{}, skip-sync:{}'.format(
                len(sync_date_all), count['sync_success'], count['sync_fail'],
                count['sync_time_out'], count['sync_skip_sync'],
            )

        ac_db_interface = ACdbInterface()
        sync_summary = {}
        try:
            sync_summary = self._make_sync_summary_dict(
                ac_db_interface.get_neutron_sync_record())
            if not sync_summary:
                return []
            total_time_used = \
                (sync_summary.get('sync_end_time') -
                 sync_summary.get('sync_start_time')).total_seconds()
        except Exception as ex:
            LOG.error("[AC]Make sync summary dict error :%s", ex)
            total_time_used = 0

        sync_summary_data = {
            "id": "Neutron Sync Results Summary",
            "sync_id": sync_summary.get('sync_id'),
            "sync_start_time": sync_summary.get('sync_start_time'),
            "sync_end_time": sync_summary.get('sync_end_time'),
            "total_time_used": total_time_used,
            "total_create_resources": count['total_create_resources'],
            "total_update_resources": count['total_update_resources'],
            "total_delete_resources": count['total_delete_resources'],
            "total_sync_resources": total_sync_resources
        }
        return [sync_summary_data]

    @staticmethod
    def _set_count(count, sync_date_all):
        for data in sync_date_all:
            if data['oper'] == 'POST':
                count['total_create_resources'] += 1
            if data['oper'] == 'PUT':
                count['total_update_resources'] += 1
            if data['oper'] == 'DELETE':
                count['total_delete_resources'] += 1
            if data['status'] == 'success':
                count['sync_success'] += 1
            if data['status'] == 'failure':
                count['sync_fail'] += 1
            if data['status'] == 'time-out':
                count['sync_time_out'] += 1
            if data['status'] == 'skip-sync':
                count['sync_skip_sync'] += 1

    @classmethod
    def delete_db_sync_results(cls, context, sync_type):
        """the function to delete sync results"""
        sync_results_query = context.session.query(schema.ACSyncResultSchema). \
            filter_by(sync_type=sync_type)
        LOG.info(_LI("[AC]Sync type is: %s, delete neutron sync repair results"
                     " : %s"), sync_type, sync_results_query.all())
        sync_results_query.delete()

    def _get_sync_db(self, context, model, res_id):
        """Get sync record by id from neutron db"""
        try:
            return self._get_by_id(context, model, res_id)
        except Exception as ex:
            LOG.error(_LE('[AC] Get AC process status resource failed: %s'), ex)

    def get_db_sync_results_query(self, context, **kwargs):
        """the function to get sync results"""
        filters = kwargs.get('filters')
        if filters and 'true' in str(filters.get("sync_summary", "")):
            LOG.info(_LI("[AC] Get sync results summary from Neutron DB."))
            sync_summary_data = self.get_sync_results_summary(context)
            return sync_summary_data

        LOG.info(_LI("[AC] Get sync results from Neutron DB. "
                     "filters :%s"), str(filters))
        sync_data_results = self._get_collection(
            context, schema.ACSyncResultSchema, self._make_record_db_dict,
            filters=filters, fields=kwargs.get('fields'),
            sorts=kwargs.get('sorts'), limit=kwargs.get('limit'),
            marker_obj=kwargs.get('marker_obj'),
            page_reverse=kwargs.get('page_reverse', False))
        LOG.info(_LI("Get sync results from Neutron DB :%s"), sync_data_results)
        return sync_data_results
