#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpc connecton cmcc db"""

from oslo_log import log as logging

try:
    from neutron.common.exceptions import InvalidInput
except ImportError:
    from neutron_lib.exceptions import InvalidInput
from sqlalchemy.orm import exc
from networking_huawei.drivers.ac.db.vpc_connection import schema
from networking_huawei.drivers.ac.extensions.vpc_connection \
    import vpc_connection
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LI

LOG = logging.getLogger(__name__)


class VpcConnectionCustomDbMixin(ncu.base_db.CommonDbMixin):
    """Vpc Connection DB
    """

    def _get_vpc_conn_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except exc.NoResultFound:
            raise vpc_connection.VpcConnNotFound(vpc_conn_id=v_id)

    @classmethod
    def make_vpc_conn_dict(cls, vpc_conn_db):
        """call by VpcConnectionCustomDbMixin and VpcConnectionDbMixin"""
        res = {'id': vpc_conn_db['id'],
               'tenant_id': vpc_conn_db['tenant_id'],
               'name': vpc_conn_db['name'],
               'description': vpc_conn_db['description'],
               'local_router': vpc_conn_db['local_router'],
               'local_subnets': vpc_conn_db['local_subnets'],
               'local_cidrs': vpc_conn_db['local_cidrs'],
               'local_firewall_enable': vpc_conn_db['local_firewall_enable'],
               'peer_router': vpc_conn_db['peer_router'],
               'peer_subnets': vpc_conn_db['peer_subnets'],
               'peer_cidrs': vpc_conn_db['peer_cidrs'],
               'peer_firewall_enable': vpc_conn_db['peer_firewall_enable'],
               'status': vpc_conn_db['status'],
               'mode': vpc_conn_db['mode'],
               'priority': vpc_conn_db['priority']}
        return res

    def _make_vpc_conn_custom_dict(self, vpc_conn_db, fields=None):
        """fix:重复代码"""
        result = self.make_vpc_conn_dict(vpc_conn_db)
        result['vpc_connection_id'] = result.pop('id', vpc_conn_db['id'])
        result['mode'] = str(result.get('mode', ''))
        result['priority'] = str(result.get('priority', ''))
        return self._fields(result, fields)

    def create_db_vpc_connection(self, context, vpc_conn, status=None):
        """the function to create vpc connection"""
        LOG.info(_LI('[AC] Create vpc connection in Neutron '
                     'DB for %s.'), vpc_conn)
        vpc_conn_info = vpc_conn['vpc_connection']
        filters = {
            'local_router': vpc_conn_info['local_router'],
            'peer_router': vpc_conn_info['peer_router']}
        query_vpc_conn = context.session.query(schema.ACVpcConnectionSchema). \
            filter_by(**filters).all()
        if query_vpc_conn:
            error_msg = "The same vpc connection has setted, " \
                        "local_router: %s, " \
                        "peer_router: %s," % \
                        (vpc_conn_info['local_router'],
                         vpc_conn_info['peer_router'])
            raise InvalidInput(error_message=error_msg)
        with context.session.begin(subtransactions=True):
            vpc_conn_db = schema.ACVpcConnectionSchema(
                tenant_id=vpc_conn_info['tenant_id'],
                name=vpc_conn_info['name'],
                description=vpc_conn_info['description'],
                local_router=vpc_conn_info['local_router'],
                local_subnets=[],
                local_cidrs=vpc_conn_info['local_cidrs'],
                local_firewall_enable=vpc_conn_info['local_firewall_enable'],
                peer_router=vpc_conn_info['peer_router'],
                peer_subnets=[],
                peer_cidrs=vpc_conn_info['peer_cidrs'],
                peer_firewall_enable=vpc_conn_info['peer_firewall_enable'],
                status=status,
                mode=vpc_conn_info['mode'],
                priority=vpc_conn_info['priority'])
            context.session.add(vpc_conn_db)
            context.session.flush()

        return self._fields(self.make_vpc_conn_dict(vpc_conn_db), None)

    def update_db_vpc_connection(self, context, vpc_conn_id, vpc_conn,
                                 original_vpc_conn=None):
        """the function to update vpc connection"""
        vpc_conn_info = self.deal_vpc_conn_before(original_vpc_conn, vpc_conn,
                                                  vpc_conn_id)
        with context.session.begin(subtransactions=True):
            vpc_conn_db = self._get_vpc_conn_resource(
                context, schema.ACVpcConnectionSchema, vpc_conn_id)
            if vpc_conn_info:
                vpc_conn_db.update(vpc_conn_info)
        return self._fields(self.make_vpc_conn_dict(vpc_conn_db), None)

    @classmethod
    def deal_vpc_conn_before(cls, original_vpc_conn, vpc_conn, vpc_conn_id):
        """fix:重复代码"""
        LOG.info(_LI('[AC] Update vpc connection in Neutron DB '
                     'by id: %s.'), vpc_conn_id)
        vpc_conn_info = vpc_conn['vpc_connection']
        if original_vpc_conn:
            if not vpc_conn_info.get('name') and \
                    not original_vpc_conn.get('name'):
                vpc_conn_info['name'] = ''
            if not vpc_conn_info.get('description') and \
                    not original_vpc_conn.get('description'):
                vpc_conn_info['description'] = ''
            if not vpc_conn_info.get('local_cidrs') and \
                    not original_vpc_conn.get('local_cidrs'):
                vpc_conn_info['local_cidrs'] = []
            if not vpc_conn_info.get('peer_cidrs') and \
                    not original_vpc_conn.get('peer_cidrs'):
                vpc_conn_info['peer_cidrs'] = []
            if not vpc_conn_info.get('mode') and \
                    not original_vpc_conn.get('mode'):
                vpc_conn_info['mode'] = 0
        return vpc_conn_info

    def delete_db_vpc_connection(self, context, vpc_conn_id):
        """the function to delete vpc connection"""
        LOG.info(_LI('[AC] Delete vpc connection in Neutron '
                     'DB by id: %s.'), vpc_conn_id)
        with context.session.begin(subtransactions=True):
            vpc_conn_db = self._get_vpc_conn_resource(
                context, schema.ACVpcConnectionSchema, vpc_conn_id)
            context.session.delete(vpc_conn_db)

    def get_db_vpc_connections(self, context, filters=None, fields=None,
                               sorts=None, limit=None, marker_obj=None,
                               page_reverse=False):
        """the function to get vpc connections"""
        LOG.info(_LI("[AC] Get vpc connections from Neutron DB."))
        return self._get_collection(context, schema.ACVpcConnectionSchema,
                                    self._make_vpc_conn_custom_dict,
                                    filters=filters, fields=fields,
                                    sorts=sorts, limit=limit,
                                    marker_obj=marker_obj,
                                    page_reverse=page_reverse)

    def get_db_vpc_connection(self, context, vpc_conn_id, fields=None):
        """the function to get vpc connection"""
        LOG.info(_LI('[AC] Get vpc connection from Neutron '
                     'DB by id: %s.'), vpc_conn_id)
        vpc_connection_db = self._get_vpc_conn_resource(
            context, schema.ACVpcConnectionSchema, vpc_conn_id)
        return self._make_vpc_conn_custom_dict(vpc_connection_db, fields=fields)
