#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpc-connection db"""

import six
import webob
from sqlalchemy.orm import exc
from oslo_log import log as logging
from networking_huawei.drivers.ac.db.vpc_connection import schema
from networking_huawei.drivers.ac.db.vpc_connection.vpc_connection_custom_db \
    import \
    VpcConnectionCustomDbMixin
from networking_huawei.drivers.ac.extensions.vpc_connection import \
    vpc_connection
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LI

LOG = logging.getLogger(__name__)


class VpcConnectionDbMixin(ncu.base_db.CommonDbMixin):
    """Vpc Connection DB
    """

    def _get_vpc_conn_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except exc.NoResultFound:
            raise vpc_connection.VpcConnNotFound(vpc_conn_id=v_id)

    def _make_vpc_conn_dict(self, vpc_conn_db, fields=None):
        """make_vpc_conn_dict"""
        result = VpcConnectionCustomDbMixin.make_vpc_conn_dict(vpc_conn_db)
        result['fw_enabled'] = vpc_conn_db['fw_enabled']
        return self._fields(result, fields)

    def create_db_vpc_connection(self, context, vpc_conn, status=None):
        """the function to create vpc connection"""
        LOG.info(_LI('[AC] Create vpc connection in Neutron '
                     'DB for %s.'), vpc_conn)
        vpc_conn_info = vpc_conn['vpc_connection']
        filters = {
            'local_router': vpc_conn_info['local_router'],
            'local_subnets': vpc_conn_info['local_subnets'],
            'peer_router': vpc_conn_info['peer_router'],
            'peer_subnets': vpc_conn_info['peer_subnets']}
        query_vpc_conn = context.session.query(schema.ACVpcConnectionSchema). \
            filter_by(**filters).all()
        if query_vpc_conn:
            error_msg = "The same vpc connection has setted, " \
                        "local_router: %s, local_subnets: %s, " \
                        "peer_router: %s, peer_subnets: %s" % \
                        (vpc_conn_info['local_router'],
                         vpc_conn_info['local_subnets'],
                         vpc_conn_info['peer_router'],
                         vpc_conn_info['peer_subnets'])
            explanation = six.text_type(error_msg)
            raise webob.exc.HTTPInternalServerError(explanation=explanation)
        if not vpc_conn_info.get('name'):
            vpc_conn_info['name'] = ''
        if not vpc_conn_info.get('description'):
            vpc_conn_info['description'] = ''
        with context.session.begin(subtransactions=True):
            vpc_conn_db = schema.ACVpcConnectionSchema(
                tenant_id=vpc_conn_info['tenant_id'],
                name=vpc_conn_info['name'],
                description=vpc_conn_info['description'],
                local_router=vpc_conn_info['local_router'],
                local_subnets=vpc_conn_info['local_subnets'],
                local_cidrs=vpc_conn_info['local_cidrs'],
                local_firewall_enable=vpc_conn_info['local_firewall_enable'],
                peer_router=vpc_conn_info['peer_router'],
                peer_subnets=vpc_conn_info['peer_subnets'],
                peer_cidrs=vpc_conn_info['peer_cidrs'],
                peer_firewall_enable=vpc_conn_info['peer_firewall_enable'],
                status=status,
                mode=vpc_conn_info['mode'],
                priority=vpc_conn_info['priority'],
                fw_enabled=vpc_conn_info['fw_enabled'])
            context.session.add(vpc_conn_db)
            context.session.flush()

        return self._make_vpc_conn_dict(vpc_conn_db)

    def update_db_vpc_connection(self, context, vpc_conn_id, vpc_conn,
                                 original_vpc_conn=None):
        """the function to update vpc connection"""
        vpc_conn_info = VpcConnectionCustomDbMixin.deal_vpc_conn_before(
            original_vpc_conn, vpc_conn, vpc_conn_id)
        with context.session.begin(subtransactions=True):
            vpc_conn_db = self._get_vpc_conn_resource(
                context, schema.ACVpcConnectionSchema, vpc_conn_id)
            if vpc_conn_info.get('local_cidrs') and \
                    vpc_conn_info.get('peer_cidrs'):
                vpc_conn_db['local_subnets'] = []
                vpc_conn_db['peer_subnets'] = []
            if vpc_conn_info.get('local_subnets') and \
                    vpc_conn_info.get('local_subnets'):
                vpc_conn_db['local_cidrs'] = []
                vpc_conn_db['peer_cidrs'] = []
            if vpc_conn_info:
                vpc_conn_db.update(vpc_conn_info)
        return self._make_vpc_conn_dict(vpc_conn_db)

    def update_db_vpc_connection_status(self, context, vpc_conn_id, status):
        """the function to update vpc connection status"""
        LOG.info(_LI('[AC] Update vpc connection %s status %s in '
                     'Neutron DB'), vpc_conn_id, status)
        with context.session.begin(subtransactions=True):
            vpc_conn_db = self._get_vpc_conn_resource(
                context, schema.ACVpcConnectionSchema, vpc_conn_id)
            vpc_conn_db.status = status
            vpc_conn_db.update(vpc_conn_db)
        return self._make_vpc_conn_dict(vpc_conn_db)

    def delete_db_vpc_connection(self, context, vpc_conn_id):
        """the function to delete vpc connection"""
        LOG.info(_LI('[AC] Delete vpc connection in Neutron '
                     'DB by id: %s.'), vpc_conn_id)
        with context.session.begin(subtransactions=True):
            vpc_conn_db = self._get_vpc_conn_resource(
                context, schema.ACVpcConnectionSchema, vpc_conn_id)
            context.session.delete(vpc_conn_db)

    def get_db_vpc_connections(self, context, filters=None, fields=None):
        """the function to get vpc connections"""
        LOG.info(_LI("[AC] Get vpc connections from Neutron DB."))
        return self._get_collection(context, schema.ACVpcConnectionSchema,
                                    self._make_vpc_conn_dict,
                                    filters=filters, fields=fields)

    def get_db_vpc_connection(self, context, vpc_conn_id, fields=None):
        """the function to get vpc connection"""
        LOG.info(_LI('[AC] Get vpc connection from Neutron '
                     'DB by id: %s.'), vpc_conn_id)
        vpc_connection_db = self._get_vpc_conn_resource(
            context, schema.ACVpcConnectionSchema, vpc_conn_id)
        return self._make_vpc_conn_dict(vpc_connection_db, fields=fields)
