# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能： neutron 插件扩展的顶级抽象类，剥离与 neutron 多版本依赖的部分
| 修改记录：2022-03-14 11:24 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import traceback

from neutron import manager, wsgi
from neutron.api import extensions
from neutron.api.v2 import resource

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
try:
    from neutron_lib.plugins import directory
except ImportError:
    directory = None
try:
    from neutron_lib.api.faults import FAULT_MAP
except ImportError:
    from neutron.api.v2.base import FAULT_MAP
from networking_huawei.drivers.ac.common.neutron_compatible_util import get_ops_version, ac_log

LOG = ac_log.getLogger(__name__)


class AcCommonController(wsgi.Controller):
    """插件控制器类，API接口"""

    def __init__(self, plugin, *args, **kwargs):
        """初始化

        :param plugin: AbstractServicePlugin,插件实例
        """
        super(AcCommonController, self).__init__(*args, **kwargs)
        self._plugin = plugin
        self._resource_name = self._plugin.get_plugin_type()

    def create(self, request, body=None, **kwargs):
        """创建资源

        :param request: neutron.api.v2.resource.Request,请求
        :param body: dict,请求体。
        :param kwargs: dict,其它参数。neutron.api.v2.resource.resource()以键值对参数传递。
        :return: dict
        """
        result = self._plugin.create_core(request, body, **kwargs)
        LOG.debug('[AC]create %s result:%s', str(self._plugin), result)
        return result

    def index(self, request, **kwargs):
        """查询列表入口

        :param request: neutron.api.v2.resource.Request,请求
        :param kwargs: dict,其它参数。neutron.api.v2.resource.resource()以键值对参数传递。
        :return: list
        """
        LOG.debug('[AC]list %s', str(self._plugin))
        result = self._plugin.list_core(request, **kwargs)
        LOG.debug('[AC]list %s', str(self._plugin))
        return result

    def show(self, request, **kwargs):
        """查询指定资源，neutron.api.v2.resource.resource()里将url里面包含的ID以键值对参数传递到调用函数

        :param request: neutron.api.v2.resource.Request,请求
        :param kwargs: dict,请求参数,neutron.api.v2.resource.resource()以键值对参数传递。
        :return: dict
        """
        LOG.debug('[AC]show %s:%s,kwargs=%s', str(self._plugin), repr(request), kwargs)
        pk_id = kwargs.get('id')  # 资源的ID,url路径中一定会包含
        result = self._plugin.show_core(request, pk_id, **kwargs)
        LOG.debug('[AC]show %s result:%s', str(self._plugin), result)
        return result

    def delete(self, request, **kwargs):
        """删除指定资源

        :param request: neutron.api.v2.resource.Request,请求
        :param kwargs: dict,请求参数,neutron.api.v2.resource.resource()以键值对参数传递。
        :return: dict
        """
        LOG.debug('[AC]delete %s:%s,kwargs=%s', str(self._plugin), repr(request), kwargs)
        pk_id = kwargs.get('id')  # 资源的ID
        result = self._plugin.delete_core(request, pk_id, **kwargs)
        LOG.debug('[AC]delete %s result:%s', str(self._plugin), result)
        return result

    def update(self, request, body=None, **kwargs):
        """更新指定资源

        :param request: neutron.api.v2.resource.Request,请求
        :param body: dict,请求体。
        :param kwargs: dict,其它请求参数,neutron.api.v2.resource.resource()以键值对参数传递。
        :return: dict
        """
        pk_id = kwargs.get('id')  # 资源的ID
        result = self._plugin.update_core(request, pk_id, body, **kwargs)
        LOG.debug('[AC]update %s result:%s', str(self._plugin), result)
        return result


class AbstractServiceExt(ExtensionDescriptor):
    """插件扩展抽象类"""
    # openstack 版本
    ops_version = get_ops_version()

    @property
    def collection_name(self):
        """The name-s of the extension."""
        return '%ss' % self.get_name()

    @property
    def attributes(self):
        """字段属性清单"""
        return {}

    @property
    def plugin(self):
        """获取扩展对应的插件

        :return: AbstractServicePlugin
        """
        return self.get_plugin()

    @classmethod
    def get_updated(cls):
        """see ExtensionDescriptor.get_updated()"""
        return '2022-03-15T11:30:00-00:00'

    def get_name(self):
        """see ExtensionDescriptor.get_name()"""
        return self.__class__.__name__.lower()

    def get_alias(self):
        """see ExtensionDescriptor.get_alias()"""
        # 影响url：url路径最后一个'/'后面的部分
        return self.get_name()

    def get_description(self):
        """see ExtensionDescriptor.get_description()"""
        return "The description of %s." % self.get_name()

    def get_plugin(self, plugin_name=None):
        """根据不同版本获取扩展插件,neutron获取版本方式有2种,具体采用哪种方式暂不知晓机制.个人推测可能和 neutron_lib 版本有关;
        和 openstack 的版本无关的原因:
        有251 和 252 相似的2个环境都是 python2.7 环境,也都在使用“Liberty”版本.
        但251使用 manager.NeutronManager 方式,而252使用 neutron_lib.plugins.directory 方式

        :return: AbstractServicePlugin
        """
        if plugin_name is None:
            plugin_name = self.get_name()
        LOG.info("AbstractServiceExt.get_plugin():versions=%s,name=%s", self.ops_version, plugin_name)
        plugin = None
        try:
            plugin = directory.get_plugin(plugin_name)
        except AttributeError:
            LOG.error("Try to get plugin by manager.NeutronManager.get_service_plugins(),because get plugin by "
                      "neutron_lib.plugins.directory() failed:%s", traceback.format_exc())

        if plugin is None:  # 用neutron_lib.plugins.directory 方式加载失败
            plugins = manager.NeutronManager.get_service_plugins()
            plugin = plugins[plugin_name]
        LOG.info("AbstractServiceExt.get_plugin():result=%s", str(plugin))
        return plugin

    def get_resources(self):
        """see ExtensionDescriptor.get_resources()"""
        controller = resource.Resource(AcCommonController(self.plugin), faults=FAULT_MAP)
        return [extensions.ResourceExtension(self.get_alias(), controller)]

    def get_extended_resources(self, version):
        """see ExtensionDescriptor.get_extended_resources()"""
        if version == "2.0":
            return dict(list(self.attributes.items()))
        return {}
